#include "relations_processor.h"
#include "relations_manager.h"

#include "objects/linear_element.h"
#include "objects/junction.h"
#include "collection.h"
#include "objects_cache.h"
#include "configs/config.h"
#include "objects/relation_object.h"

#include <yandex/maps/wiki/configs/editor/categories.h>
#include <yandex/maps/wiki/configs/editor/slave_role.h>

#include <utility>

namespace maps {
namespace wiki {

namespace {

const GeoObject*
master(const GeoObject* object, const RelationInfo& rInfo)
{
    return (rInfo.type() == RelationType::Master) ? rInfo.relative() : object;
}

const GeoObject*
slave(const GeoObject* object, const RelationInfo& rInfo)
{
    return (rInfo.type() == RelationType::Slave) ? rInfo.relative() : object;
}

} // namespace

RelationsProcessor::RelationsProcessor(ObjectsCache& cache)
    : cache_(cache)
{}

void
RelationsProcessor::deleteRelations(RelationType relationType, GeoObject* object)
{
    auto relations = rangeToInfos(object->relations(relationType).range());
    deleteFromRelations(object, relations);
}

void
RelationsProcessor::deleteAllRelations(GeoObject* object)
{
    deleteRelations(RelationType::Master, object);
    deleteRelations(RelationType::Slave, object);
    REQUIRE(object->masterRelations().range().empty()
        && object->slaveRelations().range().empty(),
        "Not all object relations deleted oid: " << object->id());
}

void
RelationsProcessor::deleteRelations(RelationType relationType,
    GeoObject* object, const StringSet& roleIds)
{
    auto relations = rangeToInfos(object->relations(relationType).range(roleIds));
    deleteFromRelations(object, relations);
}

void
RelationsProcessor::deleteFromRelations(
    const GeoObject* deleted,
    const MixedRolesInfosContainer& relations)
{
    for (const auto& relation : relations) {
        cache_.relationsManager().deleteRelation(
            master(deleted, relation)->id(),
            slave(deleted, relation)->id(),
            relation.roleId());
    }
}

bool
RelationsProcessor::canBeReplacedWithMerged(RelationType type,
    const GeoObject* deleted, const GeoObject* /*replacement*/)
{
    if (type == RelationType::Master) {
        return true;
    }
    const RelationInfos::Range& links = deleted->relations(type).range();
    for (const auto& link : links) {
        const SlaveRole& role = master(deleted, link)->category().slaveRole(link.roleId());
        if (!role.allowReplace()) {
            return false;
        }
    }
    return true;
}

bool
RelationsProcessor::canBeReplacedWithMerged(RelationType type,
    const GeoObject* deleted, const GeoObject* /*replacement*/,
    const StringSet& roleIds)
{
    if (type == RelationType::Master) {
        return true;
    }
    const RelationInfos::Range& links = deleted->relations(type).range(roleIds);
    for (const auto& link : links) {
        const SlaveRole& role = master(deleted, link)->category().slaveRole(link.roleId());
        if (!role.allowReplace()) {
            return false;
        }
    }
    return true;
}

void
RelationsProcessor::replaceWithMerged(RelationType type,
    GeoObject* deleted, GeoObject* replacement)
{
    auto relations = rangeToInfos(deleted->relations(type).range());
    replaceRelationsWithMerged(deleted, replacement, relations);
}

void
RelationsProcessor::replaceWithMerged(RelationType type,
    GeoObject* deleted, GeoObject* replacement,
    const StringSet& roleIds)
{
    auto relations = rangeToInfos(deleted->relations(type).range(roleIds));
    replaceRelationsWithMerged(deleted, replacement, relations);
}

void
RelationsProcessor::replaceRelationsWithMerged(
    const GeoObject* deleted, const GeoObject* replacement,
    const MixedRolesInfosContainer& relations)
{
    for (const auto& relation : relations) {
        const std::string& roleId = relation.roleId();
        TOid delMasterId = master(deleted, relation)->id();
        TOid delSlaveId = slave(deleted, relation)->id();
        cache_.relationsManager().deleteRelation(delMasterId, delSlaveId, roleId);
        TOid addMasterId = master(replacement, relation)->id();
        TOid addSlaveId = slave(replacement, relation)->id();
        cache_.relationsManager().createRelation(addMasterId, addSlaveId, roleId);
    }
}

namespace {
StringSet
adjacentRoles(const SlaveRoles& roles, const std::string& roleId)
{
    StringSet ret;
    auto it = roles.findByKey(roleId);
    if (it == roles.end()) {
        return ret;
    }
    if (it != roles.begin()) {
        auto prev = --it;
        ret.insert(prev->roleId());
    }
    auto next = ++it;
    if (next != roles.end()) {
        ret.insert(next->roleId());
    }
    ret.insert(roleId);
    return ret;
}

std::vector<GeoObject*>
findAdjacent(const RelationInfos::Range& slaveInfos,
    const GeoObject* source, const std::vector<GeoObject*>& parts,
    const StringSet& adjacentRoles)
{
    std::vector<GeoObject*> result;
    for (const auto& part : parts) {
        auto it = std::find_if(
            slaveInfos.begin(), slaveInfos.end(),
            [&part, &source, &adjacentRoles] (const RelationInfo& link) -> bool
            {
                return adjacentRoles.count(link.roleId()) &&
                    link.id() != source->id() &&
                    !link.geom().isNull() &&
                    part->geom()->touches(link.geom().geosGeometryPtr());
            });
        if (it != slaveInfos.end()) {
            result.push_back(part);
        }
    }
    return result;
}
}//namespace

void
RelationsProcessor::replaceWithParts(RelationType type,
    GeoObject* source, const std::vector<GeoObject*>& parts)
{
    if (type == RelationType::Slave) {
        THROW_WIKI_INTERNAL_ERROR("Replacement with parts not implemented for slave relations");
    }
    auto links = rangeToInfos(source->relations(type).range());
    for (const auto& link : links) {
        GeoObject* master = link.relative();
        const SlaveRole& role = master->category().slaveRole(link.roleId());
        DEBUG() << "Role keep-all: " << role.keepAll();
        if (role.keepAll()) {
            replace(link.roleId(), master, source, parts);
            continue;
        }
        std::vector<GeoObject*> adjacentParts = findAdjacent(
                master->slaveRelations().range(), source, parts,
                adjacentRoles(master->category().slavesRoles(), link.roleId()));
        if (!adjacentParts.empty()) {
            replace(link.roleId(), master, source, adjacentParts);
        }
    }
}

void
RelationsProcessor::replace(const std::string& roleId,
    GeoObject* masterWhere, GeoObject* slaveWhat,
    const std::vector<GeoObject*>& partsWith)
{
    cache_.relationsManager().deleteRelation(masterWhere->id(), slaveWhat->id(), roleId);
    for (auto part : partsWith) {
        cache_.relationsManager().createRelation(masterWhere->id(), part->id(), roleId);
    }
}

} // namespace wiki
} // namespace maps
