#include "rendererlayerstraverse.h"
#include "exception.h"
#include "configs/config.h"

namespace maps {
namespace wiki {

namespace {

const std::string MPRO_LAYER_TYPE = "mproLayerType";
const std::string MPRO_LAYER_CATEGORIES = "mproLayerCategories";
const std::string MPRO_LAYER_GROUP = "mproLayerGroup";
const std::string MPRO_LAYER_ROLES = "mproLayerRoles";
const std::string MPRO_LAYER_LABEL = "mproLayerLabel";
const std::string MPRO_LAYER_PROJECTS = "mproLayerProjects";
const std::string MPRO_LAYER_MAPTYPES = "mproLayerMapTypes";
const std::string MPRO_LAYER_ACL_GROUPS = "mproLayerAclGroups";

const std::string LAYER_TYPE_BASE = "base";
const std::string LAYER_TYPE_SELECTION = "selection";
const std::string LAYER_TYPE_INDOOR = "indoor";
const std::string LAYER_TYPE_REPRESENTATION = "representation";
const std::string LAYER_TYPE_FILTER = "filter";
const std::string LAYER_TYPE_TEXT = "text";
const std::string LAYER_GROUPS_KEYSET = "layer-groups";
const char MPRO_LAYER_ARRAY_DELIMITER = ',';

std::string extractMetadata(
    const std::string& key,
    const maps::tilerenderer4::LayerInfo& layer)
{
    auto value = layer.metadata(key);
    if (!value) {
        THROW_WIKI_INTERNAL_ERROR("Key '" << key << "' not found " <<
            "(layerID = " << layer.id() << ")");
    }
    return *value;
}

std::string extractMetadataOpt(
    const std::string& key,
    const maps::tilerenderer4::LayerInfo& layer)
{
    auto value = layer.metadata(key);
    return value
        ? *value
        : s_emptyString;
}
} //namespace

RendererLayersTraverse::Layer::Layer(
        const std::string& type,
        const tilerenderer4::LayerInfo& layer)
    : id_(layer.id())
    , type_(type)
    , groupId_(extractMetadata(MPRO_LAYER_GROUP, layer))
    , name_(layer.name())
    , label_(layer.metadata(MPRO_LAYER_LABEL))
    , projectParams_(
            {
                    split<StringSet>(extractMetadataOpt(MPRO_LAYER_MAPTYPES, layer), MPRO_LAYER_ARRAY_DELIMITER),
                    split<StringSet>(extractMetadataOpt(MPRO_LAYER_PROJECTS, layer), MPRO_LAYER_ARRAY_DELIMITER),
                    split<StringSet>(extractMetadataOpt(MPRO_LAYER_ACL_GROUPS, layer), MPRO_LAYER_ARRAY_DELIMITER),
            }
        )
{
    auto value = layer.metadata(MPRO_LAYER_CATEGORIES);
    if (value) {
        categoryIds_ = split<StringVec>(*value, MPRO_LAYER_ARRAY_DELIMITER);
    }
}

bool
RendererLayersTraverse::Layer::isFilterLayer() const
{
    return LAYER_TYPE_FILTER == type_;
}

RendererLayersTraverse::Layer&
RendererLayersTraverse::currentLayer()
{
    REQUIRE(!layers_.empty(), "No base layers");
    return layers_.back();
}

bool
RendererLayersTraverse::onEnterGroupLayer(
    const maps::tilerenderer4::LayerInfo& layer)
{
    const auto layerType = layer.metadata(MPRO_LAYER_TYPE);
    if (!layerType) {
        return true;
    }
    if (*layerType == LAYER_TYPE_BASE ||
        *layerType == LAYER_TYPE_FILTER ||
        *layerType == LAYER_TYPE_INDOOR)
    {
        layers_.emplace_back(*layerType, layer);
        const auto& layer = layers_.back();
        layersByGroup_[layer.groupId()].push_back(&layer);
    } else if (*layerType == LAYER_TYPE_REPRESENTATION) {
        currentLayer().appendRepresentation(Representation{
            layer.id(),
            layer.name(),
            extractMetadata(MPRO_LAYER_LABEL, layer),
                {
                    split<StringSet>(extractMetadataOpt(MPRO_LAYER_MAPTYPES, layer), MPRO_LAYER_ARRAY_DELIMITER),
                    split<StringSet>(extractMetadataOpt(MPRO_LAYER_PROJECTS, layer), MPRO_LAYER_ARRAY_DELIMITER),
                    split<StringSet>(extractMetadataOpt(MPRO_LAYER_ACL_GROUPS, layer), MPRO_LAYER_ARRAY_DELIMITER),
                }
            });
    } else if (*layerType == LAYER_TYPE_TEXT) {
        Layer& currentLayer = this->currentLayer();
        currentLayer.setTextLayer(
            {
                layer.id(),
                {
                    split<StringSet>(extractMetadataOpt(MPRO_LAYER_MAPTYPES, layer), MPRO_LAYER_ARRAY_DELIMITER),
                    split<StringSet>(extractMetadataOpt(MPRO_LAYER_PROJECTS, layer), MPRO_LAYER_ARRAY_DELIMITER),
                    split<StringSet>(extractMetadataOpt(MPRO_LAYER_ACL_GROUPS, layer), MPRO_LAYER_ARRAY_DELIMITER),
                }
            });
        const auto& currentCategories = currentLayer.categoryIds();
        labeledCategories_.insert(
            currentCategories.begin(), currentCategories.end());
    } else {
        THROW_WIKI_INTERNAL_ERROR(
            "Invalid mproLayerType " << "(layerID = " << layer.id() << ")");
    }
    return true;
}

bool RendererLayersTraverse::onVisitLayer(
    const maps::tilerenderer4::LayerInfo& layer)
{
    const auto layerType = layer.metadata(MPRO_LAYER_TYPE);
    if (layerType && *layerType == LAYER_TYPE_SELECTION) {
        layers_.emplace_back(*layerType, layer);
        const auto& layer = layers_.back();
        layersByGroup_[layer.groupId()].push_back(&layer);
    }
    return true;
}

void RendererLayersTraverse::onLeaveGroupLayer(
    const maps::tilerenderer4::LayerInfo& /*layer*/)
{}

bool RendererLayersTraverse::isDone()
{
    return false;
}

namespace
{
bool
intersects(const StringSet& set1, const StringSet& set2)
{
    auto it1 = set1.begin();
    auto it2 = set2.begin();
    while (it1 != set1.end() && it2 != set2.end()) {
        if (*it1<*it2) {
            ++it1;
        } else if (*it2<*it1) {
            ++it2;
        } else {
            return true;
        }
    }
    return false;
}
}//namespace
bool
RendererLayersTraverse::ProjectParams::match(const ProjectParams& requested) const
{
    if (!mapTypes.empty() && !intersects(mapTypes, requested.mapTypes)) {
        return false;
    }
    if (!projects.empty() && !intersects(projects, requested.projects)) {
        return false;
    }
    if (!aclGroups.empty() && !intersects(aclGroups, requested.aclGroups)) {
        return false;
    }
    return true;
}

std::string
layersGroupLabel(const std::string& grpId)
{
    return "{{" + LAYER_GROUPS_KEYSET + ":" + grpId + "}}";
}

} // namespace wiki
} // namespace maps
