#include "bbox_provider.h"
#include "bbox_helpers.h"
#include <maps/wikimap/mapspro/services/editor/src/revision_meta/common.h>

namespace maps {
namespace wiki {

BBoxProvider::BBoxProvider(ObjectsCache& cache, SlavesLoadingPolicy policy)
    : cache_(cache)
    , policy_(policy)
{
}

Geom BBoxProvider::geometryForObject(const GeoObject& object)
{
    if (!object.category().complex()) {
        DEBUG() << "BBox is requested for object of category " << object.category().id();
        return object.geom();
    }

    if (object.isDeleted() && object.prevRevision().valid()) {
        ObjectsCache cache(
            cache_.branchContext(),
            object.prevRevision().commitId(),
            REVISION_META_CACHE_POLICY);
        return geometryForObjectImpl(*cache.getExisting(object.id()));
    }
    return geometryForObjectImpl(object);
}

Geom BBoxProvider::geometryForObjectImpl(const GeoObject& object)
{
    if (!object.geom().isNull()) {
        return object.geom();
    }

    auto it = bboxCache_.find(object.id());
    if (it != bboxCache_.end()) {
        return it->second;
    }

    const auto& category = object.category();
    const auto slaveRoles = category.slaveRoleIds(roles::filters::IsGeom);
    const auto slavesThreshold = category.cacheGeomPartsThreshold();

    if (!slavesThreshold || policy_ == SlavesLoadingPolicy::All) {
        auto geom = dataFromSlaves(object.slaveRelations().range(slaveRoles));
        bboxCache_.emplace(object.id(), geom);
        return geom;
    }

    auto range = object.slaveRelations().range(slaveRoles, {*slavesThreshold, RelativesLimit::All});
    if (range) {
        auto geom = dataFromSlaves(*range);
        bboxCache_.emplace(object.id(), geom);
        return geom;
    }

    auto commitId = object.cache().headCommitId();

    auto cached = loadFromDb(object.id(), commitId);
    if (cached) {
        bboxCache_.emplace(object.id(), *cached);
        return *cached;
    }

    /// Not cached
    WARN() << "Object: " << object.id() << ", head commit: " << commitId << " bbox is missing";
    return Geom();
}

Geom BBoxProvider::dataFromSlaves(const RelationInfos::Range& slavesRange)
{
    GeomComposer gc;
    for (const auto& rInfo : slavesRange) {
        auto slaveGeom = geometryForObjectImpl(*rInfo.relative());
        gc.process(slaveGeom);
    }
    return gc.result();
}

boost::optional<Geom> BBoxProvider::loadFromDb(TOid objectId, TCommitId maxCommitId)
{
    auto branchId = cache_.branchContext().branch.id();

    std::ostringstream query;
    query <<
        "SELECT the_geom"
        " FROM revision_meta.geometry"
        " WHERE object_id=" << objectId <<
        " AND commit_id<=" << maxCommitId <<
        " AND branch_id=" << branchId <<
        " ORDER BY commit_id DESC LIMIT 1";

    auto r = cache_.workCore().exec(query.str());
    if (r.empty()) {
        return boost::none;
    }
    ASSERT(r.size() == 1);

    try {
        const auto& wkbField = r[0][0];
        return wkbField.is_null()
            ? Geom()
            : Geom(wkbField);
    } catch (const std::exception& e) {
        ERROR() << "Geometry parsing error: " << e.what();
        return boost::none;
    }
}

} // namespace wiki
} // namespace maps
