#include "bbox_helpers.h"
#include "complex_object_bbox_computer.h"
#include "filter_helpers.h"
#include <maps/wikimap/mapspro/services/editor/src/relations_manager.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/category_traits.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/complex_object.h>
#include <maps/wikimap/mapspro/services/editor/src/revision_meta/common.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/sync/db_helpers.h>

#include <yandex/maps/wiki/common/batch.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/revision/filters.h>
#include <maps/libs/common/include/profiletimer.h>

namespace rf = maps::wiki::revision::filters;

namespace maps {
namespace wiki {

namespace {

const size_t OBJECTS_BATCH_COUNT = 500;
const size_t REVISIONS_BATCH_COUNT = 500;

const std::string& extractSlaveCategory(const StringMap& attributes)
{
    auto it = attributes.find(ATTR_REL_SLAVE);
    if (it != attributes.end()) {
        return it->second;
    }
    return s_emptyString;
}

} // namespace

ComplexObjectBBoxComputer::ComplexObjectBBoxComputer(
        TOIds topLevelComplexObjectIds,
        std::string topLevelCategoryId,
        BBoxContext context)
    : context_(std::move(context))
    , writer_(context_)
    , topLevelComplexObjectIds_(std::move(topLevelComplexObjectIds))
    , topLevelCategoryId_(std::move(topLevelCategoryId))
{
}

void ComplexObjectBBoxComputer::addComplexObjectInfo(TOid objectId, const Category& category)
{
    ASSERT(category.complex());
    ASSERT(category.cacheGeomPartsThreshold()); //TODO: slave complex objects without threshold?

    allComplexObjectIds_.insert(objectId);
    allComplexObjectCategoryIds_.insert(category.id());

    complexObjectInfos_.emplace(
        objectId,
        ComplexObjectInfo{
            category.id(),
            *category.cacheGeomPartsThreshold()});
}

TOIds ComplexObjectBBoxComputer::loadComplexObjectsInitialState(const revision::Snapshot& snapshot, const TOIds& complexObjectIds)
{
    if (complexObjectIds.empty()) {
        return {};
    }

    TOIds newComplexObjectIds;

    common::applyBatchOp<TOIds>(complexObjectIds,
        OBJECTS_BATCH_COUNT,
        [&, this](const TOIds& batch) {
            auto relationsFilter = createRelationsFilter(allComplexObjectCategoryIds_)
                && rf::ObjRevAttr::masterObjectId().in(batch)
                && rf::ObjRevAttr::isNotDeleted();

            for (const auto& rev : snapshot.relationsByFilter(relationsFilter)) {
                const auto& attributes = *rev.data().attributes;
                const auto& slaveCategoryId = extractSlaveCategory(attributes);
                if (!cfg()->editor()->categories().defined(slaveCategoryId)) {
                    continue;
                }

                const auto& relationData = *rev.data().relationData;
                auto slaveId = relationData.slaveObjectId();
                auto masterId = relationData.masterObjectId();

                currentSlaves_[masterId].insert(slaveId);
                currentMasters_[slaveId].insert(masterId);

                const auto& slaveCategory = cfg()->editor()->categories()[slaveCategoryId];
                if (slaveCategory.complex()) {
                    addComplexObjectInfo(slaveId, slaveCategory);
                    newComplexObjectIds.insert(slaveId);
                } else {
                    geomPartObjectIds_.insert(slaveId);
                }
            }
        });

    return newComplexObjectIds;
}

void ComplexObjectBBoxComputer::loadGeomPartsInitialState(const revision::Snapshot& snapshot)
{
    if (geomPartObjectIds_.empty()) {
        return;
    }

    const auto filter = rf::ObjRevAttr::isNotRelation()
        && rf::ObjRevAttr::isNotDeleted()
        && rf::Geom::defined();

    RevisionIds allGeomPartsRevisionIds;

    common::applyBatchOp<TOIds>(geomPartObjectIds_,
        OBJECTS_BATCH_COUNT,
        [&](const TOIds& objectIds) {
            auto revIds = snapshot.revisionIdsByFilter(objectIds, filter);
            allGeomPartsRevisionIds.insert(revIds.begin(), revIds.end());
        });

    common::applyBatchOp<RevisionIds>(allGeomPartsRevisionIds,
        REVISIONS_BATCH_COUNT,
        [&, this](const RevisionIds& revisionIds) {
            for (auto&& rev : snapshot.reader().loadRevisions(revisionIds, filter)) {
                ASSERT(rev.data().geometry);
                currentGeometries_.emplace(rev.id().objectId(), Geom(*rev.data().geometry));
            }
        });
}

void ComplexObjectBBoxComputer::loadInitialState(const revision::Snapshot& snapshot)
{
    auto newComplexObjectIds = loadComplexObjectsInitialState(snapshot, topLevelComplexObjectIds_);

    if (!newComplexObjectIds.empty()) {
        loadComplexObjectsInitialState(snapshot, newComplexObjectIds);
    }
}

void ComplexObjectBBoxComputer::computeComplexObjectsInitialGeometry()
{
    for (auto id : allComplexObjectIds_) {
        if (topLevelComplexObjectIds_.count(id)) {
            continue;
        }
        auto geom = computeGeometryForObject(id);
        if (needWriteToDb(id)) {
            writer_.setInitialGeom(id, geom);
        }
        currentGeometries_[id] = std::move(geom);
    }

    for (auto id : topLevelComplexObjectIds_) {
        auto geom = computeGeometryForObject(id);
        if (needWriteToDb(id)) {
            writer_.setInitialGeom(id, geom);
        }
        currentGeometries_[id] = std::move(geom);
    }
}

TOIds ComplexObjectBBoxComputer::loadHistorySlaveComplexObjectIds(const revision::HistoricalSnapshot& historicalSnapshot)
{
    auto filter = createRelationsFilter({topLevelCategoryId_})
        && rf::ObjRevAttr::masterObjectId().in(topLevelComplexObjectIds_); //no need for batch here

    TOIds slaveComplexObjectIdsToPreload;

    for (auto&& rev : historicalSnapshot.relationsByFilter(filter)) {
        const auto& attributes = *rev.data().attributes;
        const auto& slaveCategoryId = extractSlaveCategory(attributes);
        if (!cfg()->editor()->categories().defined(slaveCategoryId)) {
            continue;
        }
        const auto& slaveCategory = cfg()->editor()->categories()[slaveCategoryId];
        if (slaveCategory.complex()) {
            auto slaveId = rev.data().relationData->slaveObjectId();
            addComplexObjectInfo(slaveId, slaveCategory);
            slaveComplexObjectIdsToPreload.insert(slaveId);
        }

        commitToRevisions_[rev.id().commitId()].push_back(std::move(rev));
    }

    return slaveComplexObjectIdsToPreload;
}

void ComplexObjectBBoxComputer::loadHistoryGeomPartsIds(const revision::HistoricalSnapshot& historicalSnapshot)
{
    common::applyBatchOp<TOIds>(allComplexObjectIds_,
        OBJECTS_BATCH_COUNT,
        [&, this](const TOIds& objectIds) {
            auto filter = createRelationsFilter(allComplexObjectCategoryIds_)
                && rf::ObjRevAttr::masterObjectId().in(objectIds);

            for (auto&& rev : historicalSnapshot.relationsByFilter(filter)) {
                auto slaveId = rev.data().relationData->slaveObjectId();
                if (!allComplexObjectIds_.count(slaveId)) {
                    geomPartObjectIds_.insert(slaveId);
                    commitToRevisions_[rev.id().commitId()].push_back(std::move(rev));
                }
            }
        });
}

void ComplexObjectBBoxComputer::loadHistoryGeometryData(const revision::HistoricalSnapshot& historicalSnapshot)
{
    const auto filter = rf::ObjRevAttr::isNotRelation()
        && rf::Geom::defined();

    RevisionIds allGeomPartsRevisionIds;

    common::applyBatchOp<TOIds>(geomPartObjectIds_,
        OBJECTS_BATCH_COUNT,
        [&](const TOIds& objectIds) {
            auto revIds = historicalSnapshot.revisionIdsByFilter(objectIds, filter);
            allGeomPartsRevisionIds.insert(revIds.begin(), revIds.end());
        });

    common::applyBatchOp<RevisionIds>(allGeomPartsRevisionIds,
        REVISIONS_BATCH_COUNT,
        [&, this](const RevisionIds& revisionIds) {
            for (auto&& rev : historicalSnapshot.reader().loadRevisions(revisionIds, filter)) {
                commitToRevisions_[rev.id().commitId()].push_back(std::move(rev));
            }
        });
}

Geom ComplexObjectBBoxComputer::computeGeometryForObject(TOid complexObjectId) const
{
    auto it = currentSlaves_.find(complexObjectId);
    if (it == currentSlaves_.end()) {
        return Geom();
    }

    GeomComposer gc;
    for (const auto slaveId : it->second) {
        auto it = currentGeometries_.find(slaveId);
        if (it != currentGeometries_.end()) {
            gc.process(it->second);
        }
    }
    return gc.result();
}

bool ComplexObjectBBoxComputer::needWriteToDb(TOid complexObjectId) const
{
    auto infoIt = complexObjectInfos_.find(complexObjectId);
    ASSERT(infoIt != complexObjectInfos_.end());
    const auto& info = infoIt->second;

    auto it = currentSlaves_.find(complexObjectId);
    if (it == currentSlaves_.end()) {
        return false;
    }

    const auto& slaveIds = it->second;
    if (slaveIds.size() > info.cacheGeomPartsThreshold) {
        size_t geometryCount = 0;
        for (auto slaveId : slaveIds) {
            if (currentGeometries_.count(slaveId)) {
                geometryCount++;
            }
        }

        //double check due to import commits
        if (geometryCount > info.cacheGeomPartsThreshold) {
            return true;
        }
    }

    return false;
}

void ComplexObjectBBoxComputer::updateGeometryForComplexObject(TOid complexObjectId)
{
    auto geom = computeGeometryForObject(complexObjectId);
    if (geom.isNull()) {
        return;
    }

    if (needWriteToDb(complexObjectId)) {
        auto infoIt = complexObjectInfos_.find(complexObjectId);
        ASSERT(infoIt != complexObjectInfos_.end());
        const auto& info = infoIt->second;

        writer_.writeToDb(currentCommitId_, complexObjectId, info.categoryId, geom);
    }

    currentGeometries_[complexObjectId] = std::move(geom);
}

void ComplexObjectBBoxComputer::updateGeometry(const TOIds& objectIdsToUpdateGeom)
{
    TOIds masterIds;

    for (auto id : objectIdsToUpdateGeom) {
        if (!geomPartObjectIds_.count(id)) {
            masterIds.insert(id);
        } else {
            auto it = currentMasters_.find(id);
            if (it != currentMasters_.end()) {
                masterIds.insert(it->second.begin(), it->second.end());
            }
        }
    }

    TOIds topLevelMasterIds;

    for (auto id : masterIds) {
        if (topLevelComplexObjectIds_.count(id)) {
            topLevelMasterIds.insert(id);
        } else {
            updateGeometryForComplexObject(id);

            auto it = currentMasters_.find(id);
            if (it != currentMasters_.end()) {
                topLevelMasterIds.insert(it->second.begin(), it->second.end());
            }
        }
    }

    for (auto id : topLevelMasterIds) {
        updateGeometryForComplexObject(id);
    }
}

TOid ComplexObjectBBoxComputer::applyRevision(const revision::ObjectRevision& revision)
{
    if (revision.data().revisionType() == revision::RevisionType::RegularObject) {
        ASSERT(revision.data().geometry);

        auto objectId = revision.id().objectId();

        if (revision.data().deleted) {
            currentGeometries_.erase(objectId);
            return objectId;
        }

        Geom newGeom(*revision.data().geometry);

        auto it = currentGeometries_.find(objectId);
        if (it == currentGeometries_.end()) {
            currentGeometries_.emplace(objectId, std::move(newGeom));
            return objectId;
        }

        Geom oldEnvelope(it->second->getEnvelope());
        Geom newEnvelope(newGeom->getEnvelope());

        currentGeometries_[objectId] = std::move(newGeom);

        if (!newEnvelope.equal(oldEnvelope, CALCULATION_TOLERANCE)) { //update if envelope has changed
            return objectId;
        }
    } else {
        // RevisionType::Relation
        ASSERT(revision.data().relationData);

        auto masterId = revision.data().relationData->masterObjectId();
        auto slaveId = revision.data().relationData->slaveObjectId();
        auto& slaves = currentSlaves_[masterId];
        auto& masters = currentMasters_[slaveId];

        if (revision.data().deleted) {
            slaves.erase(slaveId);
            masters.erase(masterId);
        } else {
            slaves.insert(slaveId);
            masters.insert(masterId);
        }

        return masterId;
    }

    return 0;
}

void ComplexObjectBBoxComputer::preload()
{
    INFO() << "Start preloading complex object ids size " << topLevelComplexObjectIds_.size()
        << " of category " << topLevelCategoryId_
        << " in commits [" << context_.startCommitId() << "; " << context_.endCommitId() << "]";

    ProfileTimer pt;

    ASSERT(cfg()->editor()->categories().defined(topLevelCategoryId_));
    const auto& topLevelCategory = cfg()->editor()->categories()[topLevelCategoryId_];
    ASSERT(topLevelCategory.cacheGeomPartsThreshold());

    for (auto id : topLevelComplexObjectIds_) {
        addComplexObjectInfo(id, topLevelCategory);
    }

    auto branchContext = context_.getReadContext();

    revision::RevisionsGateway gateway(branchContext.txnCore(), branchContext.branch);
    auto historicalSnapshot = gateway.historicalSnapshot(context_.startCommitId(), context_.endCommitId());

    std::unique_ptr<revision::Snapshot> initialSnapshot;
    if (context_.startCommitId() > 1) {
        initialSnapshot.reset(new revision::Snapshot(gateway.snapshot(context_.startCommitId() - 1)));
        loadInitialState(*initialSnapshot);
    }

    if (!context_.checkOk(branchContext.txnCore())) {
        return;
    }

    if (isIndirectGeomMasterCategory(topLevelCategory)) {
        INFO() << "Start preloading indirect data for category " << topLevelCategoryId_
            << " in commits [" << context_.startCommitId() << "; " << context_.endCommitId() << "]";
        auto slaveComplexObjectIdsToPreload = loadHistorySlaveComplexObjectIds(historicalSnapshot);

        if (initialSnapshot) {
            loadComplexObjectsInitialState(*initialSnapshot, slaveComplexObjectIdsToPreload);
        }
    }

    INFO() << "Start preloading history geom part ids for category " << topLevelCategoryId_
        << " in commits [" << context_.startCommitId() << "; " << context_.endCommitId() << "]";
    loadHistoryGeomPartsIds(historicalSnapshot);

    if (initialSnapshot) {
        INFO() << "Start preloading initial snapshot data for category " << topLevelCategoryId_
            << " in commits [" << context_.startCommitId() << "; " << context_.endCommitId() << "]";
        loadGeomPartsInitialState(*initialSnapshot);

        INFO() << "Compute complex objects initial geometry for category " << topLevelCategoryId_
            << " in commits [" << context_.startCommitId() << "; " << context_.endCommitId() << "]";
        computeComplexObjectsInitialGeometry();
    }

    if (!context_.checkOk(branchContext.txnCore())) {
        return;
    }

    INFO() << "Start preloading history geometry data for category " << topLevelCategoryId_
        << " in commits [" << context_.startCommitId() << "; " << context_.endCommitId() << "]";
    loadHistoryGeometryData(historicalSnapshot);

    INFO() << "Preload finished in " << pt.getElapsedTime() << "s. "
        << "Category " << topLevelCategoryId_ << ". "
        << "Top level complex objects " << topLevelComplexObjectIds_.size() << ". "
        << "Complex slaves " << (allComplexObjectIds_.size() - topLevelComplexObjectIds_.size()) << ". "
        << "Geometry objects " << geomPartObjectIds_.size() << ". "
        << "Commits " << commitToRevisions_.size() << ".";
}

void ComplexObjectBBoxComputer::run()
{
    ProfileTimer pt;

    auto txn = sync::masterCoreTransaction(AccessMode::ReadWrite);

    if (!context_.checkOk(*txn)) {
        return;
    }

    writer_.clearOldBBoxes(*txn, allComplexObjectIds_);

    for (const auto& pair : commitToRevisions_) {
        currentCommitId_ = pair.first;
        const auto& revisions = pair.second;

        TOIds objectIdsToUpdateGeom;

        for (const auto& rev : revisions) {
            auto id = applyRevision(rev);
            if (id) {
                objectIdsToUpdateGeom.insert(id);
            }
        }

        updateGeometry(objectIdsToUpdateGeom);
    }

    writer_.flush(*txn);

    txn->commit();

    INFO() << "Computation finished in " << pt.getElapsedTime() << "s. "
        << "Category " << topLevelCategoryId_ << ". "
        << "Top level complex objects " << topLevelComplexObjectIds_.size() << ". "
        << "Complex slaves " << (allComplexObjectIds_.size() - topLevelComplexObjectIds_.size()) << ". "
        << "Geometry objects " << geomPartObjectIds_.size() << ". "
        << "Commits " << commitToRevisions_.size() << ".";
}

std::map<TOid, Geom> ComplexObjectBBoxComputer::finalGeometries()
{
    std::map<TOid, Geom> geoms;
    for (auto id : topLevelComplexObjectIds_) {
        auto it = currentGeometries_.find(id);
        if (it != currentGeometries_.end()) {
            geoms.emplace(id, std::move(it->second));
        }
    }
    return geoms;
}

} // namespace wiki
} // namespace maps
