#include "create_intersections_parser.h"

#include "json_parser.h"

namespace maps {
namespace wiki {

template <>
void
CreateIntersectionsParser::parseImpl<common::FormatType::XML>(const std::string& /*requestBody*/)
{
    UNSUPPORTED_FORMAT(common::FormatType::XML, "CreateIntersections");
}

namespace {

TopoObjectData
readObject(const json::Value& objValue)
{
    auto revId = JsonParser::readObjectRevision(objValue);
    auto uuid = JsonParser::readObjectUUID(objValue);
    auto catId = JsonParser::readObjectCategory(objValue);
    auto geom = JsonParser::readObjectGeometry(objValue);

    UniqueId id{revId, std::move(uuid)};
    return TopoObjectData{id, std::move(revId), std::move(catId), std::move(geom)};

}

} // namespace

template <>
void
CreateIntersectionsParser::parseImpl<common::FormatType::JSON>(const std::string& requestBody)
{
    objects_.clear();
    auto jsonValueObjects = json::Value::fromString(requestBody);
    WIKI_REQUIRE(
        jsonValueObjects.hasField("geoObjects"),
        ERR_BAD_DATA,
        "No geoObjects field in request");
    auto objectsArray = jsonValueObjects["geoObjects"];
    WIKI_REQUIRE(
        objectsArray.isArray(),
        ERR_BAD_DATA,
        "geoObjects field should be json array");
    for (const auto& objValue : objectsArray) {
        objects_.push_back(readObject(objValue));
    }
    DEBUG() << BOOST_CURRENT_FUNCTION << " Read objects: " << objects_.size();

    WIKI_REQUIRE(
        jsonValueObjects.hasField("editContext"),
        ERR_BAD_DATA,
        "No editContext field in request");

    editContext_ = JsonParser::readObjectEditContext(jsonValueObjects);
    ASSERT(editContext_);
}

void
CreateIntersectionsParser::parse(common::FormatType formatType, const std::string& requestBody)
{
    if (formatType == common::FormatType::XML) {
        parseImpl<common::FormatType::XML>(requestBody);
    } else if (formatType == common::FormatType::JSON) {
        parseImpl<common::FormatType::JSON>(requestBody);
    } else {
        REQUIRE(false, "Unknown format type " << (int)formatType);
    }
}

} // namespace wiki
} // namespace maps
