#include "json_formatter.h"

#include <maps/wikimap/mapspro/services/editor/src/commit.h>
#include "json_writer.h"
#include <maps/wikimap/mapspro/services/editor/src/common.h>
#include "json_common.h"
#include "title.h"

#include <maps/libs/json/include/builder.h>

namespace maps {
namespace wiki {

namespace {

void
putAttrValues(
    const AttrDefPtr& attrDef,
    const std::string& valuesString,
    json::ObjectBuilder& attrDiffBuilder,
    const std::string& propName)
{
    if (!attrDef->multiValue()) {
        attrDiffBuilder[propName] = json::Verbatim(escapeAttrValue(attrDef->valueType(), valuesString));
        return;
    }
    StringSet values = split<StringSet>(valuesString, AttributeDef::MULTIVALUE_DELIMITER);
    attrDiffBuilder[propName] << [&](json::ArrayBuilder valuesBuilder) {
        for (const auto& value : values) {
            if (!value.empty()) {
                valuesBuilder << json::Verbatim(escapeAttrValue(attrDef->valueType(), value));
            }
        }
    };
}

void
putAttributeDiff(
    json::ArrayBuilder& attrsDiffBuilder,
    const revision::AttributesDiff::value_type& attrDiff,
    bool withBefore,
    const boost::optional<revision::StringDiff>& categoryDiff)
{
    if (!cfg()->editor()->isAttributeDefined(attrDiff.first)) {
        WARN() << "Attribute " << attrDiff.first << " is not defined.";
        return;
    }
    const auto& attrDef = cfg()->editor()->attribute(attrDiff.first);
    ASSERT(attrDef);
    if (attrDef->system() || attrDef->table()) {
        return;
    }
    if (attrDiff.second.before.empty() && withBefore
        && !attrDef->booleanValue()
        && attrDiff.second.after == attrDef->defaultValue()
        && !categoryDiff) {
        return;
    }
    attrsDiffBuilder << [&](json::ObjectBuilder attrDiffBuilder) {
        attrDiffBuilder[STR_ID] = attrDiff.first;
        if (withBefore) {
            putAttrValues(attrDef, attrDiff.second.before, attrDiffBuilder, STR_BEFORE);
        }
        putAttrValues(attrDef, attrDiff.second.after, attrDiffBuilder, STR_AFTER);
    };
}

void
putTableAttributesDiff(
    json::ArrayBuilder& attrsDiffBuilder,
    const DiffDetails& details)
{
    ASSERT(details.toObjectVersion);
    const auto& obj = details.toObjectVersion;
    for (const auto& attrName : obj->tableAttributes().attrNames()) {
        const auto& toAttrValues = obj->tableAttributes().find(attrName);
        if (toAttrValues.empty() && !details.fromObjectVersion) {
            continue;
        }
        if (details.fromObjectVersion) {
            const auto& fromAttrValues = details.fromObjectVersion->tableAttributes().find(attrName);
            if (fromAttrValues.equal(toAttrValues)) {
                continue;
            }
        }
        attrsDiffBuilder << [&](json::ObjectBuilder attrDiffBuilder) {
            attrDiffBuilder[STR_ID] = attrName;
            if (details.fromObjectVersion) {
                attrDiffBuilder[STR_BEFORE] << [&](json::ArrayBuilder valuesBuilder) {
                    putTableAttributeContents(
                        valuesBuilder,
                        details.fromObjectVersion->tableAttributes().find(attrName));
                };
            }
            attrDiffBuilder[STR_AFTER] << [&](json::ArrayBuilder valuesBuilder) {
                putTableAttributeContents(
                    valuesBuilder,
                    toAttrValues);
            };
        };
    }
}

void
putRelativesViewObjects(
        const TOIds& oids,
        const std::map<TOid, views::ViewObject>& relatives,
        json::ArrayBuilder& array,
        std::optional<TOid> excludeObject)
{
    for (auto oid : oids) {
        if (excludeObject == oid) {
            continue;
        }
        array << [&](json::ObjectBuilder objBuilder) {
            objBuilder[STR_ID] = common::idToJson(oid);
            auto viewObj = relatives.find(oid);
            if (viewObj != relatives.end()) {
                objBuilder[STR_CATEGORY_ID] = viewObj->second.categoryId();
                objBuilder[STR_TITLE] = title(&viewObj->second);
            };
        };
    }
}

void
putRelativesDiff(const DiffDetails& details, json::ObjectBuilder& modified)
{
    bool hasSelfInRelativesDiff = false;
    const auto selfOid =
        details.fromObjectVersion
        ? details.fromObjectVersion->id()
        : (details.toObjectVersion ? details.toObjectVersion->id() : 0);

    modified[STR_MASTERS] << [&](json::ObjectBuilder mastersBuilder) {
        for (const auto& masterDiff : details.mastersDiff) {
            mastersBuilder[masterDiff.first] << [&](json::ObjectBuilder diffBuilder) {
                hasSelfInRelativesDiff = hasSelfInRelativesDiff ||
                    masterDiff.second.added.count(selfOid) ||
                    masterDiff.second.deleted.count(selfOid);
                diffBuilder[STR_ADDED] << [&](json::ArrayBuilder addedBuilder) {
                    putRelativesViewObjects(
                        masterDiff.second.added,
                        details.addedRelatives,
                        addedBuilder,
                        std::nullopt);
                };
                diffBuilder[STR_REMOVED] << [&](json::ArrayBuilder deletedBuilder) {
                    putRelativesViewObjects(
                        masterDiff.second.deleted,
                        details.removedRelatives,
                        deletedBuilder,
                        std::nullopt);
                };
            };
        }
    };

    std::optional<TOid> excludeObject =
        hasSelfInRelativesDiff
            ? std::optional<TOid>(selfOid)
            : std::optional<TOid>();

    modified[STR_SLAVES] << [&](json::ObjectBuilder slavesBuilder) {
        for (const auto& slaveDiff : details.slavesDiff) {
            slavesBuilder[slaveDiff.first] << [&](json::ObjectBuilder diffBuilder) {
                diffBuilder[STR_ADDED] << [&](json::ArrayBuilder addedBuilder) {
                    putRelativesViewObjects(
                            slaveDiff.second.added,
                            details.addedRelatives,
                            addedBuilder,
                            excludeObject);
                };
                diffBuilder[STR_REMOVED] << [&](json::ArrayBuilder deletedBuilder) {
                    putRelativesViewObjects(
                        slaveDiff.second.deleted,
                        details.removedRelatives,
                        deletedBuilder,
                        excludeObject);
                };
            };
        }
    };
}

void
putGeoms(const std::vector<Geom>& geomArray, json::ArrayBuilder& jsonArray)
{
    for (const auto& geom : geomArray) {
        if (geom.isNull()) {
            continue;
        }
        jsonArray << json::Verbatim(prepareGeomJson(geom, GEODETIC_GEOM_PRECISION));
    }
}
void
putCategoryDiff(const boost::optional<revision::StringDiff>& categoryDiff, json::ObjectBuilder& modified)
{
    if (!categoryDiff) {
        return;
    }
    modified[STR_CATEGORY_ID] << [&](json::ObjectBuilder diffBuilder) {
        diffBuilder[STR_BEFORE] = categoryDiff->before;
        diffBuilder[STR_AFTER] = categoryDiff->after;
    };
}

void
putGeometryDiff(const GeomDiff& geomDiff, json::ObjectBuilder& modified)
{
    if (geomDiff.empty()) {
        return;
    }
    modified[STR_GEOMETRY] << [&](json::ObjectBuilder geomDiffBuilder) {
        geomDiffBuilder[STR_LIMIT_EXCEEDED] = geomDiff.isLimitExceeded();
        geomDiffBuilder[STR_BEFORE] << [&](json::ArrayBuilder geomArrayBuilder) {
            putGeoms(geomDiff.before(), geomArrayBuilder);
        };
        geomDiffBuilder[STR_AFTER] << [&](json::ArrayBuilder geomArrayBuilder) {
            putGeoms(geomDiff.after(), geomArrayBuilder);
        };
    };
}

void
putThreadStopDiffRecord(
    json::ArrayBuilder& recordsArray,
    const ThreadStopDiffRecord& stopDiff,
    bool outer)
{
    recordsArray << [&](json::ObjectBuilder stopDiffBuilder) {
        stopDiffBuilder[STR_ID] = common::idToJson(stopDiff.id);
        stopDiffBuilder[STR_TITLE] = stopDiff.title;
        if (outer) {
            stopDiffBuilder[STR_END] = true;
        }
        stopDiffBuilder[STR_TYPE] = toString(stopDiff.type);
        if (stopDiff.attrsDiff) {
            stopDiffBuilder[STR_ATTRS] << [&](json::ArrayBuilder attrsDiffBuilder) {
                for (const auto& attrDiff : *stopDiff.attrsDiff) {
                    putAttributeDiff(attrsDiffBuilder, attrDiff,
                        stopDiff.type == ThreadStopDiffRecord::Type::Modified,
                        boost::none);
                }
            };
        }
    };
}
} // namespace

std::string
JSONFormatter::operator ()(const ResultType<GetCommitDiff>& result)
{
    ASSERT(result.commitModel);
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        // FIXME pkostikov: remove action from here in schemas; use action in commit
        resultBuilder[STR_ACTION] = result.commitModel->commit().action();
        resultBuilder[STR_COMMIT] << [&](json::ObjectBuilder commitBuilder) {
            putCommitModel(commitBuilder, *result.commitModel);
        };
        putJsonIds(resultBuilder, STR_AFFECTED_OBJECTS_IDS,
            result.commitModel->contextPrimaryObjectId()
              ? TOIds{*result.commitModel->contextPrimaryObjectId()}
              : affectedObjectIds(result.commitModel->commit()));
        if (!result.details) {
            return;
        }
        const auto& details = *result.details;
        if (!details.envelope.isNull()) {
            resultBuilder[STR_BOUNDS] = json::Verbatim(json(details.envelope));
        }
        if (details.anyObject()) {
            resultBuilder[STR_GEO_OBJECT] << [&](json::ObjectBuilder geoObj) {
                putObjectIdentity(geoObj, details.anyObject().get());
            };
        }
        if (!result.categoryId.empty()) {
            resultBuilder[STR_CATEGORY_ID] = result.categoryId;
        }
        if (details.hasErrors()) {
            resultBuilder[STR_ERRORS] << [&](json::ArrayBuilder errors) {
                if (details.threadStopDiff && details.threadStopDiff->brokenSequence) {
                    errors << STR_THREAD_STOPS_BROKEN_SEQUENCE;
                }
            };
            return;
        }
        resultBuilder[STR_MODIFIED] << [&](json::ObjectBuilder modified) {
            putCategoryDiff(details.categoryDiff, modified);
            putGeometryDiff(details.geomDiff, modified);
            if (!details.toObjectVersion || !details.toObjectVersion->isDeleted()) {
                putRelativesDiff(details, modified);
            }
            if (!details.attributesDiff.empty() ||
                !areTableAttrsEqual(details.fromObjectVersion, details.toObjectVersion))
            {
                modified[STR_ATTRS] << [&](json::ArrayBuilder attrsDiffBuilder) {
                    if (!details.attributesDiff.empty()) {
                        for (const auto& attrDiff : details.attributesDiff) {
                            putAttributeDiff(
                                attrsDiffBuilder,
                                attrDiff,
                                details.fromObjectVersion != nullptr,
                                details.categoryDiff);
                        }
                    }
                    if (details.toObjectVersion && !details.toObjectVersion->isDeleted()) {
                        putTableAttributesDiff(attrsDiffBuilder, details);
                    }
                };
            }
            if (details.threadStopDiff && !details.threadStopDiff->brokenSequence) {
                modified[STR_THREAD_STOPS] << [&](json::ArrayBuilder threadDiffBuilder) {
                    const auto& diffs = details.threadStopDiff->diffs;
                    const size_t totalRecords = diffs.size();
                    for (size_t i = 0; i < totalRecords; ++i) {
                        const auto& stopDiff = diffs[i];
                        if (stopDiff.type == ThreadStopDiffRecord::Type::Context
                            &&
                            (i == 0 || diffs[i - 1].type == ThreadStopDiffRecord::Type::Context)
                            &&
                            (i == totalRecords - 1 || diffs[i + 1].type == ThreadStopDiffRecord::Type::Context)
                            ) {
                                continue;
                        }
                        putThreadStopDiffRecord(threadDiffBuilder,
                            diffs[i],
                            i == 0 || i == totalRecords - 1);
                    }
                };
            }
        };
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<GetCommitGeomDiff>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        putGeometryDiff(result.geomDiff, resultBuilder);
    };
    return builder.str();
}

} // namespace wiki
} // namespace maps
