#include "json_formatter.h"

#include "json_writer.h"
#include <maps/wikimap/mapspro/services/editor/src/moderation.h>
#include <maps/wikimap/mapspro/services/editor/src/social_utils.h>
#include <maps/wikimap/mapspro/services/editor/src/commit.h>

#include <boost/lexical_cast.hpp>

#include <yandex/maps/wiki/common/date_time.h>
#include <maps/libs/json/include/builder.h>
#include <maps/wikimap/mapspro/libs/gdpr/include/user.h>
#include <maps/wikimap/mapspro/libs/social_serv_serialize/include/jsonize_comment.h>
#include <maps/wikimap/mapspro/libs/social_serv_serialize/include/jsonize_feedback_task.h>

#include <functional>

namespace maps {
namespace wiki {

namespace {

const std::string STR_REGION = "region";
const std::string STR_REGIONS = "regions";
const std::string STR_MODE = "mode";

const std::string STR_EVENT = "event";
const std::string STR_EVENTS = "events";
const std::string STR_SUBSCRIBER = "subscriber";
const std::string STR_FEED_TYPE = "feedType";
const std::string STR_HAS_TASKS = "hasTasks";

const std::string STR_ALERTS = "alerts";
const std::string STR_PRIORITY = "priority";
const std::string STR_DESCRIPTION = "description";

const std::string STR_DATA = "data";
const std::string STR_RESOLUTION = "resolution";
const std::string STR_RESOLVED = "resolved";
const std::string STR_CLOSED = "closed";
const std::string STR_AVAILABLE = "available";
const std::string STR_ACQUIRED = "acquired";
const std::string STR_OLD = "old";

const std::string STR_TASK_COUNTERS = "taskCounters";
const std::string STR_TASK_COUNTERS_BY_CATEGORY_GROUP = "taskCountersByCategoryGroup";

const std::string STR_COMMENT = "comment";
const std::string STR_COMMENTS = "comments";
const std::string STR_ALL = "all";
const std::string STR_NOT_READ = "notRead";
const std::string STR_MODERATORS = "moderators";
const std::string STR_OLDEST_TASK_SINCE = "oldestTaskActiveSince";
const std::string STR_RECENT_NEW = "recentNew";
const std::string STR_RECENT_PROCESSED = "recentProcessed";
const std::string STR_CATEGORY_GROUPS = "categoryGroups";
const std::string STR_EVENT_TYPES = "eventTypes";
const std::string STR_OLD_TASK_AGE_IN_HOURS = "oldTaskAgeInHours";

const std::string STR_TODAY = "today";
const std::string STR_COMMITS_COUNT = "commitsCount";

void
jsonize(json::ObjectBuilder& builder, const social::Resolved& resolved)
{
    builder[STR_RESOLVED] << [&](json::ObjectBuilder o) {
        o[STR_UID] = common::idToJson(gdpr::User(resolved.uid()).uid());
        o[STR_DATE] = common::canonicalDateTimeString(resolved.date(), common::WithTimeZone::Yes);
        o[STR_RESOLUTION] = boost::lexical_cast<std::string>(resolved.resolution());
    };
}

void
jsonize(json::ObjectBuilder& builder, const social::Closed& closed)
{
    builder[STR_CLOSED] << [&](json::ObjectBuilder o) {
        o[STR_UID] = common::idToJson(gdpr::User(closed.uid()).uid());
        o[STR_DATE] = common::canonicalDateTimeString(closed.date(), common::WithTimeZone::Yes);
        o[STR_RESOLUTION] = boost::lexical_cast<std::string>(closed.resolution());
    };
}

void
jsonize(json::ObjectBuilder& builder, const social::Locked& locked)
{
    builder[STR_ACQUIRED] << [&](json::ObjectBuilder o) {
        o[STR_UID] = common::idToJson(gdpr::User(locked.uid()).uid());
        o[STR_DATE] = common::canonicalDateTimeString(locked.date(), common::WithTimeZone::Yes);
    };
}

void
jsonize(json::ObjectBuilder& builder, const social::EventAlert& alert)
{
    builder[STR_PRIORITY] = alert.priority();
    builder[STR_DESCRIPTION] = alert.description();
    if (alert.objectId()) {
        builder[STR_OBJECT_ID] = common::idToJson(alert.objectId());
    }
}

void
jsonize(json::ObjectBuilder& builder, const social::Comment& comment)
{
    socialsrv::serialize::jsonize(builder, comment);
}

void
jsonize(
    json::ObjectBuilder& builder,
    const social::Event& event,
    const boost::optional<CommitModel>& commitModel = boost::none,
    boost::optional<TUid> userId = boost::none)
{
    auto authorUidStr = common::idToJson(gdpr::User(event.createdBy()).uid());
    builder[STR_ID] = common::idToJson(event.id());
    builder[STR_UID] = authorUidStr;
    builder[STR_DATE] =  common::canonicalDateTimeString(event.createdAt(), common::WithTimeZone::Yes);

    builder[STR_ACTION] = event.action();
    if (!event.bounds().empty()) {
        builder[STR_BOUNDS] = json::Verbatim(event.bounds());
    }

    const auto& comment = event.comment();
    if (comment) {
        builder[STR_COMMENT] = [&](json::ObjectBuilder builder) {
            jsonize(builder, *comment);
        };
    }

    builder[STR_DATA] << [&](json::ObjectBuilder dataBuilder) {
        const auto& commitData = event.commitData();

        if (commitModel || commitData) {
            dataBuilder[STR_COMMIT] << [&](json::ObjectBuilder commitBuilder) {
                if (commitModel) {
                    putCommitModel(commitBuilder, *commitModel);
                } else {
                    commitBuilder[STR_ID] = common::idToJson(commitData->commitId());
                    commitBuilder[STR_UID] = authorUidStr;
                    commitBuilder[STR_DATE] = common::canonicalDateTimeString(event.createdAt(), common::WithTimeZone::Yes);
                    commitBuilder[STR_ACTION] = commitData->action();
                    if (event.primaryObjectData()) {
                        const auto& primaryData = *event.primaryObjectData();
                        commitBuilder[STR_PRIMARY_OBJECT_ID] = common::idToJson(primaryData.id());

                        const auto& editNotes = objectEditNotesTree(primaryData.editNotes());
                        commitBuilder[STR_EDIT_NOTES] << [&](json::ObjectBuilder notesBuilder) {
                            putEditNotes(notesBuilder, editNotes.rootNotes);
                        };
                    }
                }
                if (commitData && !commitData->bounds().empty()) {
                    commitBuilder[STR_BOUNDS] = json::Verbatim(commitData->bounds());
                }
            };
        }

        if (event.primaryObjectData()) {
            const auto& primaryData = *event.primaryObjectData();
            dataBuilder[STR_GEO_OBJECT] << [&](json::ObjectBuilder objectBuilder) {
                objectBuilder[STR_ID] = common::idToJson(primaryData.id());
                objectBuilder[STR_CATEGORY_ID] = primaryData.categoryId();
                objectBuilder[STR_TITLE] = primaryData.screenLabel();
            };
        }

        if (event.feedback()) {
            ASSERT(userId != boost::none);
            dataBuilder[STR_FEEDBACK_TASK] << [&](json::ObjectBuilder objectBuilder) {
                socialsrv::serialize::taskBriefToJson(
                    objectBuilder,
                    *event.feedback(),
                    *userId
                );
            };
        }
    };
}

void
jsonize(
    json::ObjectBuilder& builder,
    const social::Task& task,
    const social::EventAlerts& alerts,
    boost::optional<TUid> userId)
{
    builder[STR_ID] = common::idToJson(task.id());
    if (task.isResolved()) {
        jsonize(builder, task.resolved());
    }
    if (task.isClosed()) {
        jsonize(builder, task.closed());
    }
    if (task.isLocked()) {
        jsonize(builder, task.locked());
    }
    builder[STR_EVENT] << [&](json::ObjectBuilder eventBuilder) {
        jsonize(eventBuilder, task.event(), boost::none, userId);
    };
    builder[STR_ALERTS] << [&](json::ArrayBuilder alertsBuilder) {
        for (const auto& alert : alerts) {
            alertsBuilder << [&](json::ObjectBuilder alertBuilder) {
                jsonize(alertBuilder, alert);
            };
        }
    };
}

std::string
toJsonTask(
    const social::Task& task,
    const social::EventAlerts& alerts,
    boost::optional<TUid> requesterUid)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder taskBuilder) {
        taskBuilder[STR_TASK] = [&](json::ObjectBuilder builder) {
            jsonize(builder, task, alerts, requesterUid);
        };
    };
    return builder.str();
}

template <typename Result>
std::string
toJsonTaskIds(const Result& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_TOKEN] = result.token;
        putTaskIds(resultBuilder, result.taskIds);
    };
    return builder.str();
}

template <typename Result>
std::string
toJsonTaskIdsAndRevertingCommit(const Result& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_TOKEN] = result.token;
        putTaskIds(resultBuilder, result.taskIds);
        if (result.revertingCommitModel) {
            resultBuilder[STR_REVERTING_COMMIT] = [&](json::ObjectBuilder commitBuilder) {
                putCommitModel(commitBuilder, *result.revertingCommitModel);
            };
        }
    };
    return builder.str();
}

template <typename Result>
std::string
toJsonComment(const Result& result)
{
    ASSERT(result.comment);
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_TOKEN] = result.token;
        resultBuilder[STR_COMMENT] = [&](json::ObjectBuilder builder) {
            jsonize(builder, *(result.comment));
        };
    };
    return builder.str();
}

void
jsonize(json::ObjectBuilder& builder, const CommentsCounters& counters)
{
    builder[STR_ALL] = counters.all;
    builder[STR_NOT_READ] = counters.notRead;
    builder[STR_TOTAL_COUNT] = counters.totalCount;
}

} // namespace

std::string
JSONFormatter::operator ()(const ResultType<SocialModerationTasksAcquire>& result)
{
    JsonBuilder builder;
    const social::EventAlerts EMPTY_ALERTS;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_TOKEN] = result.token;
        resultBuilder[STR_TASKS] << [&](json::ArrayBuilder arrayBuilder) {
            for (const auto& task : result.tasks) {
                auto alertsIt = result.alertsByTaskId.find(task.id());
                const auto& alerts =
                    alertsIt != result.alertsByTaskId.end()
                    ? alertsIt->second
                    : EMPTY_ALERTS;
                arrayBuilder << [&](json::ObjectBuilder builder) {
                    // Passing boost::none instead of result.uid as it had been used for feedback moderation only
                    jsonize(builder, task, alerts, boost::none);
                };
            }
        };
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<SocialModerationTasksAcquireBlocking>& result)
{
    return operator()(
        static_cast<ResultType<SocialModerationTasksAcquire>>(result)
    );
}

std::string
JSONFormatter::operator ()(const ResultType<SocialModerationTasksRelease>& result)
{
    return toJsonTaskIds(result);
}

std::string
JSONFormatter::operator ()(const ResultType<GetSocialModerationTask>& result)
{
    ASSERT(result.task);
    return toJsonTask(*result.task, result.alerts, result.requesterUid);
}

std::string
JSONFormatter::operator ()(const ResultType<GetSocialModerationTaskByCommit>& result)
{
    ASSERT(result.task);
    return toJsonTask(*result.task, result.alerts, boost::none);
}

std::string
JSONFormatter::operator ()(const ResultType<SocialModerationTasksResolve>& result)
{
    return toJsonTaskIdsAndRevertingCommit(result);
}

std::string
JSONFormatter::operator ()(const ResultType<SocialModerationTasksClose>& result)
{
    return toJsonTaskIdsAndRevertingCommit(result);
}

std::string
JSONFormatter::operator ()(const ResultType<SocialModerationTasksDefer>& result)
{
    return toJsonTaskIds(result);
}

std::string
JSONFormatter::operator ()(const ResultType<SocialModerationTasksStat>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_TASK_COUNTERS] << [&](json::ObjectBuilder countersBuilder) {
            countersBuilder[STR_TODAY] = result.todayProcessedCount;
        };
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<GetSocialModerationTasks>& result)
{
    JsonBuilder builder;
    const social::EventAlerts EMPTY_ALERTS;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_TASKS] << [&](json::ArrayBuilder arrayBuilder) {
            for (const auto& task : result.tasks) {
                auto alertsIt = result.alertsByTaskId.find(task.id());
                const auto& alerts =
                        alertsIt != result.alertsByTaskId.end()
                        ? alertsIt->second
                        : EMPTY_ALERTS;
                arrayBuilder << [&](json::ObjectBuilder builder) {
                    jsonize(builder, task, alerts, result.uid);
                };
            }
        };
        resultBuilder[STR_HAS_MORE] = result.hasMore == social::HasMore::Yes;
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<SocialModerationRegions>& result)
{
    auto serializeCounters = [](const social::TaskCounts& counters)
        -> std::function<void(json::ObjectBuilder)>
    {
        return [&counters](json::ObjectBuilder builder)
        {
            builder[STR_AVAILABLE] = counters.available();
            builder[STR_ACQUIRED] = counters.acquired();
            builder[STR_OLD] = counters.old();
            builder[STR_TOTAL] = counters.total();
        };
    };

    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_REGIONS] << [&](json::ArrayBuilder arrayBuilder) {
            for (const auto& region : result.regions) {
                arrayBuilder << [&](json::ObjectBuilder ob) {
                    ob[STR_GEO_OBJECT] << [&](json::ObjectBuilder geoOb) {
                        geoOb[STR_ID] = common::idToJson(region.id);
                        geoOb[STR_CATEGORY_ID] = region.categoryId;
                        geoOb[STR_TITLE] = region.title;
                        geoOb[STR_GEOMETRY] =
                            json::Verbatim(prepareGeomJson(
                                region.geometry, GEODETIC_GEOM_PRECISION));
                        ASSERT(!region.envelope.isNull());
                        geoOb[STR_BOUNDS] = json::Verbatim(json(region.envelope));
                    };
                    ob[STR_MODE] << moderation::toAclRoleName(region.mode);
                    ob[STR_TASK_COUNTERS] << serializeCounters(region.taskCounters);
                    ob[STR_OLD_TASK_AGE_IN_HOURS] << region.oldTaskAgeInHours;

                    if (region.taskCountersByCategoryGroup.empty()) {
                        return;
                    }

                    ob[STR_TASK_COUNTERS_BY_CATEGORY_GROUP] <<
                        [&](json::ObjectBuilder byGroup) {
                        for (const auto& pair : region.taskCountersByCategoryGroup) {
                            byGroup[pair.first] << serializeCounters(pair.second);
                        }
                    };
                };
            }
        };
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<GetSocialModerationStat>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_UID] = common::idToJson(result.uid);
        resultBuilder[STR_HAS_TASKS] = result.hasTasks;
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<GetSocialEvent>& result)
{
    JsonBuilder builder;
    if (result.event) {
        builder << [&](json::ObjectBuilder resultBuilder) {
            jsonize(resultBuilder, *(result.event), boost::none, boost::none);
        };
    }
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<SocialModerationDashboard>& result)
{
    class RegionSerializer {
    public:
        RegionSerializer(
                TOid regionId,
                const ResultType<SocialModerationDashboard>& result)
            : regionId_(regionId)
            , result_(result)
        { }

        void json(json::ObjectBuilder regBuilder) const
        {
            regBuilder[STR_ID] = common::idToJson(regionId_);

            const auto& stat = result_.regionStatsByOid.at(regionId_);
            regBuilder[STR_TITLE] = stat.title;
            regBuilder[STR_MODERATORS] = stat.modsAssigned;
            regBuilder[STR_TOTAL_COUNT] = stat.activeTaskCount;
            regBuilder[STR_OLDEST_TASK_SINCE] = common::canonicalDateTimeString(
                stat.oldestActiveTaskActiveSince,
                common::WithTimeZone::Yes);
            regBuilder[STR_RECENT_NEW] = stat.recentNewTaskCount;
            regBuilder[STR_RECENT_PROCESSED] = stat.recentProcessedTaskCount;

            if (!result_.regionsHierarchy.empty()) {
                regBuilder[STR_CHILDREN] = [&](json::ArrayBuilder childrenBuilder) {
                    for (TOid childRegionId : result_.regionsHierarchy.at(regionId_)) {
                        childrenBuilder << RegionSerializer(childRegionId, result_);
                    }
                };
            }
        }

    private:
        TOid regionId_;
        const ResultType<SocialModerationDashboard>& result_;
    };

    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_MODE] << moderation::toAclRoleName(result.mode);
        resultBuilder[STR_REGIONS] << [&](json::ArrayBuilder arrayBuilder) {
            if (!result.regionsHierarchy.empty()) {
                const TOid REGION_HIERARCHY_ROOT_ID = 0;
                for (TOid rootRegionId :
                        result.regionsHierarchy.at(REGION_HIERARCHY_ROOT_ID)) {
                    arrayBuilder << RegionSerializer(rootRegionId, result);
                }
            } else {
                for (const auto& oidStatPair : result.regionStatsByOid) {
                    arrayBuilder << RegionSerializer(oidStatPair.first, result);
                }
            }
        };
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<SocialModerationDashboardMeta>& result)
{
    JsonBuilder builder;
    builder <<
        [&](json::ObjectBuilder objectBuilder) {
            objectBuilder[STR_CATEGORY_GROUPS] =
                [&](json::ArrayBuilder objectsArrayBuilder) {
                    for (const auto& categoryGroup : result.categoryGroups) {
                        objectsArrayBuilder << categoryGroup;
                    }
                };
            objectBuilder[STR_EVENT_TYPES] =
                [&](json::ArrayBuilder objectsArrayBuilder) {
                    for (const auto& eventType : result.eventTypes) {
                        objectsArrayBuilder << eventType;
                    }
                };
        };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<GetRegularSocialFeedMeta>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder objectBuilder) {
        objectBuilder[STR_CATEGORY_GROUPS] = [&](json::ArrayBuilder objectsArrayBuilder) {
            for (const auto& categoryGroup : result.categoryGroups) {
                objectsArrayBuilder << categoryGroup;
            }
        };
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<GetSocialFeed>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_FEED_TYPE] = boost::lexical_cast<std::string>(result.feedType);
        resultBuilder[STR_BRANCH_ID] = common::idToJson(result.branchId);
        resultBuilder[STR_SUBSCRIBER] = common::idToJson(result.subscriber);
        if (result.totalCount) {
            resultBuilder[STR_TOTAL_COUNT] = *result.totalCount;
        }
        if (result.page) {
            resultBuilder[STR_PAGE] = result.page;
        } else {
            resultBuilder[STR_HAS_MORE] = result.hasMore;
        }
        resultBuilder[STR_PER_PAGE] = result.perPage;

        resultBuilder[STR_EVENTS] << [&](json::ArrayBuilder arrayBuilder) {
            for (const auto& eventModel : result.eventModels) {
                arrayBuilder << [&](json::ObjectBuilder builder) {
                    jsonize(builder, eventModel.event, eventModel.commitModel);
                    ASSERT(eventModel.commitModel.approveStatus());
                    builder[STR_APPROVE_STATUS] = boost::lexical_cast<std::string>(
                        *eventModel.commitModel.approveStatus());
                };
            }
        };
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<GetSocialSubscriptions>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_REGIONS] << [&](json::ArrayBuilder regionsBuilder) {
            for (const auto& region : result.regionIdToTitle) {
                regionsBuilder << [&](json::ObjectBuilder geoObjectBuilder) {
                    geoObjectBuilder[STR_ID] = common::idToJson(region.first);
                    geoObjectBuilder[STR_CATEGORY_ID] = CATEGORY_FEED_REGION;
                    geoObjectBuilder[STR_TITLE] = region.second;
                };
            }
        };
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<GetSocialFeedbackTaskHistory>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder builder) {
        builder[STR_HAS_MORE] = result.hasMore;
        builder[STR_PER_PAGE] = result.perPage;
        builder[STR_EVENTS] = [&](json::ArrayBuilder builder) {
            for (const auto& eventModel: result.eventModels) {
                builder << [&](json::ObjectBuilder builder) {
                    jsonize(builder, eventModel.event, eventModel.commitModel);
                    ASSERT(eventModel.commitModel.approveStatus());
                    builder[STR_APPROVE_STATUS] = boost::lexical_cast<std::string>(
                        *eventModel.commitModel.approveStatus());
                };
            }
        };
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<GetSocialFeedbackTaskStat>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder builder) {
        builder[STR_COMMITS_COUNT] = result.commitsCount;
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<GetCommentsFeed>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_TOTAL_COUNT] = result.totalCount;

        resultBuilder[STR_HAS_MORE] = (result.hasMore == social::HasMore::Yes);

        resultBuilder[STR_COMMENTS] << [&](json::ArrayBuilder arrayBuilder) {
            for (const auto& comment : result.comments) {
                arrayBuilder << [&](json::ObjectBuilder builder) {
                    jsonize(builder, comment);
                };
            }
        };
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<GetCommentsStat>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        jsonize(resultBuilder, result.counters);
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<ClearUserComments>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_COUNT] = result.count;
        resultBuilder[STR_TOKEN] = result.token;
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<CommentsCreate>& result)
{
    return toJsonComment(result);
}

std::string
JSONFormatter::operator ()(const ResultType<CommentsDelete>& result)
{
    return toJsonComment(result);
}

} // namespace wiki
} // namespace maps
