#include "json_formatter.h"
#include "json_writer.h"

#include <yandex/maps/wiki/common/date_time.h>
#include <yandex/maps/wiki/common/geom.h>
#include <yandex/maps/wiki/common/json_helpers.h>

namespace maps::wiki {

namespace vs = validator::storage;

namespace {

void serializeNode(
    json::ArrayBuilder arrayBuilder,
    const ValidationStatsNode& node)
{
    arrayBuilder << [&](json::ObjectBuilder nodeBuilder) {
        nodeBuilder[STR_FIELD] = node.field;
        nodeBuilder[STR_VALUE] = node.value;
        nodeBuilder[STR_COUNT] = node.count;
        nodeBuilder[STR_CHILDREN] << [&](json::ArrayBuilder childrenBuilder) {
            for (const auto& child : node.children) {
                serializeNode(childrenBuilder, child);
            }
        };
    };
}

void serializeStatistics(
    json::ObjectBuilder statsBuilder,
    const std::vector<ValidationStatsNode>& severityStats,
    const std::vector<ValidationStatsNode>& regionTypeStats,
    const std::vector<ValidationStatsNode>& checkStats)
{
    statsBuilder[STR_SEVERITIES] << [&](json::ArrayBuilder severityStatsBuilder) {
        for (const auto& node : severityStats) {
            serializeNode(severityStatsBuilder, node);
        }
    };
    statsBuilder[STR_REGION_TYPES] << [&](json::ArrayBuilder regionTypeStatsBuilder) {
        for (const auto& node : regionTypeStats) {
            serializeNode(regionTypeStatsBuilder, node);
        }
    };
    statsBuilder[STR_CHECKS] << [&](json::ArrayBuilder checkStatsBuilder) {
        for (const auto& node : checkStats) {
            serializeNode(checkStatsBuilder, node);
        }
    };
}

template<typename Builder>
void serializeMessage(
    Builder jsonBuilder,
    const vs::StoredMessageDatum& storedMessage,
    TUid viewedByUid)
{
    jsonBuilder << [&](json::ObjectBuilder messageBuilder) {
        const validator::Message& message = storedMessage.message();
        {
            std::ostringstream os;
            os << storedMessage.id();
            messageBuilder[STR_ID] = os.str();
        }
        {
            std::ostringstream os;
            os << message.attributes().severity;
            messageBuilder[STR_SEVERITY] = os.str();
        }
        {
            std::ostringstream os;
            os << message.attributes().regionType;
            messageBuilder[STR_REGION_TYPE] = os.str();
        }
        messageBuilder[STR_CHECK] = message.attributes().checkId;
        messageBuilder[STR_DESCRIPTION] = message.attributes().description;
        messageBuilder[STR_ACTIVE] = storedMessage.isActive();

        const auto& exclusionInfo = storedMessage.exclusionInfo();
        if (exclusionInfo) {
            messageBuilder[STR_CREATEDAT] = common::canonicalDateTimeString(
                exclusionInfo->createdAt, common::WithTimeZone::Yes);
            messageBuilder[STR_CREATEDBY] = common::idToJson(exclusionInfo->createdBy);
            messageBuilder[STR_VIEWED] = exclusionInfo->viewedBy.count(viewedByUid) > 0;
        }

        if (!message.geomWkb().empty()) {
            Geom geom(message.geomWkb());
            messageBuilder[STR_CENTER] = json::Verbatim(
                prepareGeomJson(geom.center(), GEODETIC_GEOM_PRECISION));
            if (geom.geometryTypeName() != common::Geom::geomTypeNamePoint) {
                messageBuilder[STR_GEOMETRY] = json::Verbatim(
                    prepareGeomJson(geom, GEODETIC_GEOM_PRECISION));
            }
        }

        messageBuilder[STR_OBJECT_IDS] << [&](json::ArrayBuilder objectIdsBuilder) {
            for (const auto& revId : message.revisionIds()) {
                objectIdsBuilder << common::idToJson(revId.objectId());
            }
        };
    };
}

} // namespace

std::string
JSONFormatter::operator ()(const ResultType<GetValidationExclusions>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        resultBuilder[STR_MESSAGES] << [&](json::ArrayBuilder messagesBuilder) {
            for (const auto& exclusion : result.exclusions) {
                serializeMessage(messagesBuilder, exclusion, result.uid);
            }
        };
        resultBuilder[STR_STATISTICS] << [&](json::ObjectBuilder statsBuilder) {
            serializeStatistics(statsBuilder,
                result.severityStats, result.regionTypeStats, result.checkStats);
        };
        resultBuilder[STR_TOTAL_COUNT] = result.totalCount;
        if (result.hasMore) {
            resultBuilder[STR_HAS_MORE] = *result.hasMore;
        }
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<SaveValidationExclusion>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        serializeMessage(resultBuilder[STR_MESSAGE], *result.exclusion, result.uid);
    };
    return builder.str();
}

std::string
JSONFormatter::operator ()(const ResultType<ViewValidationExclusion>& result)
{
    JsonBuilder builder;
    builder << [&](json::ObjectBuilder resultBuilder) {
        serializeMessage(resultBuilder[STR_MESSAGE], *result.exclusion, result.uid);
    };
    return builder.str();
}

} // namespace maps::wiki
