#include "json_parser.h"

#include "json_common.h"
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/utils.h>
#include <maps/wikimap/mapspro/services/editor/src/objects_cache.h>

#include <yandex/maps/wiki/revision/exception.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <maps/libs/common/include/exception.h>

#include <string>

namespace maps {
namespace wiki {

namespace {

const std::string JSON_STR_EDIT_CONTEXT = "editContext";
const std::string JSON_STR_DIFF_ALERTS = "diffAlerts";
const std::string JSON_STR_SPLIT_POINTS = "splitPoints";
const std::string JSON_STR_SPLIT_LINES = "splitLines";

} // namespace

TOIds
JsonParser::objectIdSet(const json::Value& jsonIds)
{
    TOIds idSet;
    for (const auto& id: jsonIds) {
        REQUIRE(
            idSet.insert(std::stoull(id.toString())).second,
            "Not unique id " << id.toString()
        );
    }
    return idSet;
}

std::vector<TOid>
JsonParser::objectIdVector(const json::Value& jsonIds)
{
    std::vector<TOid> ids;
    for (const auto& id: jsonIds) {
        ids.push_back(std::stoull(id.toString()));
    }

    return ids;
}

Geom
JsonParser::readObjectGeometry(const json::Value& jsonObject)
{
    if (!jsonObject.hasField(STR_GEOMETRY)) {
        return Geom();
    }
    return Geom(createGeomFromJson(jsonObject[STR_GEOMETRY]));
}

UUID
JsonParser::readObjectUUID(const json::Value& jsonObject)
{
    return common::readOptionalField<UUID>(jsonObject, STR_UUID, s_emptyString);
}

TOid
JsonParser::readObjectId(const json::Value& jsonObject)
{
    return std::stoll(jsonObject[STR_ID].as<std::string>("0"));
}

TRevisionId
JsonParser::readObjectRevision(const json::Value& jsonObject)
{
    return common::readOptionalField<TRevisionId>(jsonObject, STR_REVISION_ID, TRevisionId());
}

std::string
JsonParser::readObjectCategory(const json::Value& jsonObject)
{
    return common::readOptionalField<std::string>(jsonObject, STR_CATEGORY_ID, s_emptyString);
}

boost::optional<std::string>
JsonParser::readObjectRichContent(const json::Value& jsonObject)
{
    if (!jsonObject.hasField(STR_RICH_CONTENT) || !jsonObject[STR_RICH_CONTENT].isString()) {
        return boost::optional<std::string>("");
    }
    const auto& value = jsonObject[STR_RICH_CONTENT].as<std::string>();
    return value.empty()
        ? boost::optional<std::string>()
        : boost::optional<std::string>(value);
}

namespace {

StringMultiMap
readPlainAttributes(const json::Value& jsonAttrs)
{
    StringMultiMap attributes;
    const auto editorCfg = cfg()->editor();
    auto fieldNames = jsonAttrs.fields();
    for (const auto& fieldName : fieldNames) {
        if ( !editorCfg->isAttributeDefined(fieldName)
             || editorCfg->attribute(fieldName)->table()) {
             continue;
        }
        const auto& attrDef = editorCfg->attribute(fieldName);
        const auto& field = jsonAttrs[fieldName];
        if (field.isArray() && attrDef->valueType() != ValueType::Json) {
            if (field.empty() && attrDef->multiValue()) {
                attributes.insert(
                    {
                        fieldName,
                        attrDef->allowedValue(s_emptyString)
                    });
            }
            for (const auto& val : field) {
                attributes.insert({fieldName, attrDef->allowedValue(val.toString())});
            }
        } else if (attrDef->valueType() == ValueType::Json) {
            std::stringstream valueStream;
            valueStream << field;
            attributes.insert({fieldName, attrDef->allowedValue(valueStream.str())});
        } else {
            attributes.insert({fieldName, attrDef->allowedValue(field.toString())});
        }
    }
    return attributes;
}

} // namespace

StringMultiMap
JsonParser::readObjectAttributes(const json::Value& jsonObject)
{
    return jsonObject.hasField(STR_ATTRS)
            ? readPlainAttributes(jsonObject[STR_ATTRS])
            : StringMultiMap();
}

TableAttributesValues
JsonParser::readObjectTableAttributes(const json::Value& jsonObject)
{
    TableAttributesValues tableAttributes;
    if (!jsonObject.hasField(STR_ATTRS)) {
        return tableAttributes;
    }
    const auto& jsonAttrs = jsonObject[STR_ATTRS];
    auto fieldNames = jsonAttrs.fields();
    const auto editorCfg = cfg()->editor();
    for (const auto& fieldName : fieldNames) {
        if (!editorCfg->isAttributeDefined(fieldName)) {
            continue;
        }
        const auto& attrDef = editorCfg->attribute(fieldName);
        if (!attrDef->table()) {
            continue;
        }
        TableValues tableValues;
        DEBUG() << "TABLE: " << fieldName;
        for (const auto& values : jsonAttrs[fieldName]) {
            StringMultiMap valuesMap = readPlainAttributes(values);
            DEBUG() << "TABLE ROW: "
                    << common::join(
                            valuesMap,
                            [](const StringMultiMap::value_type& pair)
                            { return pair.first + ":"+pair.second;}, '|');
            if (!isEmptyValues(valuesMap)) {
                tableValues.addRow(std::move(valuesMap));
            }
        }
        DEBUG() << "TABLE DONE";
        validateTableValuesSize(attrDef, tableValues, fieldName);
        tableAttributes.add(fieldName, std::move(tableValues));
    }

    return tableAttributes;
}

ObjectEditContextPtr
JsonParser::readObjectEditContext(const json::Value& jsonObject)
{
    if (!jsonObject.hasField(JSON_STR_EDIT_CONTEXT)) {
        return nullptr;
    }

    const auto& editContextField = jsonObject[JSON_STR_EDIT_CONTEXT];

    DiffAlertsData diffAlertsData;
    if (editContextField.hasField(JSON_STR_DIFF_ALERTS)) {
        for (const auto& diffAlertJson : editContextField[JSON_STR_DIFF_ALERTS]) {
            diffAlertsData.emplace_back(
                DiffAlertData {
                    diffAlertJson[STR_PRIORITY].as<uint32_t>(),
                    diffAlertJson[STR_MESSAGE].as<std::string>()
                });
        }
    }
    ObjectEditContext::SplitPoints splitPoints;
    if (editContextField.hasField(JSON_STR_SPLIT_POINTS)) {
        splitPoints = createGeoPointCollectionFromJson(
            editContextField[JSON_STR_SPLIT_POINTS]);
    }
    ObjectEditContext::SplitLines splitLines;
    if (editContextField.hasField(JSON_STR_SPLIT_LINES)) {
        splitLines = createGeomCollectionFromJson(
            editContextField[JSON_STR_SPLIT_LINES]);

        for (const auto& geom : splitLines) {
            WIKI_REQUIRE(geom->getGeometryTypeId() == geos::geom::GEOS_LINESTRING,
                         ERR_BAD_REQUEST,
                         "Split geometry type is not a line string");
        }
    }
    if (!editContextField.hasField(STR_CENTER)) {
        THROW_WIKI_LOGIC_ERROR(ERR_BAD_DATA, "View center not passed within edit context.");
    }
    if (!editContextField.hasField(STR_ZOOM)) {
        THROW_WIKI_LOGIC_ERROR(ERR_BAD_DATA, "View zoom not passed within edit context.");
    }

    bool allowIntersections = ObjectEditContext::defaultAllowIntersections();
    if (editContextField.hasField(STR_ALLOW_INTERSECTIONS)) {
        allowIntersections = editContextField[STR_ALLOW_INTERSECTIONS].as<bool>();
    }
    bool allowInvalidContours = ObjectEditContext::defaultAllowInvalidContours();
    if (editContextField.hasField(STR_ALLOW_INVALID_CONTOURS)) {
        allowInvalidContours = editContextField[STR_ALLOW_INVALID_CONTOURS].as<bool>();
    }

    return std::make_shared<ObjectEditContext>(
        View(
            createGeoPointFromJson(editContextField[STR_CENTER]),
            editContextField[STR_ZOOM].as<TZoom>()),
        allowIntersections,
        allowInvalidContours,
        std::move(splitPoints),
        std::move(splitLines),
        std::move(diffAlertsData));
}

} // namespace wiki
} // namespace maps
