#include "json_writer.h"
#include "common.h"
#include "formatter_context.h"
#include <maps/wikimap/mapspro/services/editor/src/objects/object.h>
#include <maps/wikimap/mapspro/services/editor/src/collection.h>
#include <maps/wikimap/mapspro/services/editor/src/common.h>
#include <maps/wikimap/mapspro/services/editor/src/objectvisitor.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/areal_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/point_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/complex_object.h>
#include <maps/wikimap/mapspro/services/editor/src/attributes.h>
#include <maps/wikimap/mapspro/services/editor/src/relation_infos.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/relation_object.h>
#include <maps/wikimap/mapspro/services/editor/src/edit_options.h>
#include <maps/wikimap/mapspro/services/editor/src/commit.h>

#include <maps/wikimap/mapspro/services/editor/src/actions/tile/gettile.h>
#include <maps/wikimap/mapspro/services/editor/src/actions/tile/hotspot.h>
#include <maps/wikimap/mapspro/services/editor/src/views/view_object.h>
#include <maps/wikimap/mapspro/services/editor/src/context.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/relations_manager.h>
#include <maps/wikimap/mapspro/services/editor/src/objects_cache.h>

#include <yandex/maps/wiki/configs/editor/slave_role.h>
#include <yandex/maps/wiki/configs/editor/attrdef.h>
#include <yandex/maps/wiki/common/format_type.h>
#include <yandex/maps/wiki/revision/commit.h>
#include <geos/geom/Polygon.h>

namespace maps::wiki {

namespace {

const std::string STR_MASTER_ID = "masterId";
const std::string STR_MASTER_ID_FIELD = "contour_id";
const std::string STR_RECENT_COMMIT = "recentCommit";
const std::string STR_FIRST_COMMIT = "firstCommit";
const std::string STR_JSON_NULL = "null";

void
putParent(json::ObjectBuilder& objectBuilder, const GeoObject* object, std::set<TOid>& seenIds) {
    auto parent = object->parent();
    ASSERT(parent);
    if (seenIds.contains(parent->id())) {
        WARN() << "putParent loop at: " << parent->id();
        return;
    }
    seenIds.insert(parent->id());
    objectBuilder[STR_PARENT] << [&](json::ObjectBuilder parentBuilder) {
        parentBuilder[STR_ID] = common::idToJson(parent->id());
        parentBuilder[STR_REVISION_ID] = boost::lexical_cast<std::string>(parent->revision());
        parentBuilder[STR_TITLE] = title(parent);
        parentBuilder[STR_CATEGORY_ID] = parent->category().id();
        if (parent->parent()) {
            putParent(parentBuilder, parent, seenIds);
        }
    };
}

void
putParentsChain(json::ObjectBuilder& objectBuilder, const GeoObject* object) {
    std::set<TOid> seenIds;
    seenIds.insert(object->id());
    putParent(objectBuilder, object, seenIds);
}

void
putAttribute(json::ObjectBuilder& ob, const configs::editor::AttributeDef& attrDef, const std::string& value)
{
    if (!attrDef.multiValue()) {
        ob[attrDef.id()] = json::Verbatim(escapeAttrValue(attrDef.valueType(), value));
    } else {
        ob[attrDef.id()] << [&](json::ArrayBuilder arrayBuilder) {
            const auto values = AttributeDef::unpackMultiValue(value);
            for (const auto& attrValue : values) {
                if (attrValue.empty()) {
                    continue;
                }
                arrayBuilder << json::Verbatim(escapeAttrValue(attrDef.valueType(), attrValue));
            }
        };
    }
}

void
putAttribute(json::ObjectBuilder& ob, const Attribute* attribute)
{
    const auto& attrDef = attribute->def();
    //TODO: Need to take attribute type into account
    if (!attrDef.multiValue()) {
        ob[attrDef.id()] = json::Verbatim(escapeAttrValue(attrDef.valueType(), attribute->value()));
    } else {
        ob[attrDef.id()] << [&](json::ArrayBuilder arrayBuilder) {
            for (const auto& attrValue : attribute->values()) {
                if (attrValue.empty()) {
                    continue;
                }
                arrayBuilder << json::Verbatim(escapeAttrValue(attrDef.valueType(), attrValue));
            }
        };
    }
}

void
putSlavesInfo(FormatterContext& formatterContext, json::ObjectBuilder& ob, const GeoObject* object,
    const std::set<std::string>& onlyCategories = {})
{
    const RelationInfosRange& slavesRange = object->slaveRelations().range();
    if (slavesRange.empty()) {
        return;
    }
    std::map<std::string, std::map<TOid, const RelationInfo*>> orderedSlaveInfos;
    for(const auto& link : slavesRange) {
        if (link.categoryId() == CATEGORY_AD_SUBST_FC ||
            (!onlyCategories.empty() && !onlyCategories.count(link.categoryId())) ||
            cfg()->editor()->categories()[link.categoryId()].system()) {
            continue;
        }
        const std::string& roleId = link.roleId();
        orderedSlaveInfos[roleId].insert(std::make_pair(link.id(), &link));
    }
    ob[STR_SLAVES] << [&](json::ObjectBuilder slavesBuilder) {
        for (const auto& roleIdSlaves : orderedSlaveInfos) {
            const auto& roleId = roleIdSlaves.first;
            const auto& idToRelation = roleIdSlaves.second;
            slavesBuilder[roleId] << [&](json::ObjectBuilder geoOb) {
                geoOb[STR_GEO_OBJECTS] << [&](json::ArrayBuilder arrayBuilder) {
                    for(const auto& [_, relation] : idToRelation) {
                        const GeoObject* slave = relation->relative();
                        arrayBuilder << [&](json::ObjectBuilder slaveInfoBuilder) {
                            formatterContext.visitedObjectsStack().push(slave);
                            putObjectInfo(slaveInfoBuilder, slave);
                            putAttributes(slaveInfoBuilder, slave->attributes(), &slave->tableAttributes());
                            formatterContext.visitedObjectsStack().pop();
                        };
                    }
                };
            };
        }
    };
}

void putSlave(FormatterContext& formatterContext, json::ArrayBuilder& ar,
    const GeoObject* obj, ObjectsCache& cache)
{
    if (formatterContext.needsExpansion(obj)) {
        putObject(formatterContext, ar, obj,
            formatterContext.shouldPutMastersWhenSlave(obj) ||
            formatterContext.needExtendedRelativesOutput()
                ? ExpandMasters::Yes
                : ExpandMasters::No,
            cache);
    } else {
        formatterContext.visitedObjectsStack().push(obj);
        ar << [&](json::ObjectBuilder infoBuilder) {
            putObjectInfo(infoBuilder, obj);
            if (formatterContext.needExtendedRelativesOutput()) {
                putAttributes(infoBuilder, obj->attributes(), &obj->tableAttributes());
            }
            if (formatterContext.shouldPutMastersWhenSlave(obj)) {
                putMasters(formatterContext, infoBuilder, obj);
            }
        };
        formatterContext.visitedObjectsStack().pop();
    }
}

void
putSequentialSlaves(FormatterContext& formatterContext, json::ArrayBuilder& ar,
    const RelationInfos::Range& range, ObjectsCache& cache)
{
    std::multimap<size_t, const RelationInfo*> indexBySeqNum;
    for (const auto& rInfo : range) {
        indexBySeqNum.insert(std::make_pair(rInfo.seqNum(), &rInfo));
    }
    for (const auto& slave : indexBySeqNum) {
        putSlave(formatterContext, ar, slave.second->relative(), cache);
    }
}

void
prefetchSlaveSlaves(
    FormatterContext& formatterContext,
    const RelationInfosRange& slaves,
    ObjectsCache& cache)
{
    TOIds slavesToPrefetchIds;
    for (const auto& slave : slaves) {
        if (formatterContext.needsExpansion(slave.relative())) {
            slavesToPrefetchIds.insert(slave.id());
        }
    }
    cache.relationsManager().loadRelations(
        RelationType::Slave, slavesToPrefetchIds);
}

bool
emptySlavesOutput(FormatterContext& formatterContext, const RelationInfos* slaves)
{
    const auto& slavesRange = slaves->range(formatterContext.slavesPerRoleLimitMax());
    return !slavesRange ||
        (slavesRange->empty() &&
        slavesRange->rolesWithExceededLimit().empty());
}

void
putSlaves(FormatterContext& formatterContext, json::ObjectBuilder& ob,
    const RelationInfos* slaves, const SlaveRoles& roles,
    boost::optional<const StringSet&> putAllRoles, ObjectsCache& cache,
    const std::string& noLoadSlavesRoleId = std::string{})
{
    if (emptySlavesOutput(formatterContext, slaves)) {
        return;
    }
    ob[STR_SLAVES] = [&](json::ObjectBuilder rolesBuilder) {
        for (const auto& role : roles) {
            const std::string& roleId = role.roleId();
            if (roleId == noLoadSlavesRoleId) {
                rolesBuilder[roleId] << [&](json::ObjectBuilder roleBuilder) {
                    roleBuilder["ommited"] = true;
                };
                continue;
            }
            const Category& slaveCategory = cfg()->editor()->categories()[role.categoryId()];
            if (slaveCategory.system() || role.tableRow()) {
                continue;
            }
            bool putAll = putAllRoles && putAllRoles->count(roleId);
            auto slavesByRole = putAll
                ? slaves->range(roleId)
                : *slaves->range(roleId, formatterContext.slavesPerRoleLimit(roleId));
            if (!putAll && !slavesByRole.rolesWithExceededLimit().empty()) {
                rolesBuilder[roleId] << [&](json::ObjectBuilder roleBuilder) {
                    roleBuilder["limit"] = (int64_t)formatterContext.slavesPerRoleLimit(roleId).value;
                    roleBuilder["limitExceeded"] = json::Verbatim(STR_TRUE);
                };
            } else if (!slavesByRole.empty()) {
                prefetchSlaveSlaves(formatterContext, slavesByRole, cache);
                rolesBuilder[roleId] << [&](json::ObjectBuilder roleBuilder) {
                    roleBuilder[STR_GEO_OBJECTS] << [&](json::ArrayBuilder ar){
                        if (role.sequential()) {
                            putSequentialSlaves(formatterContext, ar, slavesByRole, cache);
                        } else {
                            std::map<TOid, const GeoObject*> idToSlave;
                            for (const auto& slaveInfo : slavesByRole) {
                                idToSlave.insert(std::make_pair(slaveInfo.id(), slaveInfo.relative()));
                            }
                            for (const auto& [_, relative] : idToSlave) {
                                putSlave(formatterContext, ar, relative, cache);
                            }
                        }
                    };
                };
            }

        } //roles
    };
}

const std::string&
boolToStr(bool val)
{
    return val ? STR_TRUE : STR_FALSE;
}

} // namespace

std::string
escapeAttrValue(ValueType valueType, const std::string& value)
{
    switch(valueType) {
        case ValueType::Boolean :
            return boolToStr(!value.empty());
        case ValueType::Integer :
            if (!value.empty()) {
                return std::to_string(boost::lexical_cast<long long int>(value));
            }
        case ValueType::Float :
            if (!value.empty()) {
                return std::to_string(boost::lexical_cast<double>(value));
            }
        case ValueType::Bitmask :
            if (!value.empty()) {
                return value;
            }
        case ValueType::Enum :
        case ValueType::Login :
        case ValueType::String : {
            return "\"" + common::escapeForJSON(value) + "\"";
        }
        case ValueType::Json :
            return
                value.empty()
                ? STR_JSON_NULL
                : value;
            return value;
        case ValueType::Table:
            REQUIRE(ValueType::Table != valueType,
                "Table values shouldn't be output directly!");
    }
    return s_emptyString;
}

void
putTableAttributeContents(json::ArrayBuilder& arrayBuilder, const TableValues& contents)
{
    if (contents.empty()) {
        return;
    }
    for (size_t row = 0; row < contents.numRows(); ++row) {
        arrayBuilder << [&contents, &row](json::ObjectBuilder rowBuilder) {
            const StringMultiMap& rowValues = contents.values(row);
            StringSet columns = readKeys(rowValues);
            for (const auto& column : columns) {
                const auto& columnAttrDef =  *cfg()->editor()->attribute(column);
                auto valueIt = rowValues.find(column);
                rowBuilder[column] <<
                    json::Verbatim(escapeAttrValue(columnAttrDef.valueType(), valueIt->second));
            }
        };
    }
}

void
putAttributes(json::ObjectBuilder& ob,
    const Attributes& attributes,
    const TableAttributesValues* tableAttributes)
{
    if (attributes.empty()) {
        return;
    }
    ob[STR_ATTRS] << [&](json::ObjectBuilder attrsBuilder) {
        for (const auto& attrDef : attributes.definitions()) {
            if (attrDef->system()) {
                continue;
            }
            if (attrDef->table() && tableAttributes) {
                const auto& contents = tableAttributes->find(attrDef->id());
                attrsBuilder[attrDef->id()] << [&contents](json::ArrayBuilder arrayBuilder){
                    putTableAttributeContents(arrayBuilder, contents);
                };
            } else {
                auto aIt = attributes.find(attrDef->id());
                if (aIt != attributes.end()) {
                    putAttribute(attrsBuilder, &(*aIt));
                }
            }
        }
    };
}

void putPlainAttributes(
    json::ObjectBuilder& ob,
    const std::string& categoryId,
    const StringMap& attributes)
{
    if (attributes.empty()) {
        return;
    }
    const auto& category = cfg()->editor()->categories()[categoryId];
    ob[STR_ATTRS] << [&](json::ObjectBuilder attrsBuilder) {
        for (const auto& attrDef : category.attrDefs()) {
            if (attrDef->system()) {
                continue;
            }
            if (attrDef->table()) {
                continue;
            } else {
                auto it = attributes.find(attrDef->id());
                if (it != attributes.end()) {
                    putAttribute(attrsBuilder, *attrDef, it->second);
                }
            }
        }
    };
}

void
putCollection(
    FormatterContext& formatterContext,
    json::ObjectBuilder& objectBuilder,
    const GeoObjectCollection& collection,
    ObjectsCache& cache,
    AppendApproveStatus appendApproveStatus)
{
    objectBuilder[STR_GEO_OBJECTS] = [&](json::ArrayBuilder arrayBuilder) {
        for (const auto& object : collection) {
            putObject(formatterContext, arrayBuilder, object.get(), ExpandMasters::Yes, cache, appendApproveStatus);
        }
    };
}

void
putObject(
    FormatterContext& formatterContext,
    json::ArrayBuilder& arrayBuilder,
    const GeoObject* object,
    ExpandMasters expandMasters,
    ObjectsCache& cache,
    AppendApproveStatus appendApproveStatus)
{
    formatterContext.markExpanded(object->id());
    arrayBuilder << [&object, &formatterContext, &expandMasters, &cache, &appendApproveStatus]
            (json::ObjectBuilder geoObjectBuilder) {
        fillObject(formatterContext, geoObjectBuilder, object, expandMasters, cache,
            s_emptyString, appendApproveStatus);
    };
}

void
fillObject(
    FormatterContext& formatterContext,
    json::ObjectBuilder& geoObjectBuilder,
    const GeoObject* object,
    ExpandMasters expandMasters,
    ObjectsCache& cache,
    const std::string& noLoadSlavesRoleId,
    AppendApproveStatus appendApproveStatus)
{
    formatterContext.visitedObjectsStack().push(object);
    putObjectInfo(geoObjectBuilder, object);
    geoObjectBuilder[STR_RICH_CONTENT] = object->hasRichContent();
    if (formatterContext.visitedDepth() == 1) {
        auto recentCommit = recentAffectingCommit(
            cache, formatterContext.actualObjectRevisionId(object->revision()));
        auto firstCommit = wiki::firstCommit(cache, object->id());
        const std::list<revision::Commit> commits{firstCommit, recentCommit};
        auto branchContext = cache.branchContext();
        BatchCommitPreparedFields preparedFields;
        preparedFields.prepareStates(branchContext, commits);
        if (appendApproveStatus == AppendApproveStatus::Yes) {
            preparedFields.prepareApproveStatuses(branchContext, commits);
        }

        CommitModel recentCommitModel(std::move(recentCommit));
        recentCommitModel.setCustomContextObjectId(object->id());
        preparedFields.fillCommitModel(recentCommitModel);
        geoObjectBuilder[STR_RECENT_COMMIT] << [&](json::ObjectBuilder commitBuilder) {
            putCommitModel(commitBuilder, recentCommitModel);
        };

        CommitModel firstCommitModel(std::move(firstCommit));
        firstCommitModel.setCustomContextObjectId(object->id());
        preparedFields.fillCommitModel(firstCommitModel);
        geoObjectBuilder[STR_FIRST_COMMIT] << [&](json::ObjectBuilder commitBuilder) {
            putCommitModel(commitBuilder, firstCommitModel);
        };
    }
    geoObjectBuilder[STR_STATE] = boost::lexical_cast<std::string>(object->state());
    geos::geom::Envelope envelope = object->envelope();
    // bounds
    if (!envelope.isNull()) {
        geoObjectBuilder[STR_BOUNDS] = json::Verbatim(json(envelope));
    }
    // parent
    if (object->parent()) {
        putParentsChain(geoObjectBuilder, object);
    }
    // attrs
    if (!object->attributes().areAllSystem()) {
        putAttributes(geoObjectBuilder, object->attributes(), &object->tableAttributes());
    }
    // permissions
    putObjectPermissions(formatterContext, geoObjectBuilder, object);
    // slaves
    const Category& category = object->category();
    RelationInfos slaveInfos = object->relations(RelationType::Slave);
    putSlaves(formatterContext, geoObjectBuilder, &slaveInfos, category.slavesRoles(),
        formatterContext.putAllSlavesRoles(object), cache, noLoadSlavesRoleId);
    // masters
    if (expandMasters == ExpandMasters::Yes) {
        putMasters(formatterContext, geoObjectBuilder, object);
    }
    formatterContext.visitedObjectsStack().pop();
}

void
putTileObject(json::ArrayBuilder& arrayBuilder, const TileObject* tileObject, const Geom& hotspotGeom)
{
    arrayBuilder << [&tileObject, &hotspotGeom](json::ObjectBuilder featureBuilder) {
        featureBuilder["type"] = "Feature";
        featureBuilder["properties"] << [&](json::ObjectBuilder propBuilder) {
            if (!tileObject->label().empty()) {
                propBuilder["hintContent"] = tileObject->label();
            } else {
                const Category& cat =
                    cfg()->editor()->categories()[tileObject->categoryId()];
                propBuilder["hintContent"] = cat.label();
            }
            propBuilder["HotspotMetaData"] << [&](json::ObjectBuilder metaData) {
                metaData[STR_ID] = common::idToJson(tileObject->id());
                metaData["RenderedGeometry"] =
                    json::Verbatim(prepareGeomJson(hotspotGeom, 0, SpatialRefSystem::Mercator));
            };
            propBuilder[STR_GEO_OBJECT] << [&tileObject](json::ObjectBuilder tileObjectBuilder) {
                tileObjectBuilder[STR_ID] = common::idToJson(tileObject->id());
                tileObjectBuilder[STR_REVISION_ID] = boost::lexical_cast<std::string>(tileObject->revision());
                tileObjectBuilder[STR_CATEGORY_ID] = tileObject->categoryId();
                tileObjectBuilder[STR_GEOMETRY] =
                    json::Verbatim(
                        prepareGeomJson(
                            tileObject->geom(),
                            tileObject->simpificationPolicy() == TileObject::SimpificationPolicy::KeepPrecise
                                ? GEODETIC_GEOM_PRECISION
                                : TILE_OBJECT_GEOM_PRECISION));
                const auto& customFieldsData = tileObject->customFieldsData();
                if (customFieldsData.empty()) {
                    return;
                }
                auto it = customFieldsData.find(STR_MASTER_ID_FIELD);
                if (it != customFieldsData.end()) {
                    tileObjectBuilder[STR_MASTER_ID] = it->second;
                }
            };
        };
    };
}

void
putTileObject(json::ArrayBuilder& arrayBuilder, const TileObject* tileObject)
{
    for (const auto& hotspotGeom : tileObject->hotspot().geoms()) {
        putTileObject(arrayBuilder, tileObject, hotspotGeom);
    }
}

std::string
prepareGeomJson(
    const common::Geom& geom, size_t precision,
    SpatialRefSystem spatialRefSys /*= SpatialRefSystem::Geodetic*/)
{
    ASSERT(!geom.isNull());
    std::stringstream s;
    s.precision(precision);
    s << std::fixed;
    geom.geoJson(s, spatialRefSys);
    return s.str();
}

void putObjectPermissions(
    FormatterContext& formatterContext,
    json::ObjectBuilder& geoObjectBuilder,
    const GeoObject* object)
{
    geoObjectBuilder["permissions"] = [&object, &formatterContext](json::ObjectBuilder permissionsBuilder) {
        permissionsBuilder["deletable"] = canDelete(object, formatterContext.userId());
        permissionsBuilder["editable"] = canEdit(object, formatterContext.userId());
    };
}

void
putMasters(FormatterContext& formatterContext, json::ObjectBuilder& ob, const GeoObject* object)
{
    const RelationInfosRange& mastersRange = object->masterRelations().range();
    if (mastersRange.empty()) {
        return;
    }
    std::multimap<std::string, const RelationInfo*> orderedMasterInfos;
    StringSet orderedRoles;
    for(const auto& link : mastersRange) {
        if (link.categoryId() == CATEGORY_AD_SUBST_FC ||
            cfg()->editor()->categories()[link.categoryId()].system()) {
            continue;
        }
        const std::string& roleId = link.roleId();
        orderedMasterInfos.insert(std::make_pair(roleId, &link));
        orderedRoles.insert(roleId);
    }
    ob[STR_MASTERS] << [&](json::ObjectBuilder mastersBuilder) {
        for (const auto& roleId : orderedRoles) {
            mastersBuilder[roleId] << [&](json::ObjectBuilder geoOb) {
                geoOb[STR_GEO_OBJECTS] << [&](json::ArrayBuilder arrayBuilder) {
                    auto masterItRange = orderedMasterInfos.equal_range(roleId);
                    for(auto masterIt = masterItRange.first;
                        masterIt != masterItRange.second;
                        ++masterIt) {
                        const GeoObject* master = masterIt->second->relative();
                        arrayBuilder << [&](json::ObjectBuilder masterInfoBuilder) {
                            formatterContext.visitedObjectsStack().push(master);
                            putObjectInfo(masterInfoBuilder, master);
                            if (formatterContext.needExtendedRelativesOutput()) {
                                putAttributes(masterInfoBuilder, master->attributes(), &master->tableAttributes());
                            }
                            if (formatterContext.continueToOutputMasters(roleId) ||
                                formatterContext.mastersOnExtendedOutputPath()) {
                                putMasters(formatterContext, masterInfoBuilder, master);
                            }
                            if (formatterContext.slavesOnExtendedOutputPath()) {
                                if (formatterContext.visitedObjectsStack().depth() == 2
                                    && formatterContext.visitedObjectsStack().currentCategoryId() == CATEGORY_INDOOR_LEVEL)
                                {
                                    //NMAPS-10308
                                    putSlavesInfo(formatterContext, masterInfoBuilder, master,
                                        {CATEGORY_IMAGE_OVERLAY, CATEGORY_IMAGE_OVERLAY_PUBLIC});
                                } else {
                                    putSlavesInfo(formatterContext, masterInfoBuilder, master);
                                }
                            }
                            formatterContext.visitedObjectsStack().pop();
                        };
                    }
                };
            };
        }
    };
}

} // namespace maps::wiki
