#include "objects_update_relations_parser.h"

#include <maps/wikimap/mapspro/services/editor/src/exception.h>
#include <yandex/maps/wiki/common/json_helpers.h>

namespace maps {
namespace wiki {

namespace {
RevisionIds readRevisionsArray(const json::Value& objects)
{
    RevisionIds revisions;
    WIKI_REQUIRE(objects.isArray(), ERR_BAD_REQUEST, "Invalid input.");
    for (const auto& object : objects) {
        revisions.insert(common::readField<TRevisionId>(object, STR_REVISION_ID));
    }
    return revisions;
}

RelativesDiffRevisions
readDiffContents(const json::Value& roleNode)
{
    const auto& diff = roleNode[STR_DIFF];
    WIKI_REQUIRE(diff.isObject(), ERR_BAD_REQUEST, "Missing diff node.");
    RelativesDiffRevisions diffContents;
    if (diff.hasField(STR_ADDED)) {
        diffContents.added = readRevisionsArray(diff[STR_ADDED]);
    }
    if (diff.hasField(STR_REMOVED)) {
        diffContents.removed = readRevisionsArray(diff[STR_REMOVED]);
    }
    return diffContents;
}

void
readDiff(
    RelativesDiffRevisionsByRole& diff,
    RelativesToSetRevisionsByRole& set, 
    const json::Value& relativeTypeNode)
{
    const auto& roles = relativeTypeNode.fields();
    for (const auto& role : roles) {
        auto& roleNode = relativeTypeNode[role];
        if (roleNode.hasField(STR_SET)) {
            set.emplace(role, readRevisionsArray(roleNode[STR_SET]));
        } else {
            diff.emplace(role, readDiffContents(roleNode));
        }
    }
}

}//namespace

void
ObjectsUpdateRelationsParser::parse(const std::string& requestBody)
{
    const auto& jsonRoot = json::Value::fromString(requestBody);
    WIKI_REQUIRE(jsonRoot.isObject(), ERR_BAD_REQUEST, "Invalid input." << requestBody);
    objectsRevisionsToUpdate_ = readRevisionsArray(jsonRoot[STR_GEO_OBJECTS]);
    WIKI_REQUIRE(!objectsRevisionsToUpdate_.empty(), ERR_BAD_REQUEST,
        "Empty objects update request.");
    const auto& context = jsonRoot[STR_CONTEXT];
    WIKI_REQUIRE(context.isObject(), ERR_BAD_REQUEST, "Invalid input." << requestBody);
    if (context.hasField(STR_MASTERS)) {
        readDiff(mastersDiff_, mastersToSet_, context[STR_MASTERS]);
    }
    if (context.hasField(STR_SLAVES)) {
        readDiff(slavesDiff_, slavesToSet_, context[STR_SLAVES]);
    }
}

} // namespace wiki
} // namespace maps
