#include "xml_branches.h"
#include "common.h"
#include <yandex/maps/wiki/common/date_time.h>
#include <sstream>

namespace maps {
namespace wiki {

namespace {

void
serializeBranch(std::ostream& os, const revision::Branch& branch, bool first = false)
{
    const auto type = branch.type();
    const bool isProduction = type == revision::BranchType::Archive && first;

    os << "<branch id=\"" << branch.id() << "\" type=\"";
    if (isProduction) {
        os << BRANCH_TYPE_PRODUCTION;
    } else {
        os << type;
    }
    os << "\" state=\"" << branch.state() << "\"";
    if (type == revision::BranchType::Trunk) {
        os << "/>";
        return;
    }

    const std::string createdTagName = type == revision::BranchType::Approved
        ? STR_APPROVED
        : STR_CREATED;

    os << ">" << "<" << createdTagName << ">"
       << common::canonicalDateTimeString(
            branch.createdAt(), common::WithTimeZone::Yes) << "</" << createdTagName << ">"
       << "<" << createdTagName << "-by>"
       << branch.createdBy() << "</" << createdTagName << "-by>";

    if (branch.finishedBy()) {
       os << "<published>" << common::canonicalDateTimeString(
            branch.finishedAt(), common::WithTimeZone::Yes) << "</published>"
          << "<published-by>" << branch.finishedBy() << "</published-by>";
    }
    os << "</branch>";
}

} // namespace

std::string
branchIdStateToXml(TBranchId branchId, revision::BranchState state)
{
    std::ostringstream os;
    os << "<branch id=\"" << branchId << "\""
          " state=\"" << state << "\""
          "/>";
    return os.str();
}

std::string
branchToXml(const revision::Branch& branch)
{
    std::ostringstream ss;
    serializeBranch(ss, branch);
    return ss.str();
}

std::string
branchesToXml(const Branches& branches)
{
    std::set<revision::BranchType> branchTypeVisited;

    std::ostringstream ss;
    ss << "<branches>";
    for (const auto& branch : branches) {
        bool first = branchTypeVisited.insert(branch.type()).second;
        serializeBranch(ss, branch, first);
    }
    ss << "</branches>";
    return ss.str();
}

} // namespace wiki
} // namespace maps
