#include "xml_formatter.h"

#include "formatter_context.h"
#include <maps/wikimap/mapspro/services/editor/src/commit.h>
#include "xml_writer.h"
#include "json_writer.h"
#include "xml_parser.h"
#include <maps/wikimap/mapspro/services/editor/src/actions/filters/serialization.h>
#include "xml_branches.h"
#include "common.h"
#include "tile.h"
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/common/date_time.h>

#include <maps/libs/xml/include/encode.h>

namespace maps {
namespace wiki {

std::string
XMLFormatter::formatException(const maps::Exception& ex, bool withBacktrace)
{
    std::ostringstream bodyStream;
    exceptionToStream(bodyStream, ex, withBacktrace);

    std::ostringstream os;
    os << "<error";
    for (const auto& attr : ex.attrs()) {
        os << ' ' << attr.first
            << "=\"" << xml3::encodeSpecialChars(attr.second) << '"';
    }
    os << '>';
    auto logicExceptionPtr = dynamic_cast<const maps::wiki::LogicException*>(&ex);
    if (logicExceptionPtr && logicExceptionPtr->location()) {
        os << "<location>";
        os << prepareGeomJson(
            pointToGeom(*logicExceptionPtr->location()),
            GEODETIC_GEOM_PRECISION,
            SpatialRefSystem::Geodetic);
        os << "</location>";
    }
    os << "<message>";
    os << common::escapeCData(bodyStream.str());
    os << "</message>";
    os << "</error>";
    return os.str();
}

std::string
XMLFormatter::operator ()(const ResultType<SaveObject>& result)
{
    context().setUserId(result.userId);
    XmlOutputStream output;
    XmlWriter serialization(context());
    serialization.putHeader(output);
    serialization.putToken(output, result.taskName, result.token);
    serialization.put(output, &result.collection, *result.cache);
    serialization.putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<CloneObject>& result)
{
    XmlOutputStream output;
    XmlWriter serialization(context());
    serialization.putHeader(output);
    serialization.putToken(output, result.taskName, result.token);
    serialization.put(output, result.clone.get(), *result.cache);
    serialization.putFooter(output);
    return output.str();
}

namespace
{
std::string
simpleUpdateTaskResponse(const std::string& taskName, const Token& token)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    XmlWriter::putToken(output, taskName, token);
    XmlWriter::putFooter(output);
    return output.str();
}
}

std::string
XMLFormatter::operator ()(const ResultType<ObjectsUpdateAttributes>& result)
{
    return simpleUpdateTaskResponse(result.taskName, result.token);
}

namespace {
std::string formatPedestrianPoints(
    const std::string& tagName,
    const std::vector<Geom>& points)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<" << tagName << ">";
    output << "<points>";
    for (auto& point : points) {
        XmlWriter::putGeometry(output, point, SpatialRefSystem::Geodetic);
    }
    output << "</points>";
    output << "</" << tagName << ">";
    XmlWriter::putFooter(output);
    return output.str();
}
} // unnamed namespace

std::string
XMLFormatter::operator ()(const ResultType<ToolsPedestrianPoints>& result)
{
    return formatPedestrianPoints("response-tools-pedestrian_points", result.points);
}

std::string
XMLFormatter::operator ()(const ResultType<ToolsLoadPedestrianPoints>& result)
{
    return formatPedestrianPoints("response-tools-load_pedestrian_points", result.points);
}

std::string
XMLFormatter::operator ()(const ResultType<ToolsSavePedestrianPoints>& result)
{
    return simpleUpdateTaskResponse("ToolsSavePedestrianPoints", result.token);
}

std::string
XMLFormatter::operator ()(const ResultType<ObjectsUpdateState>& result)
{
    return simpleUpdateTaskResponse(result.taskName, result.token);
}
std::string
XMLFormatter::operator ()(const ResultType<ObjectsUpdateMove>& result)
{
    return simpleUpdateTaskResponse(result.taskName, result.token);
}

std::string
XMLFormatter::operator ()(const ResultType<ObjectsUpdateSyncGeometry>& result)
{
    return simpleUpdateTaskResponse(result.taskName, result.token);
}

std::string
XMLFormatter::operator ()(const ResultType<ObjectsUpdateRelation>& result)
{
    return simpleUpdateTaskResponse(result.taskName, result.token);
}
std::string
XMLFormatter::operator ()(const ResultType<ObjectsUpdateUnion>& result)
{
    return simpleUpdateTaskResponse(result.taskName, result.token);
}

std::string
XMLFormatter::operator ()(const ResultType<ObjectsUpdateMerge>& result)
{
    return simpleUpdateTaskResponse(result.taskName, result.token);
}

std::string
XMLFormatter::operator ()(const ResultType<ObjectsUpdateSnap>& result)
{
    return simpleUpdateTaskResponse(result.taskName, result.token);
}

std::string
XMLFormatter::operator ()(const ResultType<ObjectsCleanupRelations>& result)
{
    return simpleUpdateTaskResponse(result.taskName, result.token);
}

std::string
XMLFormatter::operator ()(const ResultType<GetObject>& result)
{
    context().setActualObjectRevisionId(result.actualObjectRevisionId);
    context().setUserId(result.userId);

    XmlOutputStream output;
    XmlWriter serialization(context());
    serialization.putHeader(output);
    output << "<response-object>";
    serialization.put(output, &result.collection, *result.cache);
    output << "</response-object>";
    serialization.putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<GetHistory>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-history><object id=\"" << result.oid << "\"><commits ";
    output << " total-count=\"" << result.totalCount << "\"";
    output << " page=\"" << result.page << "\"";
    output << " per-page=\"" << result.perPage << "\"";
    output << ">";
    for (const auto& commitModel : result.commitModels) {
        XmlWriter::putCommitModel(output, commitModel);
    }
    output << "</commits></object></response-history>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<GetObjectsByTile>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);

    auto orderedObjects = sortTileObjects(result.objects);

    output << "<objects>";
    for (const auto& object : orderedObjects) {
        for (auto sameZOrderObj : object.second) {
            XmlWriter::put(output, sameZOrderObj);
        }
    }
    output << "</objects>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<ObjectsQueryLasso>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-objects-query-lasso>";
    output << "<objects>";
    for (const auto& object : result.viewObjects) {
        XmlWriter::put(output, &object);
    }
    output << "</objects>";
    output << "</response-objects-query-lasso>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<ObjectsQueryIds>& result)

{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-objects-query-ids>";
    output << "<objects>";
    for (const auto& object : result.collection) {
        output << "<object id=\"" << object->id() << "\">";
        XmlWriter::putRevision(output, object.get());
        XmlWriter::putCategory(output, object.get());
        XmlWriter::putScreenLabel(output, object.get());
        output << "<state>"
               << object->state()
               << "</state>";
        output << "</object>";
    }
    output << "</objects>";
    output << "</response-objects-query-ids>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<ObjectsQueryCategory>& result)
{
    XmlOutputStream output;
    XmlWriter serialization(context());
    serialization.putHeader(output);
    output << "<response-objects-query-category>";
    output << "<objects>";
    for (const auto& object : result.collection) {
        output << "<object id=\"" << object->id() << "\">";
        XmlWriter::putRevision(output, object.get());
        XmlWriter::putCategory(output, object.get());
        XmlWriter::putScreenLabel(output, object.get());
        if (!object->attributes().areAllSystem()) {
            serialization.put(output, &object->attributes(), &object->tableAttributes());
        }
        XmlWriter::putGeometry(output, object->geom(), SpatialRefSystem::Geodetic);
        output << "</object>";
    }
    output << "</objects>";
    output << "</response-objects-query-category>";
    serialization.putFooter(output);
    return output.str();
}


std::string
XMLFormatter::operator ()(const ResultType<ObjectsQueryPath>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-objects-query-path>";
    output << "<objects>";
    for (const auto& object : result.elements) {
        XmlWriter::put(output, &object);
    }
    output << "</objects>";
    output << "</response-objects-query-path>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<ObjectsQueryFilter>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-objects-query-filter>";
    output << "<statistics"
             << " total-count=\"" << result.totalCount << "\">";
    for (const auto& pair : result.countsByCategory) {
        output << "<category id=\"" << pair.first
                 << "\" count=\"" << pair.second << "\"/>";
    }
    output << "</statistics>";

    if (result.totalCount <= result.limit) {
        output << "<objects>";
        for (const auto& object : result.viewObjects) {
            XmlWriter::put(output, &object);
        }
        output << "</objects>";
    }
    output << "</response-objects-query-filter>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<GetAsyncResult>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << (*this)(*result.asyncTaskResult);
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const AsyncTaskResult& asyncResult)
{
    XmlOutputStream output;
    output
            << "<task id=\"" << asyncResult.token().str()
            << "\" status=\"" << toString(asyncResult.status()) << "\">";
    if (asyncResult.status() == taskutils::TaskStatus::Done) {
        output << "<result>" << asyncResult.result() << "</result>";
    }
    else if (asyncResult.status() == taskutils::TaskStatus::Failed) {
        if (asyncResult.error().empty()) {
            output << "<error status=\"INTERNAL_ERROR\">unknown error</error>";
        }
        else {
            output << asyncResult.error();
        }
    }
    output << "</task>";
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<GetSlaveInfos>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-slaves>";
    if (!result.slaveInfos.empty()) {
        output << "<slaves";
        if (result.bbox.empty()) {
            output << " limit=\"" << result.limit << "\"";
            output << " offset=\"" << result.offset << "\"";
        }
        output << ">";
        output << "<role id=\"" << result.roleId << "\">";
        for (const auto& slaveInfo : result.slaveInfos) {
            XmlWriter::putRelative(output, &slaveInfo);
        }
        output << "</role>";
        output << "</slaves>";
    }
    output << "</response-slaves>";
    XmlWriter::putFooter(output);
    return output.str();
}

namespace
{
std::string
stringsToXML(const StringVec& values, const std::string& tag)
{
    return "<" + tag + " id=\"" +
        common::join(values, "\"/> <" + tag + " id=\"") +
        "\"/>";
}

void
putRendererLayer(std::ostream& out,
    const RendererLayersTraverse::Layer& layer,
    const RendererLayersTraverse::ProjectParams& requestParams)
{
    out << "<layer id=\"" << layer.id() << "\" type=\"" << layer.type() << "\">";
    if (layer.label()) {
        out << "<name>" << layer.name() << "</name>";
        out << "<label>" << *layer.label() << "</label>";
    }
    if (!layer.categoryIds().empty()) {
        out << "<categories>" << stringsToXML(layer.categoryIds(), "category") << "</categories>";
    }
    if (layer.textLayer() && layer.textLayer()->projectParams.match(requestParams)) {
        out << "<text id=\"" << layer.textLayer()->id << "\"/>";
    }
    std::list<const RendererLayersTraverse::Representation*> matchingRepresentations;
    for (const auto& repr : layer.representations()) {
        if (repr.projectParams.match(requestParams)) {
            matchingRepresentations.push_back(&repr);
        }
    }
    if (!matchingRepresentations.empty()) {
        out << "<representations>";
        for (const auto* repr : matchingRepresentations) {
            out << "<representation id=\"" << repr->id << "\">"
            << "<name>" << repr->name << "</name>"
            << "<label>" << repr->label << "</label>"
            << "</representation>";
        }
        out << "</representations>";
    }
    out << "</layer>";
}
}
std::string
XMLFormatter::operator ()(const ResultType<GetLayers>& result)
{
    XmlOutputStream output;
    output
    << "<editor xmlns=\"" + PROJECT_NAMESPACE + "\">"
    << "<response-renderer-sublayers>"
    << "<layers>";
    const auto& layersTraverse = result.layersTraverse;
    for (const auto& group : layersTraverse.layersByGroup()) {
        std::vector<const RendererLayersTraverse::Layer*> matchingLayers;
        for (const auto& layer : group.second) {
            if (layer->projectParams().match(result.requestProjectParams)) {
                matchingLayers.push_back(layer);
            }
        }
        if (matchingLayers.empty()) {
            continue;
        }
        const auto& id = group.first;
        output << "<layer-group id=\"" << id << "\">";
        output << "<label>" << layersGroupLabel(id) << "</label>";
        for (const auto& layer : matchingLayers) {
            if (layer->projectParams().match(result.requestProjectParams)) {
                putRendererLayer(output, *layer, result.requestProjectParams);
            }
        }
        output << "</layer-group>";
    }
    output << "</layers>"
    << "</response-renderer-sublayers>"
    << "</editor>";
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<GetFilters>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-filters>";
    output << "<filters"
             << " total-count=\"" << result.totalCount << "\""
             << " page=\"" << result.page << "\""
             << " per-page=\"" << result.perPage << "\">";

    for (const auto& filter : result.filters) {
        serializeFilter(filter, output);
    }
    output << "</filters>";
    output << "</response-filters>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<SaveFilter>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-save-filter>";
    output << "<internal><token>" << result.token << "</token></internal>";
    serializeFilter(*result.filter, output);
    output << "</response-save-filter>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<DeleteFilter>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-delete-filter>";
    output << "<internal><token>" << result.token << "</token></internal>";
    output << "</response-delete-filter>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<GetFilterExpression>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-filter-expression>";
    if (result.isSQLFormat) {
        output << common::escapeCData(result.viewSql);
    } else {
        serializeExpression(*result.expression, output);
    }
    output << "</response-filter-expression>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<SaveExpression>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-save-expression>";
    output << "<internal><token>" << result.token << "</token></internal>";
    serializeExpression(*result.expression, output);
    output << "</response-save-expression>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<GetSuggest>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output <<"<response-suggest>";
    for (const auto& suggestedObject : result.suggestedObjects) {
        output
            << "<object id=\"" << suggestedObject.id << "\">"
            << "<category-id>"<< suggestedObject.categoryId << "</category-id>"
            << "<revision id=\"" << suggestedObject.id << ":" << suggestedObject.commitId << "\"/>"
            << "<name>" << common::escapeCData(suggestedObject.name) << "</name>";

        if (!suggestedObject.path.empty()) {
            output << "<path>" << common::escapeCData(suggestedObject.path) << "</path>";
        }
        output << "</object>";
    }
    output << "</response-suggest>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<GetBranches>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output <<"<response-branches>";
    output << branchesToXml(result.branches);
    output << "</response-branches>";
    XmlWriter::putFooter(output);
    return output.str();
}

namespace {

template <typename Result>
std::string
serializeBranches(const Result& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-save-branches>"
         << "<internal><token>" << result.token << "</token></internal>"
         << branchesToXml(result.branches)
         << "</response-save-branches>";
    XmlWriter::putFooter(output);
    return output.str();
}

} // namespace

std::string
XMLFormatter::operator ()(const ResultType<SaveBranchType>& result)
{
    return serializeBranches(result);
}

std::string
XMLFormatter::operator ()(const ResultType<SaveBranchState>& result)
{
    return serializeBranches(result);
}

std::string
XMLFormatter::operator ()(const ResultType<AsyncBaseController>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    if (!result.result.empty()) {
        output << result.result;
    }
    else {
        output << "<task name=\"" << result.taskName
            << "\" id=\"" << result.tokenStr << "\" status=\"created\"/>";
    }
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<GetGeoLocks>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-geolocks>";
    output << branchIdStateToXml(result.branchId, result.branchState);
    output << "<geolocks>";
    for (const auto& lock : result.locks) {
        output << "<lock id=\"" << lock.id() << "\""
                 << " branch=\"" << lock.branchId() << "\""
                 << " created-at=\"" << common::canonicalDateTimeString(lock.createdAt(),
                    common::WithTimeZone::Yes) << "\""
                 << " created-by=\"" << lock.createdBy() << "\"/>";
    }
    output << "</geolocks>";
    output << "</response-geolocks>";
    XmlWriter::putFooter(output);
    return output.str();

}

std::string
XMLFormatter::operator ()(const ResultType<GetCommitDiff>& result)
{
    ASSERT(result.commitModel);
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-commit-diff>";
    output << "<commit-diff";
    if (result.commitModel->contextPrimaryObjectId()) {
        output << " object-id=\"" << *result.commitModel->contextPrimaryObjectId() << "\"";
    }
    output << ">";
    XmlWriter::putCommitModel(output, *result.commitModel);
    putXmlIds(output, "affected-objects-ids", affectedObjectIds(result.commitModel->commit()));
    if (result.details)
    {
        const auto& details = *result.details;
        if (!details.envelope.isNull()) {
            output << "<bbox>" << json(details.envelope) << "</bbox>";
        }
        if (result.commitModel->contextPrimaryObjectId()) {
            const auto& obj = details.anyObject();
            if (obj) {
                output << "<object id=\"" << obj->id() << "\">";
                XmlWriter::putCategory(output, obj.get());
                output << "</object>";
            }

            XmlWriter::put(output, details);
        }
    }
    output << "</commit-diff>";
    output << "</response-commit-diff>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<GetBranchDiff>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-branch-diff>";
    output << "<branch-diff"
           << " from-branch=\"" << result.fromBranchId << "\""
           << " to-branch=\"" << result.toBranchId << "\">";

    const auto& details = result.details;

    if (!details.envelope.isNull()) {
        output << "<bbox>" << json(details.envelope) << "</bbox>";
    }

    const auto& obj = details.anyObject();
    ASSERT(obj);
    output << "<object id=\"" << obj->id() << "\">";
    XmlWriter::putCategory(output, obj.get());
    output << "</object>";

    XmlWriter::put(output, details);

    output << "</branch-diff>";
    output << "</response-branch-diff>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<CreateToken>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-create-token>";
    output << "<internal><token>" << result.token << "</token></internal>";
    output << "</response-create-token>";
    XmlWriter::putFooter(output);
    return output.str();
}

std::string
XMLFormatter::operator ()(const ResultType<GetCommit>& result)
{
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-get-commit>";
    ASSERT(result.commitModelPtr);
    XmlWriter::putCommitModel(output, *result.commitModelPtr);
    output << "</response-get-commit>";
    XmlWriter::putFooter(output);
    return output.str();
}

} // namespace wiki
} // namespace maps
