#include "xml_formatter.h"

#include "xml_writer.h"
#include <maps/wikimap/mapspro/libs/gdpr/include/user.h>
#include <maps/wikimap/mapspro/services/editor/src/utils.h>
#include <maps/wikimap/mapspro/services/editor/src/commit.h>
#include <maps/wikimap/mapspro/services/editor/src/social_utils.h>
#include <yandex/maps/wiki/common/date_time.h>

namespace maps {
namespace wiki {

namespace {

const std::string SOCIAL_XML_NS = "http://maps.yandex.ru/mapspro/social/1.x";
const std::string SOCIAL_XML_NS_DECL = "xmlns=\"" + SOCIAL_XML_NS + "\"";

std::string
eventXmlData(
    const social::Event& event,
    const CommitModel& commitModel)
{
    std::ostringstream os;
    os << "<data>";

    const auto& commitData = event.commitData();
    ASSERT(commitData);
    os << "<commit xmlns=\"" << PROJECT_NAMESPACE << "\" id=\"" << commitData->commitId() << "\">";
    XmlWriter::putCommitModel(os, commitModel, /*putRootTag=*/false);
    os << "<bounds>" << commitData->bounds() << "</bounds>" "</commit>";

    if (event.primaryObjectData()) {
        const auto& primaryObj = *event.primaryObjectData();
        os  << "<object xmlns=\"" << PROJECT_NAMESPACE << "\" id=\"" << primaryObj.id() << "\">"
            << "<category-id>" << primaryObj.categoryId() << "</category-id>"
            << "<screen-label>" << common::escapeCData(primaryObj.screenLabel()) << "</screen-label>"
            << "<edit-notes>" << primaryObj.editNotes() << "</edit-notes>"
            << "</object>";
    }

    os << "</data>";
    return os.str();
}

} // namespace

std::string
XMLFormatter::operator ()(const ResultType<GetSocialFeed>& result)
{
    std::ostringstream os;
    os  << "<feed " << SOCIAL_XML_NS_DECL
        << " subscriber=\"" << result.subscriber << "\""
        << " branch=\"" << result.branchId << "\""
        << " type=\"" << result.feedType << "\"";

    if (result.page) {
        ASSERT(result.totalCount);
        os << " total-count=\"" << *result.totalCount << "\"";
        os << " page=\"" << result.page << "\"";
    } else {
        os << " has-more=\"" << XmlWriter::boolToStr(result.hasMore) << "\"";
    }

    os << " per-page=\"" << result.perPage << "\">";

    os << "<events>";
    for (const auto& eventModel : result.eventModels) {
        if (!eventModel.event.commitData()) {
            continue;
        }
        os << "<event action=\"" << eventModel.event.commitData()->action() << "\""
           << " created=\"" <<
                    common::canonicalDateTimeString(
                        eventModel.event.createdAt(), common::WithTimeZone::Yes) << "\""
           << " uid=\"" << gdpr::User(eventModel.event.createdBy()).uid() << "\""
           << " id=\"" << eventModel.event.id() << "\">";

        os << eventXmlData(eventModel.event, eventModel.commitModel);
        os << "</event>";
    }
    os << "</events>"
       << "</feed>";
    return os.str();
}


namespace {

void
putTaskInfo(const social::Task& task, XmlOutputStream& output)
{
    output << "<moderation-task id=\"" << task.id() << "\">";
    if (task.isResolved()) {
        output
            << "<resolved"
            << " by=\"" << task.resolved().uid() << "\""
            << " at=\"" << common::canonicalDateTimeString(
                    task.resolved().date(), common::WithTimeZone::Yes) << "\""
            << " resolution=\"" << task.resolved().resolution() << "\""
            << "/>";
    }
    if (task.isClosed()) {
        output
            << "<closed"
            << " by=\"" << task.closed().uid() << "\""
            << " at=\"" << common::canonicalDateTimeString(
                    task.closed().date(), common::WithTimeZone::Yes) << "\""
            << " resolution=\"" << task.closed().resolution() << "\""
            << "/>";
    }
    if (task.isLocked()) {
        output
            << "<acquired"
            << " by=\"" << task.locked().uid() << "\""
            << " at=\"" << common::canonicalDateTimeString(
                    task.locked().date(), common::WithTimeZone::Yes) << "\""
            << "/>";
    }
    output << "</moderation-task>";
}

} // namespace

std::string
XMLFormatter::operator ()(const ResultType<GetSocialModerationTaskByCommit>& result)
{
    ASSERT(result.task);
    XmlOutputStream output;
    XmlWriter::putHeader(output);
    output << "<response-moderation-task>";
    putTaskInfo(*result.task, output);
    output << "</response-moderation-task>";
    XmlWriter::putFooter(output);
    return output.str();
}

} // namespace wiki
} // namespace maps
