#include "xml_writer.h"
#include <maps/wikimap/mapspro/services/editor/src/actions/commit_diff/diff_details.h>

namespace maps {
namespace wiki {

template<>
inline std::string title(const ThreadStopDiffRecord* stopDiff)
{
    return stopDiff->title;
}

namespace {

void
putStateDiff(const DiffDetails& details, std::ostream& output)
{
    auto exists = [](const ObjectPtr& obj) { return obj && !obj->isDeleted(); };
    if (!exists(details.fromObjectVersion)) {
        output << "<state>created</state>";
    } else if (!exists(details.toObjectVersion)) {
        output << "<state>deleted</state>";
    }
}

void
putCategoryDiff(const DiffDetails& details, std::ostream& output)
{
    if (!details.categoryDiff) {
        return;
    }
    output << "<categoryId>";
    output << "<before>" << details.categoryDiff->before << "</before>";
    output << "<after>" << details.categoryDiff->after << "</after>";
    output << "</categoryId>";
}

void
putAttributeDiff(
        const revision::AttributesDiff::value_type& attrDiff,
        bool withBefore,
        std::ostream& output)
{
    if (!cfg()->editor()->isAttributeDefined(attrDiff.first)) {
        WARN() << "Attribute " << attrDiff.first << " is not defined.";
        return;
    }
    const auto& attrDef = cfg()->editor()->attribute(attrDiff.first);
    if (attrDef->system()) {
        return;
    }

    auto putValues = [&](const std::string& valuesStr)
    {
        output << "<values>";
        if (attrDef->multiValue()) {
            StringSet vals = split<StringSet>(
                    valuesStr, AttributeDef::MULTIVALUE_DELIMITER);
            for (const auto& value : vals) {
                output << "<value>" << common::escapeCData(value) << "</value>";
            }
        } else {
            output << "<value>" << common::escapeCData(valuesStr) << "</value>";
        }
        output << "</values>";
    };

    output << "<attribute id=\"" << attrDiff.first << "\">";
    if (withBefore) {
        output << "<before>";
        putValues(attrDiff.second.before);
        output << "</before>";
    }
    output << "<after>";
    putValues(attrDiff.second.after);
    output << "</after>";
    output << "</attribute>";
}

void
putTableAttributesDiff(const DiffDetails& details, std::ostream& output)
{
    ASSERT(details.toObjectVersion);
    for (const auto& attrName : details.toObjectVersion->tableAttributes().attrNames()) {
        output << "<attribute id=\"" << attrName << "\">";
        if (details.fromObjectVersion) {
            output << "<before>";
            XmlWriter::putTableAttributeContents(output,
               details.fromObjectVersion->tableAttributes().find(attrName));
            output << "</before>";
        }
        output << "<after>";
        XmlWriter::putTableAttributeContents(output,
            details.toObjectVersion->tableAttributes().find(attrName));
        output << "</after>";
        output << "</attribute>";
    }
}

void putAttributesDiff(const DiffDetails& details, std::ostream& output)
{
    bool tableAttrsEqual = areTableAttrsEqual(
            details.fromObjectVersion, details.toObjectVersion);
    if (details.attributesDiff.empty() && tableAttrsEqual) {
        return;
    }

    output << "<attributes>";
    for (const auto& attrDiff : details.attributesDiff) {
        putAttributeDiff(
                attrDiff,
                /*withBefore = */details.fromObjectVersion != nullptr,
                output);
    }
    if (!tableAttrsEqual) {
        putTableAttributesDiff(details, output);
    }
    output << "</attributes>";
}

void
putRelativesDiff(
        const RoleToRelativesIdsDiff& roleToRelativesDiff,
        const DiffDetails& details,
        std::ostream& output)
{
    auto putViewObjects = [&](
            const TOIds& oids, const std::map<TOid, views::ViewObject>& relatives)
    {
        for (auto oid : oids) {
            auto it = relatives.find(oid);
            if (it == relatives.end()) {
                continue;
            }
            const auto& obj = it->second;
            output << "<object id=\"" << obj.id() << "\">";
            XmlWriter::putCategory(output, &obj);
            XmlWriter::putScreenLabel(output, &obj);
            output << "</object>";
        }
    };

    for (const auto& pair : roleToRelativesDiff) {
        const auto& role = pair.first;
        const auto& diff = pair.second;
        output << "<role id=\"" << role << "\">";
        output << "<added>";
        putViewObjects(diff.added, details.addedRelatives);
        output << "</added>";
        output << "<removed>";
        putViewObjects(diff.deleted, details.removedRelatives);
        output << "</removed>";
        output << "</role>";
    }
}

void
putMastersDiff(const DiffDetails& details, std::ostream& output)
{
    if (details.mastersDiff.empty()) {
        return;
    }

    output << "<masters>";
    putRelativesDiff(details.mastersDiff, details, output);
    output << "</masters>";
}

void
putSlavesDiff(const DiffDetails& details, std::ostream& output)
{
    if (details.slavesDiff.empty()) {
        return;
    }

    output << "<slaves>";
    putRelativesDiff(details.slavesDiff, details, output);
    output << "</slaves>";
}

void
putGeoms(const std::vector<Geom>& geoms, std::ostream& output)
{
    for (const auto& geom : geoms) {
        if (geom.isNull()) {
            continue;
        }
        XmlWriter::putGeometry(output, geom, SpatialRefSystem::Geodetic);
    }
}

void
putGeometryDiff(const DiffDetails& details, std::ostream& output)
{
    if (details.geomDiff.empty()) {
        return;
    }

    const auto& geomDiff = details.geomDiff;
    output << "<geometry limit-exceeded=\""
           << XmlWriter::boolToStr(geomDiff.isLimitExceeded())
           << "\">";
    output << "<before>";
    putGeoms(geomDiff.before(), output);
    output << "</before>";
    output << "<after>";
    putGeoms(geomDiff.after(), output);
    output << "</after>";
    output << "</geometry>";
}

void
putThreadStopDiffs(const DiffDetails& details, std::ostream& output)
{
    if (!details.threadStopDiff) {
        return;
    }
    if (details.threadStopDiff->brokenSequence) {
        return;
    }
    const auto& diffs = details.threadStopDiff->diffs;

    typedef ThreadStopDiffRecord::Type Type;

    output << "<thread-stops>";
    const size_t totalRecords = diffs.size();
    for (size_t i = 0; i < totalRecords; ++i) {
        const auto& stopDiff = diffs[i];
        if (stopDiff.type == Type::Context
            && (i == 0 || diffs[i - 1].type  == Type::Context)
            && (i == totalRecords - 1 || diffs[i + 1].type == Type::Context)) {
            continue;
        }

        output
            << "<thread-stop"
            << " id=\"" << stopDiff.id << "\""
            << " type=\"" << toString(stopDiff.type) << "\"";
        if (i == 0 || i == totalRecords - 1) {
            output << " end=\"true\"";
        }
        output << ">";

        XmlWriter::putScreenLabel(output, &stopDiff);

        if (stopDiff.attrsDiff) {
            output << "<attributes>";
            for (const auto& attrDiff : *stopDiff.attrsDiff) {
                putAttributeDiff(
                        attrDiff,
                        /*withBefore = */stopDiff.type == Type::Modified,
                        output);
            }
            output << "</attributes>";
        }

        output << "</thread-stop>";
    }
    output << "</thread-stops>";
}

void
putErrors(const DiffDetails& details, std::ostream& output)
{
    if (details.hasErrors()) {
        output << "<errors>";
            if (details.threadStopDiff && details.threadStopDiff->brokenSequence) {
                output << "<thread-stops-broken-sequence/>";
            }
        output << "</errors>";
    }
}

} // namespace

std::ostream&
XmlWriter::put(std::ostream& output, const DiffDetails& details)
{
    output << "<diff>";
    putStateDiff(details, output);
    putCategoryDiff(details, output);

    if (details.toObjectVersion && !details.toObjectVersion->isDeleted()) {
        putAttributesDiff(details, output);

        putMastersDiff(details, output);

        putSlavesDiff(details, output);
    }
    putGeometryDiff(details, output);

    putThreadStopDiffs(details, output);
    putErrors(details, output);
    output << "</diff>";
    return output;
}

} // namespace wiki
} // namespace maps
