#include "ad.h"
#include "calc.h"
#include <maps/wikimap/mapspro/services/editor/src/objects/attr_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/areal_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/point_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/complex_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/relation_object.h>
#include <maps/wikimap/mapspro/services/editor/src/edit_options.h>
#include "registry.h"
#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include "generic.h"

namespace maps {
namespace wiki {
namespace srv_attrs {

typedef std::multimap<std::string, TOid> LevelKindToId;

namespace {

StringSet
levelKindCfgValues()
{
    StringSet values;
    ASSERT(cfg()->editor()->attribute(ATTR_AD_LEVEL_KIND));
    for (const auto& value : cfg()->editor()->attribute(ATTR_AD_LEVEL_KIND)->values()) {
        values.insert(value.value);
    }
    return values;
}

std::string
levelKindsMinValue(const StringSet& lvlKinds)
{
    std::set<size_t> parsed;
    for (const auto& lvlKind : lvlKinds) {
        try
        {
            parsed.insert(boost::lexical_cast<size_t>(lvlKind));
        } catch (...) {
        }
    }
    if (parsed.empty()) {
        return s_emptyString;
    }
    return std::to_string(*parsed.begin());
}

LevelKindToId
levelKindsToMastersId(ObjectsCache& cache, const std::string& masterRole, TOid slaveId)
{
    LevelKindToId levelKindToId;
    for (const auto& masterInfo
        : cache.getExisting(slaveId)->masterRelations().range(masterRole)) {
        auto master = masterInfo.relative();
        levelKindToId.insert(
            {
                master->attributes().value(
                    masterInfo.categoryId() == CATEGORY_AD
                    ? ATTR_AD_LEVEL_KIND
                    : ATTR_AD_SUBST_LEVEL_KIND),
                master->id()
            }
        );
    }
    return levelKindToId;
}

std::string
formatAdName(const std::string& officialAdName, const std::string& levelKindValue)
{
    if (!officialAdName.empty()) {
        return officialAdName + " ("
            + valueLabel(ATTR_AD_LEVEL_KIND, levelKindValue)
            + ")";
    }
    return officialAdName;
}

} // namespace

class AD_JC : public ServiceAttributesRegistry::Registrar
{
public:
    AD_JC(ServiceAttributesRegistry& registry)
        : ServiceAttributesRegistry::Registrar(registry, CATEGORY_AD_JC)
    {
        registerAttr(CATEGORY_AD_JC + SUFFIX_CAN_DELETE, CallbackWrapper<Junction>(canDelete))
            .depends(
        {
            {Aspect::Type::Attribute, ""},
            {{RelationType::Master, ROLE_START, CATEGORY_AD_EL}}
        })
            .depends(
        {
            {Aspect::Type::Attribute, ""},
            {{RelationType::Master, ROLE_END, CATEGORY_AD_EL}}
        });
        registerAttr(SRV_VALENCY, CallbackWrapper<Junction>(valency))
            .depends(
        {
            {Aspect::Type::Relations, STR_TO_ATTR_OWNER},
            {{RelationType::Master, ROLE_START, CATEGORY_AD_EL}}
        })
            .depends(
        {
            {Aspect::Type::Relations, STR_TO_ATTR_OWNER},
            {{RelationType::Master, ROLE_END, CATEGORY_AD_EL}}
        });
    }

private:
    static std::string canDelete(const Junction* jc, ObjectsCache&)
    {
        return maps::wiki::canDelete(jc, 0)
            ? SRV_ATTR_TRUE
            : s_emptyString;
    }

    static std::string valency(const Junction* jc, ObjectsCache&)
    {
        return std::to_string(jc->valency());
    }
};
namespace {
std::string
isPartOfRegionN(const LinearElement* el, ObjectsCache&, const std::string& regionRole)
{
    TOIds regions;
    for (const auto& contour : el->masterRelations().range(ROLE_PART)) {
        for (const auto& ad : contour.relative()->masterRelations().range(ROLE_PART)) {
            for (const auto& region : ad.relative()->masterRelations().range(regionRole)) {
                ASSERT(region.categoryId() == CATEGORY_REGION);
                regions.insert(region.id());
            }
        }
    }
    return regions.empty()
        ? SRV_ATTR_FALSE
        : std::to_string(regions.size());
}

SuffixValues
isPartOfRegionXN(const LinearElement* el, ObjectsCache& cache, const std::string& regionRole)
{
    std::map<TOid, TOIds> regionToADs;
    for (const auto& contour : el->masterRelations().range(ROLE_PART)) {
        for (const auto& ad : contour.relative()->masterRelations().range(ROLE_PART)) {
            for (const auto& region : ad.relative()->masterRelations().range(regionRole)) {
                ASSERT(region.categoryId() == CATEGORY_REGION);
                regionToADs[region.id()].insert(ad.id());
            }
        }
    }
    SuffixValues regionNames;
    for (const auto& regionToAD : regionToADs) {
        const auto regionId = regionToAD.first;
        const auto adCount = regionToAD.second.size();
        const auto regionName = objectNameByType(
                        regionId,
                        NAME_TYPE_OFFICIAL,
                        cache);
        regionNames.emplace_back(regionName, std::to_string(adCount));
    }
    return regionNames;
}

std::string
isNotPartOfRegion(const LinearElement* el, ObjectsCache&, const std::string& regionRole)
{
    for (const auto& contour : el->masterRelations().range(ROLE_PART)) {
        for (const auto& ad : contour.relative()->masterRelations().range(ROLE_PART)) {
            if (ad.relative()->attributes().value(ATTR_AD_LEVEL_KIND) == AD_LEVEL_KIND_1_COUNTRY &&
                ad.relative()->masterRelations().range(regionRole).empty()) {
                return SRV_ATTR_TRUE;
            }
        }
    }
    return SRV_ATTR_FALSE;
}
} // namespace

class AD_EL : public ServiceAttributesRegistry::Registrar
{
public:
    AD_EL(ServiceAttributesRegistry& registry)
        : ServiceAttributesRegistry::Registrar(registry, CATEGORY_AD_EL)
    {
        registerAttr(SRV_IS_INTERIOR,
            CallbackWrapper<LinearElement, std::string>(GenericEL::isPartOfInterior, CATEGORY_AD_FC))
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_FC_IS_INTERIOR},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_FC}
            }
        });
        registerAttr(SRV_HOTSPOT_LABEL, CallbackWrapper<LinearElement>(hotSpotLabel))
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_LEVEL_KIND},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_FC},
                {RelationType::Master, ROLE_PART, CATEGORY_AD}
            }
        })
            .depends(
        {
            {Aspect::Type::Attribute, AD_NM},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_FC},
                {RelationType::Master, ROLE_PART, CATEGORY_AD}
            }
        });
        registerAttr(SRV_SUBSTITUTION, CallbackWrapper<LinearElement>(ownedByDisputedAD))
            .depends(
        {
            {Aspect::Type::Attribute, ""},
            {{RelationType::Master, ROLE_PART, CATEGORY_AD_SUBST_FC}}
        })
            .depends(
        {
            {Aspect::Type::Attribute, ""},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_FC},
                {RelationType::Master, ROLE_PART, CATEGORY_AD},
                {RelationType::Slave, ROLE_SUBSTITUTION, CATEGORY_AD_SUBST}
            }
        });

        for (const std::string& lvlKind : levelKindCfgValues()) {
            registerAttr("ad_el:level_kind_" + lvlKind,
                    CallbackWrapper<LinearElement, std::string>(findLevelKind, lvlKind))
                .depends(
            {
                {Aspect::Type::Attribute, ATTR_AD_LEVEL_KIND},
                {
                    {RelationType::Master, ROLE_PART, CATEGORY_AD_FC},
                    {RelationType::Master, ROLE_PART, CATEGORY_AD}
                }
            })
                .depends(
            {
                {Aspect::Type::Attribute, ATTR_AD_SUBST_LEVEL_KIND},
                {
                    {RelationType::Master, ROLE_PART, CATEGORY_AD_SUBST_FC},
                    {RelationType::Master, ROLE_PART, CATEGORY_AD_SUBST}
                }
            });
        }
        registerAttr("ad_el:level_kind_min", CallbackWrapper<LinearElement>(levelKindMin))
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_LEVEL_KIND},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_FC},
                {RelationType::Master, ROLE_PART, CATEGORY_AD}
            }
        })
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_LEVEL_KIND},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_SUBST_FC},
                {RelationType::Master, ROLE_PART, CATEGORY_AD_SUBST}
            }
        });

        registerAttr("ad_el:is_part_of_region",
            CallbackWrapper<LinearElement, const std::string&>(
                isPartOfRegionN,
                ROLE_ASSIGNED))
            .depends(
        {
            {Aspect::Type::Relations, STR_TO_MASTER},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_FC},
                {RelationType::Master, ROLE_PART, CATEGORY_AD}
            }
        });
        registerAttr("ad_el:is_not_part_of_region",
            CallbackWrapper<LinearElement, const std::string&>(
                isNotPartOfRegion,
                ROLE_ASSIGNED))
            .depends(
        {
            {Aspect::Type::Relations, STR_TO_MASTER},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_FC},
                {RelationType::Master, ROLE_PART, CATEGORY_AD}
            }
        });

        registerAttr("ad_el:is_part_of_region_",
            SuffixValuesCallbackWrapper<LinearElement, const std::string&>(
                isPartOfRegionXN,
                ROLE_ASSIGNED))
            .depends(
        {
            {Aspect::Type::Attribute, AD_NM},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_FC},
                {RelationType::Master, ROLE_PART, CATEGORY_AD},
                {RelationType::Master, ROLE_ASSIGNED, CATEGORY_REGION}
            }
        });
    }

private:
    static std::string hotSpotLabel(const LinearElement* el, ObjectsCache& cache)
    {
        const auto& contours = el->masterRelations().range(ROLE_PART);
        if (contours.empty()) {
            return s_emptyString;
        }
        LevelKindToId levelKindToIdAll;
        for (const auto& contour : contours) {
            if (contour.categoryId() != CATEGORY_AD_FC) {
                continue;
            }
            auto levelKindToId = levelKindsToMastersId(cache, ROLE_PART, contour.id());
            levelKindToIdAll.insert(levelKindToId.begin(), levelKindToId.end());
        }

        StringSet labelLines;
        for (const auto& levelKindAndId : levelKindToIdAll) {
            auto labelLine =
                formatAdName(
                    objectNameByType(
                        levelKindAndId.second,
                        NAME_TYPE_OFFICIAL,
                        cache),
                levelKindAndId.first);
            if (!labelLine.empty()) {
                labelLines.insert(labelLine);
            }
        }
        return common::join(labelLines, '\n');
    }

    static std::string ownedByDisputedAD(const LinearElement* el, ObjectsCache&)
    {
        const auto& contours = el->masterRelations().range(ROLE_PART);
        if (contours.empty()) {
            return s_emptyString;
        }
        for (const auto& contour : contours) {
            if (contour.categoryId() == CATEGORY_AD_SUBST_FC) {
                return SRV_ATTR_TRUE;
            }
        }
        for (const auto& contour : contours) {
            const auto& ads = contour.relative()->masterRelations().range(ROLE_PART);
            for (const auto& ad : ads) {
                if (!ad.relative()->slaveRelations().range(ROLE_SUBSTITUTION).empty()) {
                    return SRV_ATTR_TRUE;
                }
            }
        }
        return SRV_ATTR_FALSE;
    }

    static std::string findLevelKind(
            const LinearElement* el,
            ObjectsCache&,
            const std::string& levelKindVal)
    {
        const auto& contours = el->masterRelations().range(ROLE_PART);
        for (const auto& contour : contours) {
            for (const auto& ad : contour.relative()->masterRelations().range(ROLE_PART)) {
                if (ad.relative()->attributes().value(ATTR_AD_LEVEL_KIND) == levelKindVal) {
                    return SRV_ATTR_TRUE;
                }
            }
        }
        return s_emptyString;
    }

    static std::string levelKindMin(
            const LinearElement* el,
            ObjectsCache&)
    {
        const auto& contours = el->masterRelations().range(ROLE_PART);
        if (contours.empty()) {
            return s_emptyString;
        }
        StringSet levelKinds;
        for (const auto& contour : contours) {
            for (const auto& ad : contour.relative()->masterRelations().range(ROLE_PART)) {
                levelKinds.insert(
                        ad.relative()->attributes().value(ATTR_AD_LEVEL_KIND));
            }
        }
        return levelKindsMinValue(levelKinds);
    }
};

class AD_NEUTRAL_EL : public ServiceAttributesRegistry::Registrar
{
public:
    AD_NEUTRAL_EL(ServiceAttributesRegistry& registry)
        : ServiceAttributesRegistry::Registrar(registry, CATEGORY_AD_NEUTRAL_EL)
    {
        registerAttr(CATEGORY_AD_NEUTRAL_FC + SUFFIX_IS_INTERIOR,
            CallbackWrapper<LinearElement, std::string>(GenericEL::isPartOfInterior, CATEGORY_AD_NEUTRAL_FC))
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_NEUTRAL_FC_IS_INTERIOR},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_NEUTRAL_FC}
            }
        });
        registerAttr(SRV_HOTSPOT_LABEL, CallbackWrapper<LinearElement>(hotSpotLabel))
            .depends(
        {
            {Aspect::Type::Attribute, AD_NM},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_NEUTRAL_FC},
                {RelationType::Master, ROLE_PART, CATEGORY_AD_NEUTRAL}
            }
        });

        registerAttr("ad_neutral_el:is_part_of_region",
            CallbackWrapper<LinearElement, const std::string&>(
                isPartOfRegionN,
                ROLE_ASSIGNED_AD_NEUTRAL))
            .depends(
        {
            {Aspect::Type::Relations, STR_TO_MASTER},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_NEUTRAL_FC},
                {RelationType::Master, ROLE_PART, CATEGORY_AD_NEUTRAL}
            }
        });

        registerAttr("ad_neutral_el:is_not_part_of_region",
            CallbackWrapper<LinearElement, const std::string&>(
                isNotPartOfRegion,
                ROLE_ASSIGNED_AD_NEUTRAL))
            .depends(
        {
            {Aspect::Type::Relations, STR_TO_MASTER},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_NEUTRAL_FC},
                {RelationType::Master, ROLE_PART, CATEGORY_AD_NEUTRAL}
            }
        });

        registerAttr("ad_neutral_el:is_part_of_region_",
            SuffixValuesCallbackWrapper<LinearElement, const std::string&>(
                isPartOfRegionXN,
                ROLE_ASSIGNED_AD_NEUTRAL))
            .depends(
        {
            {Aspect::Type::Attribute, AD_NM},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_AD_FC},
                {RelationType::Master, ROLE_PART, CATEGORY_AD},
                {RelationType::Master, ROLE_ASSIGNED_AD_NEUTRAL, CATEGORY_REGION}
            }
        });
    }

private:
    static std::string hotSpotLabel(const LinearElement* el, ObjectsCache& cache)
    {
        TOIds adNeutrals;
        for (const auto& contour : el->masterRelations().range(ROLE_PART)) {
            for (const auto& adNeutral : contour.relative()->masterRelations().range(ROLE_PART)) {
                adNeutrals.insert(adNeutral.id());
            }
        }
        StringSet labelLines;
        for (const auto& adNeutralId : adNeutrals) {
            auto labelLine = objectNameByType(
                        adNeutralId,
                        NAME_TYPE_OFFICIAL,
                        cache);
            if (!labelLine.empty()) {
                labelLines.insert(labelLine);
            }
        }
        return common::join(labelLines, '\n');
    }
};


class AD_CNT : public ServiceAttributesRegistry::Registrar
{
public:
    AD_CNT(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_AD_CNT)
    {
        registerAttr(SRV_RENDER_LABEL, CallbackWrapper<PointObject>(renderLabel))
            .depends(
        {
            {Aspect::Type::Attribute, AD_NM},
            {{RelationType::Master, ROLE_CENTER, CATEGORY_AD}}
        });
        registerAttr(SRV_HOTSPOT_LABEL, CallbackWrapper<PointObject>(hotSpotLabel))
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_LEVEL_KIND},
            {{RelationType::Master, ROLE_CENTER, CATEGORY_AD}}
        })
            .depends(
        {
            {Aspect::Type::Attribute, AD_NM},
            {{RelationType::Master, ROLE_CENTER, CATEGORY_AD}}
        });
        for (const std::string& lvlKind : levelKindCfgValues()) {
            registerAttr("ad_cnt:level_kind_" + lvlKind,
                    CallbackWrapper<PointObject, std::string>(findLevelKind, lvlKind))
                .depends(
            {
                {Aspect::Type::Attribute, ATTR_AD_LEVEL_KIND},
                {{RelationType::Master, ROLE_CENTER, CATEGORY_AD}}
            });
        }
        registerAttr("ad_cnt:capital",
                    CallbackWrapper<PointObject>(capital))
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_CAPITAL},
            {{RelationType::Master, ROLE_CENTER, CATEGORY_AD}}
        });
        registerAttr("ad_cnt:population",
                    CallbackWrapper<PointObject>(population))
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_POPULATION},
            {{RelationType::Master, ROLE_CENTER, CATEGORY_AD}}
        });
        registerAttr("ad_cnt:town",
                    CallbackWrapper<PointObject>(town))
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_TOWN},
            {{RelationType::Master, ROLE_CENTER, CATEGORY_AD}}
        });
        registerAttr("ad_cnt:disp_class",
                    CallbackWrapper<PointObject>(dispclass))
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_DISP_CLASS},
            {{RelationType::Master, ROLE_CENTER, CATEGORY_AD}}
        });
        registerAttr("srv:zoom_interval",
                    CallbackWrapper<PointObject>(zoomInterval))
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_LEVEL_KIND},
            {{RelationType::Master, ROLE_CENTER, CATEGORY_AD}}
        });
    }

private:
    static std::string hotSpotLabel(const PointObject* pt, ObjectsCache& cache)
    {
        const auto& ads = pt->masterRelations().range(ROLE_CENTER);
        for (const auto& ad : ads) {
            return formatAdName(objectNameByType(ad.id(), NAME_TYPE_OFFICIAL, cache),
                    ad.relative()->attributes().value(ATTR_AD_LEVEL_KIND));
        }
        return s_emptyString;
    }

    static std::string renderLabel(const PointObject* pt, ObjectsCache& cache)
    {
        const auto& ads = pt->masterRelations().range(ROLE_CENTER);
        for (const auto& ad : ads) {
            std::string label = objectNameByType(ad.id(), NAME_TYPE_RENDER_LABEL, cache);
            return label.empty()
                ? objectNameByType(ad.id(), NAME_TYPE_OFFICIAL, cache)
                : label;
        }
        return s_emptyString;
    }

    static std::string findLevelKind(const PointObject* pt,
                                    ObjectsCache&,
                                    const std::string& levelKindVal)
    {
        const auto& ads = pt->masterRelations().range(ROLE_CENTER);
        for (const auto& ad : ads) {
            if (ad.relative()->attributes().value(ATTR_AD_LEVEL_KIND) == levelKindVal) {
                return SRV_ATTR_TRUE;
            }
        }
        return s_emptyString;
    }

    static std::string capital(const PointObject* pt, ObjectsCache&)
    {
        const auto& ads = pt->masterRelations().range(ROLE_CENTER);
        for (const auto& ad : ads) {
            return ad.relative()->attributes().value(ATTR_AD_CAPITAL);
        }
        return s_emptyString;
    }

    static std::string population(const PointObject* pt, ObjectsCache&)
    {
        const auto& ads = pt->masterRelations().range(ROLE_CENTER);
        for (const auto& ad : ads) {
            return ad.relative()->attributes().value(ATTR_AD_POPULATION);
        }
        return s_emptyString;
    }
    static std::string town(const PointObject* pt, ObjectsCache&)
    {
        const auto& ads = pt->masterRelations().range(ROLE_CENTER);
        for (const auto& ad : ads) {
            return ad.relative()->attributes().value(ATTR_AD_TOWN);
        }
        return s_emptyString;
    }
    static std::string dispclass(const PointObject* pt, ObjectsCache&)
    {
        const auto& ads = pt->masterRelations().range(ROLE_CENTER);
        for (const auto& ad : ads) {
            return ad.relative()->attributes().value(ATTR_AD_DISP_CLASS);
        }
        return s_emptyString;
    }
    static std::string zoomInterval(const PointObject* pt, ObjectsCache&)
    {
        const auto& ads = pt->masterRelations().range(ROLE_CENTER);
        for (const auto& ad : ads) {
            auto levelKind =  ad.relative()->attributes().value(ATTR_AD_LEVEL_KIND);
            auto capital = ad.relative()->attributes().value(ATTR_AD_CAPITAL);
            auto populationStr = ad.relative()->attributes().value(ATTR_AD_POPULATION);
            auto town = ad.relative()->attributes().value(ATTR_AD_TOWN);
            int population = populationStr.empty() ?  0 : std::stoi(populationStr);

            if (levelKind == "1") {
                return "4-6";
            } else if (levelKind == "4" && capital == "1"){
                return "4-16";
            } else if (levelKind == "4" && capital == "2"){
                return "5-16";
            } else if (levelKind == "2"){
                return "7-8";
            } else if (levelKind == "4" && population > 50000) {
                return "7-16";
            } else if  (levelKind == "3" ){
                return "9-10";
            } else if  (levelKind == "4" and town == SRV_ATTR_TRUE){
                return "11-16";
            } else if  (levelKind == "4" ){
                return "12-18";
            } else if  (levelKind == "5" ){
                return "11-13";
            }
        }
        return "14-18";
    }
};

class AD : public ServiceAttributesRegistry::Registrar
{
public:
    AD(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_AD)
    {
        registerAttr(SRV_SCREEN_LABEL, CallbackWrapper<ComplexObject>(screenLabel));
        registerSuggestCallback(SuggestCallbackWrapper<ComplexObject>(GenericNamedObject::suggestTexts));
    }

private:
    static std::string screenLabel(const ComplexObject* ad, ObjectsCache& cache)
    {
        return objectNameByType(ad->id(), NAME_TYPE_OFFICIAL, cache);
    }
};

class AD_SUBST : public ServiceAttributesRegistry::Registrar
{
public:
    AD_SUBST(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_AD_SUBST)
    {
        registerAttr(SRV_SCREEN_LABEL, CallbackWrapper<ComplexObject>(screenLabel));
    }

private:
    static std::string screenLabel(const ComplexObject* adSubst, ObjectsCache& cache)
    {
        return objectNameByType(adSubst->id(), NAME_TYPE_OFFICIAL, cache);
    }
};

class AD_FC : public ServiceAttributesRegistry::Registrar
{
public:
    AD_FC(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_AD_FC)
    {
        registerAttr(SRV_SCREEN_LABEL, CallbackWrapper<ComplexObject>(screenLabel))
            .depends(
        {
            {Aspect::Type::Attribute, AD_NM},
            {{RelationType::Master, ROLE_PART, CATEGORY_AD}}
        })
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_LEVEL_KIND},
            {{RelationType::Master, ROLE_PART, CATEGORY_AD}}
        });
    }

private:
    static std::string screenLabel(const ComplexObject* obj, ObjectsCache& cache)
    {
        auto levelKindToId = levelKindsToMastersId(cache, ROLE_PART, obj->id());
        std::string s;
        for(const auto& levelKindAndId: levelKindToId) {
            std::string ad = objectNameByType(levelKindAndId.second, NAME_TYPE_OFFICIAL, cache);
            if (!ad.empty()) {
                if (!s.empty()) {
                    s += ", ";
                }
                s += formatAdName(ad, levelKindAndId.first);
            }
        }
        if (!s.empty()) {
            std::string interior = obj->attributes().value(ATTR_AD_FC_IS_INTERIOR);
            if (interior == SRV_ATTR_TRUE) {
                const auto& attrIsInterior = obj->attributes().find(ATTR_AD_FC_IS_INTERIOR)->def();
                s += " - " + attrIsInterior.label();
            }
        }
        return s;
    }
};

class AD_SUBST_FC : public ServiceAttributesRegistry::Registrar
{
public:
    AD_SUBST_FC(ServiceAttributesRegistry& registry)
        : ServiceAttributesRegistry::Registrar(registry, CATEGORY_AD_SUBST_FC)
    {
        registerAttr(SRV_SCREEN_LABEL, CallbackWrapper<ComplexObject>(screenLabel))
            .depends(
        {
            {Aspect::Type::Attribute, AD_NM},
            {{RelationType::Master, ROLE_PART, CATEGORY_AD_SUBST}}
        })
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_AD_SUBST_LEVEL_KIND},
            {{RelationType::Master, ROLE_PART, CATEGORY_AD_SUBST}}
        });
    }

private:
    static std::string screenLabel(const ComplexObject* obj, ObjectsCache& cache)
    {
        auto levelKindToId = levelKindsToMastersId(cache, ROLE_PART, obj->id());
        std::string s;
        for(const auto& levelKindAndId: levelKindToId) {
            std::string ad = objectNameByType(levelKindAndId.second, NAME_TYPE_OFFICIAL, cache);
            if (!ad.empty()) {
                if (!s.empty()) {
                    s += ", ";
                }
                s += formatAdName(ad, levelKindAndId.first);
            }
        }
        if (!s.empty()) {
            std::string excluded = obj->attributes().value(ATTR_AD_SUBST_FC_IS_EXCLUDED);
            if (excluded == SRV_ATTR_TRUE) {
                const auto& attrIsExcluded = obj->attributes().find(ATTR_AD_SUBST_FC_IS_EXCLUDED)->def();
                s += " - " + attrIsExcluded.label();
            }
        }
        return s;
    }
};

ATDServiceAttributes::ATDServiceAttributes(ServiceAttributesRegistry &registry)
{
    AD_JC adJc(registry);
    AD_EL adEl(registry);
    AD_CNT adCnt(registry);
    AD_FC adFc(registry);
    AD ad(registry);

    AD_SUBST_FC adSubstFc(registry);
    AD_SUBST adSubst(registry);

    AD_NEUTRAL_EL neutralEl(registry);
}
}//srv_attrs
}//wiki
}//maps
