#include "generic.h"
#include "calc.h"
#include <maps/wikimap/mapspro/services/editor/src/objects/attr_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/areal_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/point_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/complex_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/relation_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/edit_options.h>
#include "registry.h"
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>

#include <yandex/maps/wiki/configs/editor/categories.h>

namespace maps {
namespace wiki {
namespace srv_attrs {

namespace {
const std::string PLANE_NAME_ATTR_SUFFIX = ":name";

const std::vector<std::string> GENERIC_POINT_CATEGORIES =
{
    CATEGORY_POI_MEDICINE,
    CATEGORY_POI_EDU,
    CATEGORY_POI_FINANCE,
    CATEGORY_POI_SHOPPING,
    CATEGORY_POI_GOVERMENT,
    CATEGORY_POI_RELIGION,
    CATEGORY_POI_FOOD,
    CATEGORY_POI_AUTO,
    CATEGORY_POI_SPORT,
    CATEGORY_POI_LEISURE,
    CATEGORY_POI_URBAN,
    CATEGORY_POI_SERVICE,
    CATEGORY_URBAN_ROADNET_PARKING_LOT
};

const std::vector<std::string> GENERIC_NAMED_CATEGORIES =
{
    CATEGORY_POI_MEDICINE,
    CATEGORY_POI_EDU,
    CATEGORY_POI_FINANCE,
    CATEGORY_POI_SHOPPING,
    CATEGORY_POI_GOVERMENT,
    CATEGORY_POI_RELIGION,
    CATEGORY_POI_FOOD,
    CATEGORY_POI_AUTO,
    CATEGORY_POI_SPORT,
    CATEGORY_POI_LEISURE,
    CATEGORY_POI_URBAN,
    CATEGORY_POI_SERVICE,
    CATEGORY_COND_CLOSURE,
    CATEGORY_INDOOR_LEVEL,
    CATEGORY_INDOOR_PLAN,
    CATEGORY_INDOOR_POI_AUTO,
    CATEGORY_INDOOR_POI_FOOD,
    CATEGORY_INDOOR_POI_INFO,
    CATEGORY_INDOOR_POI_INFRA,
    CATEGORY_INDOOR_POI_SERVICE,
    CATEGORY_INDOOR_POI_FINANCE,
    CATEGORY_INDOOR_POI_MEDICINE,
    CATEGORY_INDOOR_POI_EDU,
    CATEGORY_INDOOR_POI_LEISURE,
    CATEGORY_INDOOR_POI_SPORT,
    CATEGORY_INDOOR_POI_GOVERMENT,
    CATEGORY_INDOOR_POI_SHOPPING,
    CATEGORY_INDOOR_POI_RELIGION,
    CATEGORY_TRANSPORT_BUS_ROUTE,
    CATEGORY_TRANSPORT_OPERATOR,
    CATEGORY_TRANSPORT_AIRPORT,
    CATEGORY_TRANSPORT_AIRPORT_TERMINAL,
    CATEGORY_TRANSPORT_HELICOPTER,
    CATEGORY_TRANSPORT_TRAM_ROUTE,
    CATEGORY_TRANSPORT_TERMINAL,
    CATEGORY_VEGETATION,
    CATEGORY_REGION,
    CATEGORY_AD_NEUTRAL,
    CATEGORY_RELIEF,
    CATEGORY_RELIEF_POINT,
    CATEGORY_TRANSPORT_WATERWAY_ROUTE,
    CATEGORY_TRANSPORT_WATERWAY_STOP,
    CATEGORY_URBAN,
    CATEGORY_URBAN_ROADNET,
    CATEGORY_URBAN_ROADNET_AREAL,
    CATEGORY_URBAN_ROADNET_PARKING_CONTROLLED_ZONE,
    CATEGORY_ZIPCODE
};

const AttrDefPtr EMPTY_ATTR_DEF_PTR;
} // namespace

const AttrDefPtr&
ftTypeAttrDefForCategory(const std::string& categoryId)
{
    std::string attrId = categoryId + SUFFIX_FT_TYPE_ID;
    if (cfg()->editor()->isAttributeDefined(attrId)) {
        return cfg()->editor()->attribute(attrId);
    }
    return EMPTY_ATTR_DEF_PTR;
}

const AttrDefPtr&
enumeratedTypeAttrDefForCategory(const std::string& categoryId)
{
    std::string attrId = categoryId + SUFFIX_TYPE_ATTR;
    if (cfg()->editor()->isAttributeDefined(attrId)) {
        const auto& defPtr = cfg()->editor()->attribute(attrId);
        if (defPtr->valueType() == ValueType::Enum) {
            return defPtr;
        }
    }
    return EMPTY_ATTR_DEF_PTR;
}

const AttrDefPtr&
plainNameAttrDefForCategory(const std::string& categoryId)
{
    const auto& categories = cfg()->editor()->categories();
    if (!categories.defined(categoryId)) {
        return EMPTY_ATTR_DEF_PTR;
    }
    const auto& category = categories[categoryId];
    for (const auto& attrDefPtr : category.attrDefs()) {
        if (attrDefPtr->valueType() == ValueType::String &&
            attrDefPtr->id().ends_with(PLANE_NAME_ATTR_SUFFIX))
        {
            return attrDefPtr;
        }
    }
    return EMPTY_ATTR_DEF_PTR;
}

GenericJC::GenericJC(const std::string& jcCategoryId,
    const std::string& elCategoryId,
    ServiceAttributesRegistry& registry)
    : ServiceAttributesRegistry::Registrar(registry, jcCategoryId)
{
    registerAttr(elCategoryId + SUFFIX_CAN_DELETE, CallbackWrapper<Junction>(canDelete))
        .depends(
    {
        {Aspect::Type::Attribute, ""},
        {{RelationType::Master, ROLE_START, elCategoryId}}
    })
        .depends(
    {
        {Aspect::Type::Attribute, ""},
        {{RelationType::Master, ROLE_END, elCategoryId}}
    });
    registerAttr(SRV_VALENCY, CallbackWrapper<Junction>(valency))
        .depends(
    {
        {Aspect::Type::Relations, STR_TO_ATTR_OWNER},
        {{RelationType::Master, ROLE_START, elCategoryId}}
    })
        .depends(
    {
        {Aspect::Type::Relations, STR_TO_ATTR_OWNER},
        {{RelationType::Master, ROLE_END, elCategoryId}}
    });
}

std::string
GenericJC::canDelete(const Junction* jc, ObjectsCache&)
{
    return maps::wiki::canDelete(jc, 0)
        ? SRV_ATTR_TRUE
        : SRV_ATTR_FALSE;
}

std::string
GenericJC::valency(const Junction* jc, ObjectsCache&)
{
    return std::to_string(jc->valency());
}


GenericEL::GenericEL(
        const std::string& elCategoryId,
        const std::string& featureCategoryId,
        const std::string& contourCategoryId,
        const std::string& nameAttribute,
        ServiceAttributesRegistry& registry)
    : ServiceAttributesRegistry::Registrar(registry, elCategoryId)
{
    DependenceType dependenceType = contourCategoryId.empty()
        ? DependenceType::Direct
        : DependenceType::Indirect;
    registerScreenLabel(
        dependenceType,
        featureCategoryId,
        contourCategoryId,
        nameAttribute);
    registerFtTypes(dependenceType, featureCategoryId, contourCategoryId);
    if (!contourCategoryId.empty()) {
        registerSrvIsInterior(contourCategoryId);
    }
}

void
GenericEL::registerSrvIsInterior(const std::string& countourCategoryId)
{
    registerAttr(SRV_IS_INTERIOR, CallbackWrapper<LinearElement, std::string>(isPartOfInterior, countourCategoryId))
            .depends(
        {
            {Aspect::Type::Attribute, countourCategoryId + SUFFIX_IS_INTERIOR},
            {
                {RelationType::Master, ROLE_PART, countourCategoryId}
            }
        });
}

std::string
GenericEL::isPartOfInterior(
            const LinearElement* el,
            ObjectsCache& /*cache*/,
            const std::string& countourCategoryId)
{
    const auto& contours = el->masterRelations().range(ROLE_PART);
    if (contours.empty()) {
        return SRV_ATTR_FALSE;
    }
    for (const auto& contour : contours) {
        if (contour.categoryId() != countourCategoryId) {
            continue;
        }
        if (!contour.relative()->attributes().value(countourCategoryId + SUFFIX_IS_INTERIOR).empty()) {
            return SRV_ATTR_TRUE;
        }
    }
    return SRV_ATTR_FALSE;
}

void
GenericEL::registerScreenLabel(
        DependenceType dependenceType,
        const std::string& featureCategoryId,
        const std::string& contourCategoryId,
        const std::string& nameAttribute)
{
    auto& srvAttr = registerAttr(SRV_SCREEN_LABEL,
        CallbackWrapper<LinearElement,DependenceType>(
        screenLabel,
        dependenceType
        ));
    if (dependenceType == DependenceType::Direct) {
        srvAttr.depends(
        {
            {Aspect::Type::Attribute, nameAttribute},
            {
                {RelationType::Master, ROLE_PART, featureCategoryId}
            }
        });
    } else {
        srvAttr.depends(
        {
            {Aspect::Type::Attribute, nameAttribute},
            {
                {RelationType::Master, ROLE_PART, contourCategoryId},
                {RelationType::Master, ROLE_PART, featureCategoryId}
            }
        });
    }
}

void
GenericEL::registerFtTypes(
        DependenceType dependenceType,
        const std::string& featureCategoryId,
        const std::string& contourCategoryId)
{
    registerFtTypes(*this, dependenceType, featureCategoryId, contourCategoryId);
}

void
GenericEL::registerFtTypes(
        ServiceAttributesRegistry::Registrar& registrar,
        DependenceType dependenceType,
        const std::string& featureCategoryId,
        const std::string& contourCategoryId)
{
    const auto& ftTypeAttrDef = ftTypeAttrDefForCategory(featureCategoryId);
    if (!ftTypeAttrDef) {
        return;
    }
    for (const auto& ftTypeAttrValue :  ftTypeAttrDef->values()) {
        const std::string ftTypeSrvAttrName = SRV_FT_TYPE_ID
            + "_" + ftTypeAttrValue.value;
        auto& srvAttr = registrar.registerAttr(ftTypeSrvAttrName,
            CallbackWrapper<LinearElement, std::string, std::string, DependenceType>(
            GenericEL::ftTypeSrvAttrValue,
            ftTypeAttrDef->id(),//rd:ft_type_id
            ftTypeAttrValue.value,//611
            dependenceType
            ));
        if (dependenceType == DependenceType::Direct) {
            srvAttr.depends(
            {
                {Aspect::Type::Attribute, ftTypeAttrDef->id()},
                {{RelationType::Master, ROLE_PART, featureCategoryId}}
            });
        } else {
            srvAttr.depends(
            {
                {Aspect::Type::Attribute, ftTypeAttrDef->id()},
                {
                    {RelationType::Master, ROLE_PART, contourCategoryId},
                    {RelationType::Master, ROLE_PART, featureCategoryId}
                }
            });
        }
    }
}

std::string
GenericEL::screenLabel(const LinearElement* el, ObjectsCache& cache,
        DependenceType dependenceType)
{
    TOIds sources;
    const auto& partMasters = el->masterRelations().range(ROLE_PART);
    if (partMasters.empty()) {
        return s_emptyString;
    }
    for (const auto& partMaster : partMasters) {
        if (dependenceType == DependenceType::Direct) {
            sources.insert(partMaster.id());
        } else {
            for (const auto& feature :
                partMaster.relative()->masterRelations().range(ROLE_PART)) {
                sources.insert(feature.id());
            }
        }
    }
    std::string screenLabelResult;
    for (const auto& oid : sources) {
        auto name = GenericNamedObject::hotspotLabel(
            cache.getExisting(oid).get(), cache);
        if (name.empty()) {
            continue;
        }
        if (!screenLabelResult.empty()) {
            screenLabelResult += ", ";
        }
        screenLabelResult += name;
    }
    return screenLabelResult;
}

std::string
GenericEL::ftTypeSrvAttrValue(
        const LinearElement* el,
        ObjectsCache&/* cache*/,
        const std::string& ftTypeAttrId,
        const std::string& ftTypeValue,
        DependenceType dependenceType)
{
    const auto& partMasters = el->masterRelations().range(ROLE_PART);
    for (const auto& partMaster : partMasters) {
        if (dependenceType == DependenceType::Direct) {
            if (partMaster.relative()->attributes().value(ftTypeAttrId)
                    == ftTypeValue) {
                    return SRV_ATTR_TRUE;
            }
        } else {
            for (const auto& feature :
                partMaster.relative()->masterRelations().range(ROLE_PART)) {
                if (feature.relative()->attributes().value(ftTypeAttrId)
                    == ftTypeValue) {
                    return SRV_ATTR_TRUE;
                }
            }
        }
    }

    return SRV_ATTR_FALSE;
}

GenericCNT::GenericCNT(
        const std::string& cntCategoryId,
        const std::string& contourCategoryId,
        const std::string& nameAttribute,
        ServiceAttributesRegistry& registry)
    : ServiceAttributesRegistry::Registrar(registry, cntCategoryId)
{
    registerAttr(SRV_RENDER_LABEL, CallbackWrapper<PointObject>(renderLabel))
        .depends(
    {
        {Aspect::Type::Attribute, nameAttribute},
        {{RelationType::Master, ROLE_CENTER, contourCategoryId}}
    });
    registerAttr(SRV_HOTSPOT_LABEL, CallbackWrapper<PointObject>(hotSpotLabel))
        .depends(
    {
        {Aspect::Type::Attribute, nameAttribute},
        {{RelationType::Master, ROLE_CENTER, contourCategoryId}}
    });
    registerAttr(cntCategoryId + SUFFIX_DISPLAY_CLASS, CallbackWrapper<PointObject>(dispclass))
        .depends(
    {
        {Aspect::Type::Attribute, contourCategoryId + SUFFIX_DISPLAY_CLASS},
        {{RelationType::Master, ROLE_CENTER, contourCategoryId}}
    });
    const auto& ftTypeAttrDef = ftTypeAttrDefForCategory(contourCategoryId);
    if (!ftTypeAttrDef) {
        return;
    }
    for (const auto& ftTypeAttrValue : ftTypeAttrDef->values()) {
        auto ftTypeSrvAttrName = SRV_FT_TYPE_ID + "_" + ftTypeAttrValue.value;
        registerAttr(ftTypeSrvAttrName,
                CallbackWrapper<PointObject, std::string>(ftTypeSrvAttrValue, ftTypeAttrValue.value))
            .depends(
        {
            {Aspect::Type::Attribute, ftTypeAttrDef->id()},
            {{RelationType::Master, ROLE_CENTER, contourCategoryId}}
        });
    }
}

std::string GenericCNT::hotSpotLabel(const PointObject* pt, ObjectsCache& cache)
{
    const auto& parents = pt->masterRelations().range(ROLE_CENTER);
    for (const auto& relInfo : parents) {
        return GenericNamedObject::hotspotLabel(relInfo.relative(), cache);
    }
    return s_emptyString;
}

std::string GenericCNT::renderLabel(const PointObject* pt, ObjectsCache& cache)
{
    const auto& parents = pt->masterRelations().range(ROLE_CENTER);
    for (const auto& relInfo : parents) {
        return GenericNamedObject::renderLabel(relInfo.relative(), cache);
    }
    return s_emptyString;
}

std::string GenericCNT::dispclass(const PointObject* pt, ObjectsCache&)
{
    const auto& parents = pt->masterRelations().range(ROLE_CENTER);
    for (const auto& relInfo : parents) {
        return relInfo.relative()->attributes().value(
            relInfo.relative()->categoryId() + SUFFIX_DISPLAY_CLASS);
    }
    return s_emptyString;
}

std::string GenericCNT::ftTypeSrvAttrValue(
    const PointObject* pt,
    ObjectsCache&,
    const std::string& ftTypeValue)
{
    const auto& parents = pt->masterRelations().range(ROLE_CENTER);
    for (const auto& relInfo : parents) {
        if (relInfo.relative()->attributes().value(
                relInfo.relative()->categoryId() + SUFFIX_FT_TYPE_ID)
                    == ftTypeValue) {
            return SRV_ATTR_TRUE;
        }
    }
    return SRV_ATTR_FALSE;
}

GenericNamedObject::GenericNamedObject(const std::string& categoryId,
        ServiceAttributesRegistry& registry)
    :ServiceAttributesRegistry::Registrar(registry, categoryId)
{
    const auto& category = cfg()->editor()->categories()[categoryId];
    registerAttr(SRV_SCREEN_LABEL, CallbackWrapper<GeoObject>(screenLabel));
    if (category.suggest()) {
        registerSuggestCallback(SuggestCallbackWrapper<GeoObject>(suggestTexts));
    }
    if (!category.complex()) {
        registerAttr(SRV_RENDER_LABEL, CallbackWrapper<GeoObject>(renderLabel));
        registerAttr(SRV_HOTSPOT_LABEL, CallbackWrapper<GeoObject>(hotspotLabel));
    }
}

namespace {
std::string
ftTypeLabel(const GeoObject* obj)
{
    const auto& ftTypeDef = ftTypeAttrDefForCategory(obj->categoryId());
    if (!ftTypeDef) {
        return std::string();
    }
    const auto& ftTypeValue = obj->attributes().value(ftTypeDef->id());
    if (ftTypeValue.empty()) {
        return std::string();
    }
    return valueLabel(ftTypeDef->id(), ftTypeValue);
}
}//namespace

std::string
GenericNamedObject::hotspotLabel(const GeoObject* obj, ObjectsCache& cache)
{
    std::string label = objectNameByType(obj->id(), NAME_TYPE_OFFICIAL, cache);
    const auto& typeLabel = ftTypeLabel(obj);
    if (label.empty()) {
        return typeLabel;
    }
    return typeLabel.empty() ? label : label + " (" + typeLabel + ")";
}

std::string
GenericNamedObject::screenLabel(const GeoObject* obj, ObjectsCache& cache)
{
    return objectNameByType(obj->id(), NAME_TYPE_OFFICIAL, cache);
}

SuggestTexts
GenericNamedObject::suggestTexts(const GeoObject* obj, ObjectsCache& cache)
{
    SuggestTexts suggestLangAndText;
    for (const auto& lang : langs(obj->id(), cache)) {
        auto text = objectNameByType(obj->id(), NAME_TYPE_OFFICIAL, cache, lang);
        if (!text.empty()) {
            suggestLangAndText.insert({lang, text});
        }
    }
    return suggestLangAndText;
}

std::string
GenericNamedObject::renderLabel(const GeoObject* obj, ObjectsCache& cache)
{
    auto label = objectNameByType(
        obj->id(), NAME_TYPE_RENDER_LABEL, cache);

    return label.empty()
           ? objectNameByType(
                obj->id(), NAME_TYPE_OFFICIAL, cache)
           : label;
}

GenericPointObject::GenericPointObject(
    const std::string& categoryId,
    ServiceAttributesRegistry& registry)
    :ServiceAttributesRegistry::Registrar(registry, categoryId)
{
    registerSrvHasUrbanAreal();
}

std::string
GenericPointObject::hasUrbanAreal(const GeoObject* obj, ObjectsCache&)
{
    return
        obj->slaveRelations().range(ROLE_URBAN_AREAL_ASSIGNED).empty()
        ? SRV_ATTR_FALSE
        : SRV_ATTR_TRUE;
}

void GenericPointObject::registerSrvHasUrbanAreal()
{
    registerAttr(SRV_HAS_URBAN_AREAL, CallbackWrapper<GeoObject>(hasUrbanAreal))
           .depends(
           {
                {Aspect::Type::Attribute, ""},
                {{RelationType::Slave, ROLE_URBAN_AREAL_ASSIGNED, ""}}
            });

}

GenericServiceAttributes::GenericServiceAttributes(ServiceAttributesRegistry &registry)
{
    GenericJC(CATEGORY_AD_NEUTRAL_JC, CATEGORY_AD_NEUTRAL_EL, registry);

    GenericJC(CATEGORY_VEGETATION_JC, CATEGORY_VEGETATION_EL, registry);
    GenericEL(CATEGORY_VEGETATION_EL, CATEGORY_VEGETATION,
              CATEGORY_VEGETATION_FC, VEGETATION_NM, registry);
    GenericCNT(CATEGORY_VEGETATION_CNT, CATEGORY_VEGETATION, VEGETATION_NM, registry);

    GenericJC(CATEGORY_RELIEF_JC, CATEGORY_RELIEF_EL, registry);
    GenericEL(CATEGORY_RELIEF_EL, CATEGORY_RELIEF, CATEGORY_RELIEF_FC, RELIEF_NM, registry);

    GenericJC(CATEGORY_TRANSPORT_TRAM_JC, CATEGORY_TRANSPORT_TRAM_EL, registry);
    GenericEL(CATEGORY_TRANSPORT_TRAM_EL, CATEGORY_TRANSPORT_TRAM_ROUTE, s_emptyString,
              TRANSPORT_NM, registry);

    GenericJC(CATEGORY_TRANSPORT_WATERWAY_JC, CATEGORY_TRANSPORT_WATERWAY_EL, registry);
    GenericEL(CATEGORY_TRANSPORT_WATERWAY_EL, CATEGORY_TRANSPORT_WATERWAY_ROUTE,
              s_emptyString, TRANSPORT_WATERWAY_NM, registry);

    for (const auto& category : GENERIC_NAMED_CATEGORIES){
        GenericNamedObject(category, registry);
    }

    for (const auto& category : GENERIC_POINT_CATEGORIES){
        GenericPointObject(category, registry);
    }

    //OBSOLETE CONTOUR CATEGOEIES
    GenericJC(CATEGORY_URBAN_JC, CATEGORY_URBAN_EL, registry);
    GenericEL(CATEGORY_URBAN_EL, CATEGORY_URBAN, CATEGORY_URBAN_FC, URBAN_NM, registry);
    GenericJC(CATEGORY_URBAN_ROADNET_JC, CATEGORY_URBAN_ROADNET_EL, registry);
    GenericEL(CATEGORY_URBAN_ROADNET_EL, CATEGORY_URBAN, CATEGORY_URBAN_ROADNET_FC, URBAN_ROADNET_NM, registry);
    GenericJC(CATEGORY_URBAN_ROADNET_FENCE_JC, CATEGORY_URBAN_ROADNET_FENCE_EL, registry);
}

} // namespace srv_attrs
} // namespace wiki
} // namespace maps
