#include "hydro.h"
#include "generic.h"
#include "calc.h"
#include "registry.h"
#include <maps/wikimap/mapspro/services/editor/src/objects/attr_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/areal_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/point_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/complex_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/relation_object.h>
#include <maps/wikimap/mapspro/services/editor/src/edit_options.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>

namespace maps
{
namespace wiki
{
namespace srv_attrs
{

class HydroFCEL : public ServiceAttributesRegistry::Registrar
{
public:
    explicit HydroFCEL(ServiceAttributesRegistry& registry)
        : ServiceAttributesRegistry::Registrar(registry, CATEGORY_HYDRO_FC_EL)
    {
        registerAttr(SRV_IS_INTERIOR, CallbackWrapper<LinearElement, std::string>(GenericEL::isPartOfInterior, CATEGORY_HYDRO_FC))
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_HYDRO_FC_IS_INTERIOR},
            {
                {RelationType::Master, ROLE_PART, CATEGORY_HYDRO_FC}
            }
        });
        registerAttr(SRV_SCREEN_LABEL,
            CallbackWrapper<LinearElement>(screenLabel))
            .depends(
            {
                {Aspect::Type::Attribute, HYDRO_NM},
                {
                    {RelationType::Master, ROLE_PART, CATEGORY_HYDRO_FC},
                    {RelationType::Master, ROLE_FC_PART, CATEGORY_HYDRO}
                }
            });
        for (const auto& value : cfg()->editor()->attribute(ATTR_HYDRO_FT_TYPE_ID)->values()) {
            registerAttr(SRV_FT_TYPE_ID + "_" + value.value,
            CallbackWrapper<LinearElement, std::string>(hasFtType, value.value))
            .depends(
            {
                {Aspect::Type::Attribute, ATTR_HYDRO_FT_TYPE_ID},
                {
                    {RelationType::Master, ROLE_PART, CATEGORY_HYDRO_FC},
                    {RelationType::Master, ROLE_FC_PART, CATEGORY_HYDRO}
                }
            });
        }
    }

private:
    static std::string screenLabel(const LinearElement* el, ObjectsCache& cache)
    {
        TOIds sources;
        for (const auto& contour : el->masterRelations().range(ROLE_PART)) {
            for (const auto& feature :
                contour.relative()->masterRelations().range(ROLE_FC_PART)) {
                sources.insert(feature.id());
            }
        }
        if (sources.empty()) {
            return s_emptyString;
        }
        size_t line = 0;
        std::stringstream screenLabelStream;
        for (const auto& oid : sources) {
            auto name = objectNameByType(oid,
                NAME_TYPE_OFFICIAL,
                cache);
            if (name.empty()) {
                continue;
            }
            if (line) {
                screenLabelStream << ", ";
            }
            screenLabelStream << name;
            ++line;
        }
        return screenLabelStream.str();
    }

    static std::string hasFtType(
            const LinearElement* el,
            ObjectsCache&/* cache*/,
            const std::string& ftType)
    {
        for (const auto& contour : el->masterRelations().range(ROLE_PART)) {
            for (const auto& feature :
                contour.relative()->masterRelations().range(ROLE_FC_PART)) {
                if (feature.relative()->attributes().value(ATTR_HYDRO_FT_TYPE_ID)
                    == ftType) {
                    return SRV_ATTR_TRUE;
                }
            }
        }
        return SRV_ATTR_FALSE;
    }
};

class HydroLNEL : public ServiceAttributesRegistry::Registrar
{
public:
    explicit HydroLNEL(ServiceAttributesRegistry& registry)
        : ServiceAttributesRegistry::Registrar(registry, CATEGORY_HYDRO_LN_EL)
    {
        registerAttr(SRV_SCREEN_LABEL,
            CallbackWrapper<LinearElement, std::string>(label, NAME_TYPE_OFFICIAL))
            .depends(
            {
                {Aspect::Type::Attribute, HYDRO_NM},
                {{RelationType::Master, ROLE_LN_PART, CATEGORY_HYDRO_LN}}
            });
        registerAttr(SRV_RENDER_LABEL,
            CallbackWrapper<LinearElement, std::string>(label, NAME_TYPE_RENDER_LABEL))
            .depends(
            {
                {Aspect::Type::Attribute, HYDRO_NM},
                {{RelationType::Master, ROLE_LN_PART, CATEGORY_HYDRO_LN}}
            });
        registerAttr(SRV_HOTSPOT_LABEL,
            CallbackWrapper<LinearElement>(hotspotLabel))
            .depends(
            {
                {Aspect::Type::Attribute, HYDRO_NM},
                {{RelationType::Master, ROLE_LN_PART, CATEGORY_HYDRO_LN}}
            });

        for (const auto& value : cfg()->editor()->attribute(ATTR_HYDRO_LN_FT_TYPE_ID)->values()) {
            registerAttr(SRV_FT_TYPE_ID + "_" + value.value,
            CallbackWrapper<LinearElement, std::string>(hasFtType, value.value))
            .depends(
            {
                {Aspect::Type::Attribute, ATTR_HYDRO_LN_FT_TYPE_ID},
                {{RelationType::Master, ROLE_LN_PART, CATEGORY_HYDRO_LN}}
            });
        }
    }

private:
    static std::string label(const LinearElement* el, ObjectsCache& cache, const std::string& nameType)
    {
        auto master = el->masterRelations().range(ROLE_LN_PART);
        if (master.empty()) {
            return s_emptyString;
        };
        return objectNameByType(master.begin()->id(),
                nameType,
                cache);
    }

    static std::string hotspotLabel(const LinearElement* el, ObjectsCache& cache)
    {
        auto masters = el->masterRelations().range(ROLE_LN_PART);
        if (masters.size() < 2) {
            return s_emptyString;
        }
        std::set<std::string> names;
        for (const auto& master : masters) {
            names.insert(objectNameByType(master.id(),
                NAME_TYPE_OFFICIAL,
                cache));
        }
        return common::join(names, ", ");
    }

    static std::string hasFtType(const LinearElement* el,
            ObjectsCache&/* cache*/,
            const std::string& ftType)
    {
        auto master = el->masterRelations().range(ROLE_LN_PART);
        if (master.empty()) {
            return s_emptyString;
        };
        return master.begin()->relative()->attributes().value(ATTR_HYDRO_LN_FT_TYPE_ID)
            == ftType
            ? SRV_ATTR_TRUE
            : SRV_ATTR_FALSE;
    }
};

HydroServiceAttributes::HydroServiceAttributes(ServiceAttributesRegistry &registry)
{
    HydroLNEL lnEl(registry);
    HydroFCEL fcEl(registry);
    GenericJC(CATEGORY_HYDRO_LN_JC, CATEGORY_HYDRO_LN_EL, registry);
    GenericJC(CATEGORY_HYDRO_FC_JC, CATEGORY_HYDRO_FC_EL, registry);
    GenericNamedObject(CATEGORY_HYDRO, registry);
    GenericNamedObject(CATEGORY_HYDRO_LN, registry);
    GenericNamedObject(CATEGORY_HYDRO_POINT, registry);
}

}//srv_attrs
}//wiki
}//maps
