#include "indoor.h"
#include "calc.h"
#include "generic.h"
#include <maps/wikimap/mapspro/services/editor/src/objects/attr_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/areal_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/line_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/point_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/complex_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/relation_object.h>
#include <maps/wikimap/mapspro/services/editor/src/edit_options.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>
#include "registry.h"

namespace maps
{
namespace wiki
{
namespace srv_attrs
{

const std::vector<std::string> INDOOR_POI_CATEGORIES =
{
    CATEGORY_INDOOR_POI_AUTO,
    CATEGORY_INDOOR_POI_FOOD,
    CATEGORY_INDOOR_POI_INFO,
    CATEGORY_INDOOR_POI_INFRA,
    CATEGORY_INDOOR_POI_SERVICE,
    CATEGORY_INDOOR_POI_FINANCE,
    CATEGORY_INDOOR_POI_MEDICINE,
    CATEGORY_INDOOR_POI_EDU,
    CATEGORY_INDOOR_POI_LEISURE,
    CATEGORY_INDOOR_POI_SPORT,
    CATEGORY_INDOOR_POI_GOVERMENT,
    CATEGORY_INDOOR_POI_SHOPPING,
    CATEGORY_INDOOR_POI_RELIGION,
};

class INDOOR_PLAN : public ServiceAttributesRegistry::Registrar
{
public:
    INDOOR_PLAN(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_INDOOR_PLAN)
    {
        registerAttr(SRV_HAS_INDOOR_RADIOMAP_CAPTURER_PATHS, CallbackWrapper<PointObject>(hasIndoorRadiomapCapturerPaths))
            .depends(
        {
            {Aspect::Type::Relations, STR_TO_MASTER},
            {
                {RelationType::Slave, ROLE_ASSIGNED, CATEGORY_INDOOR_LEVEL},
                {RelationType::Slave, ROLE_ASSIGNED_INDOOR_RADIOMAP_CAPTURER_PATH, CATEGORY_INDOOR_RADIOMAP_CAPTURER_PATH}
            }
        });
    }

private:
    static std::string hasIndoorRadiomapCapturerPaths(const PointObject* pt, ObjectsCache& )
    {
        const auto& indoorLevels = pt->slaveRelations().range(ROLE_ASSIGNED);
        for (const auto& levelRel : indoorLevels) {
            if (!levelRel.relative()->slaveRelations()
                .range(ROLE_ASSIGNED_INDOOR_RADIOMAP_CAPTURER_PATH).empty())
            {
                return SRV_ATTR_TRUE;
            }
        }
        return SRV_ATTR_FALSE;
    }
};

class INDOOR_LEVEL : public ServiceAttributesRegistry::Registrar
{
public:
    INDOOR_LEVEL(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_INDOOR_LEVEL)
    {
        registerAttr(SRV_INDOOR_LEVEL_ID, CallbackWrapper<ArealObject>(indoorLevelIdSrvAttrValue));
    }

private:
    static std::string indoorLevelIdSrvAttrValue(const ArealObject* pt, ObjectsCache& )
    {
        return std::to_string(pt->id());
    }
};

class INDOOR_BARRIER : public ServiceAttributesRegistry::Registrar
{
public:
    INDOOR_BARRIER(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_INDOOR_BARRIER)
    {
        registerAttr(SRV_INDOOR_LEVEL_ID, CallbackWrapper<LineObject>(indoorLevelIdSrvAttrValue));
    }

private:
    static std::string indoorLevelIdSrvAttrValue(const LineObject* lo, ObjectsCache& )
    {
        const auto& indoorLevels = lo->masterRelations().range(ROLE_ASSIGNED_INDOOR_BARRIER);
        return indoorLevels.empty() ? s_emptyString : std::to_string(indoorLevels.begin()->id());
        /*
        if (indoorLevels.empty()) {
            return s_emptyString;
        }
        return std::to_string(indoorLevels.begin()->id());*/
    }
};

class INDOOR_RADIOMAP_CAPTURER_PATH : public ServiceAttributesRegistry::Registrar
{
public:
    INDOOR_RADIOMAP_CAPTURER_PATH(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_INDOOR_RADIOMAP_CAPTURER_PATH)
    {
        registerAttr(SRV_INDOOR_LEVEL_ID, CallbackWrapper<LineObject>(indoorLevelIdSrvAttrValue));
        registerAttr(SRV_SCREEN_LABEL, CallbackWrapper<LineObject>(screenLabel));
        registerAttr(SRV_RENDER_LABEL, CallbackWrapper<LineObject>(screenLabel));
    }

private:
    static std::string indoorLevelIdSrvAttrValue(const LineObject* lo, ObjectsCache& )
    {
        const auto& indoorLevels = lo->masterRelations().range(ROLE_ASSIGNED_INDOOR_RADIOMAP_CAPTURER_PATH);
        return indoorLevels.empty() ? s_emptyString : std::to_string(indoorLevels.begin()->id());
    }

    static std::string screenLabel(const LineObject* lo, ObjectsCache& )
    {
        return lo->attributes().value(ATTR_INDOOR_RADIOMAP_CAPTURER_PATH_DESCRIPTION);
    }
};

class INDOOR_AREA : public ServiceAttributesRegistry::Registrar
{
public:
    INDOOR_AREA(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_INDOOR_AREA)
    {
        registerAttr(SRV_INDOOR_LEVEL_ID, CallbackWrapper<ArealObject>(indoorLevelIdSrvAttrValue));
    }

private:
    static std::string indoorLevelIdSrvAttrValue(const ArealObject* ao, ObjectsCache& )
    {
        const auto& indoorLevels = ao->masterRelations().range(ROLE_ASSIGNED_INDOOR_AREA);
        return indoorLevels.empty() ? s_emptyString : std::to_string(indoorLevels.begin()->id());
/*
        if (indoorLevels.empty()) {
            return s_emptyString;
        }
        return std::to_string(indoorLevels.begin()->id());*/
    }
};

class INDOOR_POI : public ServiceAttributesRegistry::Registrar
{
public:
    INDOOR_POI(ServiceAttributesRegistry& registry, const std::string& categoryId)
    :ServiceAttributesRegistry::Registrar(registry, categoryId)
    {
        registerAttr(SRV_INDOOR_LEVEL_ID, CallbackWrapper<PointObject>(indoorLevelIdSrvAttrValue));
    }

private:
    static std::string indoorLevelIdSrvAttrValue(const PointObject* po, ObjectsCache& )
    {
        const std::string& categoryId = po->categoryId();
        const auto& indoorLevels = po->masterRelations().range("assigned_" + categoryId);
        return indoorLevels.empty() ? s_emptyString : std::to_string(indoorLevels.begin()->id());
        /*
        if (indoorLevels.empty()) {
            return s_emptyString;
        }
        return std::to_string(indoorLevels.begin()->id());*/
    }
};

INDOORServiceAttributes::INDOORServiceAttributes(ServiceAttributesRegistry& registry)
{
    INDOOR_PLAN plan(registry);
    INDOOR_LEVEL level(registry);
    INDOOR_BARRIER barrier(registry);
    INDOOR_RADIOMAP_CAPTURER_PATH imrcPath(registry);
    INDOOR_AREA area(registry);

    for (const auto& category : INDOOR_POI_CATEGORIES) {
        INDOOR_POI poi(registry, category);
    }
}

}//srv_attrs
}//wiki
}//maps
