#include "metro.h"
#include "calc.h"
#include "generic.h"
#include <maps/wikimap/mapspro/services/editor/src/objects/attr_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/areal_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/point_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/complex_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/relation_object.h>
#include <maps/wikimap/mapspro/services/editor/src/edit_options.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>
#include <yandex/maps/wiki/configs/editor/categories.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include "registry.h"

namespace maps
{
namespace wiki
{
namespace srv_attrs
{

typedef std::multimap<std::string, TOid> LevelKindToId;

class METRO_JC : public ServiceAttributesRegistry::Registrar
{
public:
    METRO_JC(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_TRANSPORT_METRO_JC)
    {
        registerAttr(
            CATEGORY_TRANSPORT_METRO_JC + SUFFIX_CAN_DELETE,
            CallbackWrapper<Junction>(canDelete))
            .depends(
        {
            {Aspect::Type::Attribute, ""},
            {{RelationType::Master, ROLE_START, CATEGORY_TRANSPORT_METRO_EL}}
        })
            .depends(
        {
            {Aspect::Type::Attribute, ""},
            {{RelationType::Master, ROLE_END, CATEGORY_TRANSPORT_METRO_EL}}
        });
        registerAttr(SRV_VALENCY, CallbackWrapper<Junction>(valency))
            .depends(
        {
            {Aspect::Type::Relations, STR_TO_ATTR_OWNER},
            {{RelationType::Master, ROLE_START, CATEGORY_TRANSPORT_METRO_EL}}
        })
            .depends(
        {
            {Aspect::Type::Relations, STR_TO_ATTR_OWNER},
            {{RelationType::Master, ROLE_END, CATEGORY_TRANSPORT_METRO_EL}}
        });
    }

private:
    static std::string canDelete(const Junction* jc, ObjectsCache&)
    {
        return maps::wiki::canDelete(jc, 0)
        ? SRV_ATTR_TRUE
        : s_emptyString;
    }
    static std::string valency(const Junction* jc, ObjectsCache&)
    {
        return boost::lexical_cast<std::string>(jc->valency());
    }
};

class METRO_EL : public ServiceAttributesRegistry::Registrar
{
public:
    METRO_EL(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_TRANSPORT_METRO_EL)
    {
        registerAttr(SRV_SCREEN_LABEL, CallbackWrapper<LinearElement, std::string>(uiLabel, NAME_TYPE_OFFICIAL))
            .depends(
        {
            {Aspect::Type::Attribute, TRANSPORT_METRO_NM},
            {{RelationType::Master, ROLE_PART, CATEGORY_TRANSPORT_METRO_LINE}}
        });
        registerAttr(SRV_RENDER_LABEL, CallbackWrapper<LinearElement, std::string>(uiLabel, NAME_TYPE_RENDER_LABEL))
            .depends(
        {
            {Aspect::Type::Attribute, TRANSPORT_METRO_NM},
            {{RelationType::Master, ROLE_PART, CATEGORY_TRANSPORT_METRO_LINE}}
        });
        GenericEL::registerFtTypes(*this, GenericEL::DependenceType::Direct, CATEGORY_TRANSPORT_METRO_LINE, std::string());
    }

private:
    static std::string uiLabel(const LinearElement* el, ObjectsCache& cache, const std::string& nameType)
    {
        const auto& metroLines = el->masterRelations().range(ROLE_PART);
        for (const auto& metroLine : metroLines) {
            std::string lineName = objectNameByType(
                    metroLine.id(),
                    nameType,
                    cache);
            if (!lineName.empty() ) {
                return lineName;
            }
        }
        return s_emptyString;
    }
};

class METRO_EXIT : public ServiceAttributesRegistry::Registrar
{
public:
    METRO_EXIT(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_TRANSPORT_METRO_EXIT)
    {
        registerSuggestCallback(SuggestCallbackWrapper<PointObject>(GenericNamedObject::suggestTexts));
        registerAttr(SRV_SCREEN_LABEL, CallbackWrapper<PointObject>(screenLabel));
        registerAttr(SRV_RENDER_LABEL, CallbackWrapper<PointObject>(renderLabel));
        registerAttr(SRV_HOTSPOT_LABEL, CallbackWrapper<PointObject>(hotspotLabel))
            .depends(
        {
            {Aspect::Type::Attribute, TRANSPORT_METRO_NM},
            {{RelationType::Master, ROLE_ASSIGNED, CATEGORY_TRANSPORT_METRO_STATION}}
        });
    }

private:
    static std::string screenLabel(const PointObject* pt, ObjectsCache& cache)
    {
        return objectNameByType(
                pt->id(),
                NAME_TYPE_OFFICIAL,
                cache);
    }

    static std::string hotspotLabel(const PointObject* pt, ObjectsCache& cache)
    {
        const auto& metroStations = pt->masterRelations().range(ROLE_ASSIGNED);
        std::string exitName = objectNameByType(
                pt->id(),
                NAME_TYPE_OFFICIAL,
                cache);
        for (const auto& metroStation : metroStations) {
            std::string metroStationName = objectNameByType(
                    metroStation.id(),
                    NAME_TYPE_OFFICIAL,
                    cache);
            if (!metroStationName.empty()) {
                return exitName + " (" + metroStationName + ")";
            }
        }
        return exitName;
    }

    static std::string renderLabel(const PointObject* pt, ObjectsCache& cache)
    {
        return objectNameByType(pt->id(),
                NAME_TYPE_RENDER_LABEL,
                cache);
    }
};

class METRO_STATION : public ServiceAttributesRegistry::Registrar
{
public:
    METRO_STATION(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_TRANSPORT_METRO_STATION)
    {
        registerSuggestCallback(SuggestCallbackWrapper<PointObject>(GenericNamedObject::suggestTexts));
        registerAttr(SRV_SCREEN_LABEL, CallbackWrapper<PointObject>(screenLabel));
        registerAttr(SRV_RENDER_LABEL, CallbackWrapper<PointObject>(renderLabel));
        registerAttr(SRV_HOTSPOT_LABEL, CallbackWrapper<PointObject>(hotspotLabel))
            .depends(
        {
            {Aspect::Type::Attribute, TRANSPORT_METRO_NM},
            {{RelationType::Master, ROLE_ASSIGNED, CATEGORY_TRANSPORT_METRO_LINE}}
        });
        registerAttr(SRV_FT_TYPE_ID, CallbackWrapper<PointObject>(ftTypeSrvAttrValue))
            .depends(
        {
            {Aspect::Type::Attribute, ATTR_TRANSPORT_METRO_LINE_FT_TYPE_ID},
            {{RelationType::Master, ROLE_ASSIGNED, CATEGORY_TRANSPORT_METRO_LINE}}
        });
    }

private:
    static std::string screenLabel(const PointObject* pt, ObjectsCache& cache)
    {
        return objectNameByType(
                pt->id(),
                NAME_TYPE_OFFICIAL,
                cache);
    }

    static std::string hotspotLabel(const PointObject* pt, ObjectsCache& cache)
    {
        std::string label = objectNameByType(
                            pt->id(),
                            NAME_TYPE_OFFICIAL,
                            cache);
        const auto& metroLines = pt->masterRelations().range(ROLE_ASSIGNED);
        for (const auto& metroLine : metroLines) {
            std::string metroLineName = objectNameByType(
                    metroLine.id(),
                    NAME_TYPE_OFFICIAL,
                    cache);
            if (!metroLineName.empty()) {
                return label + " (" + metroLineName + ")";
            }
        }
        return label;
    }

    static std::string renderLabel(const PointObject* pt, ObjectsCache& cache)
    {
        return objectNameByType(pt->id(),
                NAME_TYPE_RENDER_LABEL,
                cache);
    }
    static std::string ftTypeSrvAttrValue(const PointObject* pt, ObjectsCache& /*cache*/)
    {
        for ( const auto& masterRel : pt->masterRelations().range()) {
            return masterRel.relative()->attributes().value(ATTR_TRANSPORT_METRO_LINE_FT_TYPE_ID);
        }
        return {};
    }
};

class METRO_LINE : public ServiceAttributesRegistry::Registrar
{
public:
    METRO_LINE(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_TRANSPORT_METRO_LINE)
    {
        registerSuggestCallback(SuggestCallbackWrapper<ComplexObject>(GenericNamedObject::suggestTexts));
        registerAttr(SRV_SCREEN_LABEL, CallbackWrapper<ComplexObject>(screenLabel));
    }
    static std::string screenLabel(const ComplexObject* obj, ObjectsCache& cache)
    {
        return objectNameByType(
                obj->id(),
                NAME_TYPE_OFFICIAL,
                cache);
    }
};

namespace {
const GeoObject* getSingleMaster(const GeoObject* object, const std::string& role)
{
    for (const auto& rel : object->masterRelations().range(role)) {
        return rel.relative();
    }
    return nullptr;
}

const GeoObject* getSingleSlave(const GeoObject* object, const std::string& role)
{
    for (const auto& rel : object->slaveRelations().range(role)) {
        return rel.relative();
    }
    return nullptr;
}

std::string objectTitle(const GeoObject* obj, ObjectsCache& cache)
{
    auto name = objectNameByType(obj->id(), NAME_TYPE_OFFICIAL, cache);
    return name.empty()
        ? obj->category().label()
        : name;
}

std::string
titleForBoardings(const std::vector<const GeoObject*> stations, const GeoObject* boarding, ObjectsCache& cache)
{
    std::vector<std::string> stationNames;
    for (const auto& station : stations) {
        stationNames.push_back(objectTitle(station, cache));
    }

    auto addLineName = [&](const GeoObject* station, const std::string& stationName) {
        const auto* line = getSingleMaster(station, ROLE_ASSIGNED);
        if (!line || line->categoryId() != CATEGORY_TRANSPORT_METRO_LINE) {
            return stationName + " (" + station->category().label() + ")";
        }
        return stationName + " (" + objectTitle(line, cache) + ")";
    };

    for (size_t stationIdx = 0; stationIdx < stationNames.size() - 1; ++stationIdx) {
        for (size_t nextStationIdx = stationIdx + 1; nextStationIdx < stationNames.size(); ++nextStationIdx) {
            if (stationNames[stationIdx] != stationNames[nextStationIdx]) {
                continue;
            }
            stationNames[stationIdx] =
                addLineName(stations[stationIdx], stationNames[stationIdx]);
            stationNames[nextStationIdx] =
                addLineName(stations[nextStationIdx], stationNames[nextStationIdx]);

        }
    }
    std::string title = common::join(stationNames, NICE_DASH);
    if (!boarding->attributes().isDefined(ATTR_TRANSPORT_BOARDING_OPTIONS)) {
        return title;
    }
    const auto options = boarding->attributes().values(ATTR_TRANSPORT_BOARDING_OPTIONS);
    const auto& attrDef = boarding->attributes().find(ATTR_TRANSPORT_BOARDING_OPTIONS)->def();
    std::vector<std::string> originalOrderOptionsLabels;
    originalOrderOptionsLabels.reserve(options.size());
    for (const auto& valueDef : attrDef.values()) {
        if (options.contains(valueDef.value)) {
            originalOrderOptionsLabels.push_back(valueDef.label);
        }
    }
    if (!originalOrderOptionsLabels.empty()) {
        title += " (" + common::join(originalOrderOptionsLabels, ", ") +")";
    }
    return title;
}
} // namespace

class METRO_TRANSITION_BOARDING : public ServiceAttributesRegistry::Registrar
{
    public:
    METRO_TRANSITION_BOARDING(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_TRANSPORT_TRANSITION_BOARDING)
    {
        registerAttr(SRV_SCREEN_LABEL, CallbackWrapper<ComplexObject>(screenLabel))
            .depends(
        {
            {Aspect::Type::Attribute, TRANSPORT_METRO_NM},
            {
                {RelationType::Master, ROLE_BOARDING, CATEGORY_TRANSPORT_TRANSITION},
                {RelationType::Slave, ROLE_STATION_A, CATEGORY_TRANSPORT_METRO_STATION}
            }
        })
            .depends(
        {
            {Aspect::Type::Attribute, TRANSPORT_METRO_NM},
            {
                {RelationType::Master, ROLE_BOARDING, CATEGORY_TRANSPORT_TRANSITION},
                {RelationType::Slave, ROLE_STATION_B, CATEGORY_TRANSPORT_METRO_STATION}
            }
        })
            .depends(
        {
            {Aspect::Type::Attribute, TRANSPORT_METRO_NM},
            {
                {RelationType::Slave, ROLE_STATION_PREVIOUS, CATEGORY_TRANSPORT_METRO_STATION}
            }
        })
            .depends(
        {
            {Aspect::Type::Attribute, TRANSPORT_METRO_NM},
            {
                {RelationType::Slave, ROLE_STATION_NEXT, CATEGORY_TRANSPORT_METRO_STATION}
            }
        });

    }

    static std::string screenLabel(const ComplexObject* obj, ObjectsCache& cache)
    {
        const auto* stationPrev = getSingleSlave(obj, ROLE_STATION_PREVIOUS);
        const auto* transition = getSingleMaster(obj, ROLE_BOARDING);
        if (!transition) {
            return {};
        }
        const auto* stationA = getSingleSlave(transition, ROLE_STATION_A);
        const auto* stationB = getSingleSlave(transition, ROLE_STATION_B);
        const auto* stationNext = getSingleSlave(obj, ROLE_STATION_NEXT);
        if (obj->attributes().value(ATTR_TRANSPORT_TRANSITION_BOARDING_ONEWAY) == "T") {
            std::swap(stationA, stationB);
        }
        std::vector<const GeoObject*> stations;
        if (stationPrev) {
            stations.push_back(stationPrev);
        }
        if (stationA) {
            stations.push_back(stationA);
        }
        if (stationB) {
            stations.push_back(stationB);
        }
        if (stationNext) {
            stations.push_back(stationNext);
        }
        return titleForBoardings(stations, obj, cache);
    }
};

class METRO_TRANSPORT_PASSAGEWAY_BOARDING : public ServiceAttributesRegistry::Registrar
{
    public:
    METRO_TRANSPORT_PASSAGEWAY_BOARDING(ServiceAttributesRegistry& registry):ServiceAttributesRegistry::Registrar(registry, CATEGORY_TRANSPORT_PASSAGEWAY_BOARDING)
    {
        registerAttr(SRV_SCREEN_LABEL, CallbackWrapper<ComplexObject>(screenLabel))
            .depends(
        {
            {Aspect::Type::Attribute, TRANSPORT_METRO_NM},
            {
                {RelationType::Master, ROLE_BOARDING, CATEGORY_TRANSPORT_PASSAGEWAY},
                {RelationType::Slave, ROLE_STATION, CATEGORY_TRANSPORT_METRO_STATION}
            }
        })
            .depends(
        {
            {Aspect::Type::Attribute, TRANSPORT_METRO_NM},
            {
                {RelationType::Master, ROLE_BOARDING, CATEGORY_TRANSPORT_PASSAGEWAY},
                {RelationType::Slave, ROLE_EXIT, CATEGORY_TRANSPORT_METRO_EXIT}
            }
        })
            .depends(
        {
            {Aspect::Type::Attribute, TRANSPORT_METRO_NM},
            {
                {RelationType::Slave, ROLE_STATION_PREVIOUS, CATEGORY_TRANSPORT_METRO_STATION}
            }
        });
    }

    static std::string screenLabel(const ComplexObject* obj, ObjectsCache& cache)
    {
        const auto* stationPrev = getSingleSlave(obj, ROLE_STATION_PREVIOUS);
        const auto* passageway = getSingleMaster(obj, ROLE_BOARDING);
        if (!passageway) {
            return {};
        }
        const auto* station = getSingleSlave(passageway, ROLE_STATION);
        const auto* exit = getSingleSlave(passageway, ROLE_EXIT);
        std::vector<const GeoObject*> stations;
        if (stationPrev) {
            stations.push_back(stationPrev);
        }
        if (station) {
            stations.push_back(station);
        }
        if (exit) {
            stations.push_back(exit);
        }
        return titleForBoardings(stations, obj, cache);
    }
};

METROServiceAttributes::METROServiceAttributes(ServiceAttributesRegistry& registry)
{
    METRO_LINE line(registry);
    METRO_EXIT exit(registry);
    METRO_STATION station(registry);
    METRO_EL el(registry);
    METRO_JC jc(registry);
    METRO_TRANSITION_BOARDING transitionBoarding(registry);
    METRO_TRANSPORT_PASSAGEWAY_BOARDING passagewayBaording(registry);
    GenericNamedObject opeator(CATEGORY_TRANSPORT_METRO_OPERATOR, registry);
}
}//srv_attrs
}//wiki
}//maps
