#pragma once
#include "relations_path.h"
#include <maps/wikimap/mapspro/services/editor/src/objects/object.h>
#include <functional>

namespace maps
{
namespace wiki
{

class GeoObjectCollection;
class EditorCfg;

namespace srv_attrs
{

/*****************************************************************************/
/*                    S E R V I C E   A T T R I B U T E S                    */
/*****************************************************************************/
const std::string SRV_FT_TYPE_ID = "srv:ft_type_id";
const std::string SRV_HAS_FLAT_RANGES = "srv:has_flat_ranges"; /**< Shows if an address (addr) is connected to one/several flat ranges (flat_range). */
const std::string SRV_HAS_INDOOR_RADIOMAP_CAPTURER_PATHS = "srv:has_indoor_radiomap_capturer_paths";
const std::string SRV_HAS_MODEL3D = "srv:has_model3d";
const std::string SRV_HAS_URBAN_AREAL = "srv:has_urban_areal";
const std::string SRV_HAS_ZIPCODE = "srv:has_zipcode"; /**< Shows if an address (addr) is assigned to a zipcode. */
const std::string SRV_HOTSPOT_LABEL = "srv:hotspot_label";
const std::string SRV_INVALID = "srv:invalid";
const std::string SRV_IS_ASSIGNED_TO_ZONE = "srv:is_assigned_to_zone"; /**< Shows if a point or a liniar parking is connected to a controlled parking zone. */
const std::string SRV_IS_INTERIOR = "srv:is_interior";
const std::string SRV_PRIMARY_POINT_CAT = "srv:primary_point_cat";
const std::string SRV_RENDER_LABEL = "srv:render_label";
const std::string SRV_SCREEN_LABEL = "srv:screen_label";
const std::string SRV_SUBSTITUTION = "srv:is_involve_in_ad_subst";
const std::string SRV_VALENCY = "srv:valency";
const std::string SRV_INDOOR_LEVEL_ID = "srv:indoor_level_id";
const std::string SRV_ROAD_SURFACE_ID = "srv:road_surface_id";
const std::string RD_EL_IS_PART_OF_ROAD = "rd_el:is_part_of_road";
const std::string RD_EL_IS_PART_OF_SPORT_TRACK = "rd_el:is_part_of_sport_track";
const std::string RD_EL_IS_PART_OF_BUS_THREAD = "rd_el:is_part_of_bus_thread";
const std::string RD_EL_IS_PART_OF_BUS_THREAD_CONNECTOR = "rd_el:is_part_of_bus_thread_connector";
const std::string RD_EL_SPORT_TRACK_FT_TYPE_IDS = "rd_el:sport_track_ft_type_ids";
const std::string RD_EL_VEHICLE_RESTRICTION_UNIVERSAL_IDS = "rd_el:vehicle_restriction_universal_ids";
const std::string RD_EL_VEHICLE_RESTRICTION_UNIVERSAL_IDS_B = "rd_el:vehicle_restriction_universal_ids_b";
const std::string RD_EL_VEHICLE_RESTRICTION_UNIVERSAL_IDS_T = "rd_el:vehicle_restriction_universal_ids_t";
const std::string RD_EL_VEHICLE_RESTRICTION_UNIVERSAL_IDS_F = "rd_el:vehicle_restriction_universal_ids_f";
const std::string RD_JC_IS_ALL_COND_VALID = "rd_jc:is_all_cond_valid";
const std::string RD_EL_IS_INVOLVE_IN_COND_CLOSURE = "rd_el:is_involve_in_cond_closure";
const std::string RD_JC_IS_INVOLVE_IN_COND = "rd_jc:is_involve_in_cond";
const std::string RD_JC_IS_INVOLVE_IN_COND_ANNOTATION = "rd_jc:is_involve_in_cond_annotation";
const std::string RD_JC_IS_INVOLVE_IN_COND_BORDER_CHECKPOINT = "rd_jc:is_involve_in_cond_border_checkpoint";
const std::string RD_JC_IS_INVOLVE_IN_COND_CAM = "rd_jc:is_involve_in_cond_cam";
const std::string RD_JC_IS_INVOLVE_IN_COND_CLOSURE = "rd_jc:is_involve_in_cond_closure";
const std::string RD_JC_IS_INVOLVE_IN_COND_DS = "rd_jc:is_involve_in_cond_ds";
const std::string RD_JC_IS_INVOLVE_IN_COND_LANE = "rd_jc:is_involve_in_cond_lane";
const std::string RD_JC_IS_INVOLVE_IN_COND_RAILWAY_CROSSING = "rd_jc:is_involve_in_cond_railway_crossing";
const std::string RD_JC_IS_INVOLVE_IN_COND_SPEED_BUMP = "rd_jc:is_involve_in_cond_speed_bump";
const std::string RD_JC_IS_INVOLVE_IN_COND_TOLL = "rd_jc:is_involve_in_cond_toll";
const std::string RD_JC_IS_INVOLVE_IN_COND_TRAFFIC_LIGHT = "rd_jc:is_involve_in_cond_traffic_light";
const std::string RD_JC_IS_INVOLVE_IN_COND_WITH_COND_DT = "rd_jc:is_involve_in_cond_with_cond_dt";

const std::string RD_JC_IS_INVOLVE_IN_COND_RESTRICTED = "rd_jc:is_involve_in_cond_restricted";
const std::string RD_JC_IS_INVOLVE_IN_COND_RESTRICTED_TRUCK = "rd_jc:is_involve_in_cond_restricted_truck";
const std::string RD_JC_IS_INVOLVE_IN_COND_RESTRICTED_V_R = "rd_jc:is_involve_in_cond_restricted_v_r";
const std::string RD_JC_IS_INVOLVE_IN_COND_RESTRICTED_TRUCK_V_R = "rd_jc:is_involve_in_cond_restricted_truck_v_r";
const std::string RD_JC_IS_INVOLVE_IN_COND_PASS_TRUCK = "rd_jc:is_involve_in_cond_pass_truck";
const std::string RD_JC_IS_INVOLVE_IN_COND_PASS_TRUCK_V_R = "rd_jc:is_involve_in_cond_pass_truck_v_r";

const std::string ADDR_IS_RELATED_TO_ROAD_WITHOUT_GEOMETRY = "addr:is_related_to_road_without_geometry";


const std::set<std::string> ALL_RD_JC_IS_INVOLVE_IN_COND {
    RD_JC_IS_INVOLVE_IN_COND,
    RD_JC_IS_INVOLVE_IN_COND_ANNOTATION,
    RD_JC_IS_INVOLVE_IN_COND_BORDER_CHECKPOINT,
    RD_JC_IS_INVOLVE_IN_COND_CAM,
    RD_JC_IS_INVOLVE_IN_COND_CLOSURE,
    RD_JC_IS_INVOLVE_IN_COND_DS,
    RD_JC_IS_INVOLVE_IN_COND_LANE,
    RD_JC_IS_INVOLVE_IN_COND_RAILWAY_CROSSING,
    RD_JC_IS_INVOLVE_IN_COND_SPEED_BUMP,
    RD_JC_IS_INVOLVE_IN_COND_TOLL,
    RD_JC_IS_INVOLVE_IN_COND_TRAFFIC_LIGHT,
    RD_JC_IS_INVOLVE_IN_COND_WITH_COND_DT,
};

/*****************************************************************************/
/*             S E R V I C E   A T T R I B U T E S   V A L U E S             */
/*****************************************************************************/
const std::string SRV_ATTR_TRUE = "1";
const std::string SRV_ATTR_EMPTY = "";
const std::string SRV_ATTR_FALSE = "";

const std::string STR_TO_MASTER = "master";
const std::string STR_TO_SLAVE = "slave";
const std::string STR_TO_ATTR_OWNER = "";


//Object's property which is aspect of other object
//service attribute value calculation
struct Aspect
{
public:
    enum class Type
    {
        Geometry,
        Attribute,
        Relations
    };
    Type type;
    std::string id;
};

//Define aspect and how to reach it, which is going to
//be used during service attributes callculation
class Dependence
{
public:
    Dependence(Aspect&& aspect, RelationsPath&& aspectLocation)
        : aspect_(std::move(aspect))
        , path_(std::move(aspectLocation))
    {}

    const Aspect& aspect() const
    {
        return aspect_;
    }

    const RelationsPath& path() const
    {
        return path_;
    }

private:
    Dependence() = delete;
    Aspect aspect_;
    RelationsPath path_;
};

// Template wrapper for callback used at runtime
// to calculate service attribute value.
// Allows to store custom callback with
// predefined invocation arguments
template <typename RetType, typename ObjClass, typename ... Args>
class CommonCallbackWrapper
{
public:
    template <typename Func>
    explicit CommonCallbackWrapper(Func func, Args ... args)
        : func_(std::bind(func, std::placeholders::_1, std::placeholders::_2, args ...)) {}

    RetType operator ()(const GeoObject* o, ObjectsCache& cache)
    {
        return func_(as<ObjClass>(o), cache);
    }

private:
    std::function<RetType(const ObjClass*, ObjectsCache& cache)> func_;
};

template <typename ObjClass, typename ... Args>
class CallbackWrapper : public CommonCallbackWrapper<std::string, ObjClass, Args ...>
{
public:
    template <typename Func>
    explicit CallbackWrapper(Func func, Args ... args)
        : CommonCallbackWrapper<std::string, ObjClass, Args ...>(func, args ...){}
};

typedef std::map<std::string, std::string> SuggestTexts;

template <typename ObjClass, typename ... Args>
class SuggestCallbackWrapper : public CommonCallbackWrapper<SuggestTexts, ObjClass, Args ...>
{
public:
    template <typename Func>
    explicit SuggestCallbackWrapper(Func func, Args ... args)
        : CommonCallbackWrapper<SuggestTexts, ObjClass, Args ...>(func, args ...){}
};

struct SuffixValue
{
    SuffixValue(std::string suffix, std::string value)
        : suffix(std::move(suffix))
        , value(std::move(value))
        {}
    std::string suffix;
    std::string value;
};
typedef std::vector<SuffixValue> SuffixValues;

template <typename ObjClass, typename ... Args>
class SuffixValuesCallbackWrapper : public CommonCallbackWrapper<SuffixValues, ObjClass, Args ...>
{
public:
    template <typename Func>
    explicit SuffixValuesCallbackWrapper(Func func, Args ... args)
        : CommonCallbackWrapper<SuffixValues, ObjClass, Args ...>(func, args ...){}
};


class ServiceAttribute
{
public:
    enum class ResultType
    {
        String,
        SuffixValue
    };
    typedef std::function<std::string(const GeoObject*, ObjectsCache&)> Callback;
    typedef std::function<bool(const GeoObject*)> IsAffectedByCallback;

    typedef std::function<SuffixValues(const GeoObject*, ObjectsCache&)> SuffixValueCallback;

    ServiceAttribute(const std::string& categoryId,
            const std::string& id,
            Callback callback);
    ServiceAttribute(const std::string& categoryId,
            const std::string& prefix,
            SuffixValueCallback suffixValueCallback);
    ServiceAttribute(const std::string& categoryId,
            const std::string& id);

    //Get all dependencies for this attribute
    const std::vector<Dependence>& dependencies() const { return dependencies_; }
    //Add dependence for attribute
    ServiceAttribute& depends(Dependence&& dep)
    {
        dependencies_.emplace_back(std::move(dep));
        return *this;
    }
    ServiceAttribute& setIsAffectedByCallback(IsAffectedByCallback isAffectedByCallback)
    {
        isAffectedByCallback_ = std::move(isAffectedByCallback);
        return *this;
    }

    const std::string& id() const { return id_; }
    ResultType resultType() const { return suffixValueCallback_ ? ResultType::SuffixValue : ResultType::String; }

    //Calculate value for attribute
    std::string calc(const ObjectPtr&, ObjectsCache&) const;

    SuffixValues calcWithSuffix(const ObjectPtr&, ObjectsCache&) const;

    //Check if the object affects this attribute
    bool isAffectedBy(const GeoObject*) const;

    const std::string& categoryId() const { return categoryId_;}

private:
    ServiceAttribute() = delete;
    std::string categoryId_;
    std::string id_;
    Callback callback_;
    SuffixValueCallback suffixValueCallback_;
    std::vector<Dependence> dependencies_;
    IsAffectedByCallback isAffectedByCallback_;
};

class SuggestCalc
{
public:
    typedef std::function<SuggestTexts(const GeoObject*, ObjectsCache&)> Callback;

    SuggestCalc(const std::string& categoryId, Callback callback);

    const std::vector<Dependence>& dependencies() const { return dependencies_; }
    SuggestCalc& depends(Dependence&& dep)
    {
        dependencies_.emplace_back(std::move(dep));
        return *this;
    }
    SuggestTexts calc(const GeoObject*, ObjectsCache&) const;

private:
    SuggestCalc() = delete;
    std::string categoryId_;
    Callback callback_;
    std::vector<Dependence> dependencies_;
};

class ServiceAttributesRegistry
{
public:
    //Base class for use during registration
    class Registrar
    {
    public:
        explicit Registrar(ServiceAttributesRegistry&, const std::string&);

        void registerSuggestCallback(SuggestCalc::Callback callback);

        ServiceAttribute& registerAttr(const std::string& attributeId,
            ServiceAttribute::Callback callback);

        ServiceAttribute& registerAttr(const std::string& attributeId,
            ServiceAttribute::SuffixValueCallback suffixValueCallback);

    private:
        ServiceAttributesRegistry& registry_;
        const std::string targetCategoryId_;
    };

    struct ServiceAttrSubPath
    {
        const ServiceAttribute* srvAttr;
        const Dependence* dependence;
        size_t pathOffset;
    };

    //Get service attributes computed for objects of category
    const std::vector<ServiceAttribute>& categoryAttrs(const std::string& categoryId) const;

    //Get dependency paths going through objects of the category
    std::vector<ServiceAttrSubPath> dependedAttrs(const std::string& categoryId) const;

    //Set of all categories having service attributes
    StringSet registeredCategories() const;

    //Get callback and dependencies for names put to suggest data
    SuggestTexts calculateSuggest(const std::string& categoryId, const GeoObject* obj,
        ObjectsCache& cache) const;
    const std::vector<Dependence>& suggestDependencies(const std::string& categoryId) const;

    static void init(const EditorCfg& editorCfg);
    static const ServiceAttributesRegistry& get();

private:
    ServiceAttributesRegistry();
    ServiceAttributesRegistry(const ServiceAttributesRegistry&) = delete;

    static std::unique_ptr<ServiceAttributesRegistry> s_serviceAttributes;
    static std::string dummyCallback(const GeoObject*, ObjectsCache&)
    {
        return s_emptyString;
    }

    ServiceAttribute& registerAttr(const std::string& categoryId,
        const std::string& attributeId,
        ServiceAttribute::Callback callback);

    ServiceAttribute& registerAttr(const std::string& categoryId,
        const std::string& attributeId,
        ServiceAttribute::SuffixValueCallback suffixValueCallback);

    void registerSuggestCallback(const std::string& categoryId,
            SuggestCalc::Callback callback);
    void buildSuggestDependencies(const EditorCfg&);
    void buildPaths();

private:
    std::map<std::string, std::vector<ServiceAttribute>> attributesByCategory_;
    std::multimap<std::string, ServiceAttrSubPath> pathsThroughCategory_;
    std::map<std::string, SuggestCalc> suggestCallbacks_;
};

void
preloadRequiredObjects(const TOIds& clients, ObjectsCache& cache);

GeoObjectCollection
affectedDependentObjects(const GeoObject* obj, ObjectsCache& cache);

void
preloadDependentObjects(const TOIds& clients, ObjectsCache& cache);

}//srv_attrs
}//wiki
}//maps
