#include "create_schema.h"
#include "db_helpers.h"
#include <maps/wikimap/mapspro/services/editor/src/common.h>
#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>

#include <yandex/maps/wiki/common/pg_retry_helpers.h>
#include <yandex/maps/wiki/common/retry_duration.h>

#include <maps/libs/pgpool/include/pool_configuration.h>

namespace maps {
namespace wiki {
namespace sync {

namespace {

const std::string SCHEMANAME_TMP = "vrevisions_tmp";
const std::string DROP_SCHEMANAME_TMP =
    "DROP SCHEMA IF EXISTS " + SCHEMANAME_TMP + " CASCADE;";

std::map<std::string, std::string>
splitConnectionString(const std::string& str)
{
    ASSERT(!str.empty());

    std::map<std::string, std::string> keyValues;

    StringList parts;
    boost::split(parts, str, [](char c) { return c == ' '; });
    for (const auto& part : parts){
        StringList kv;
        boost::split(kv, part, [](char c) { return c == '='; });
        ASSERT(kv.size() == 2);
        ASSERT(keyValues.insert(std::make_pair(kv.front(), kv.back())).second);
    }
    return keyValues;
}

std::string getSchemaSql(pgpool3::Pool& pool, const std::string& schemaName)
{
    auto poolState = pool.state();
    auto master = poolState.configuration.master();
    REQUIRE(master, "master not exists");
    const auto params = splitConnectionString(poolState.connParams);

    std::stringstream ss;
    ss << "PGPASSWORD=" << params.at("password")
       << " pg_dump --schema-only --schema=" << schemaName
       << " --port=" << master->port()
       << " --host=" << master->host()
       << " --username=" << params.at("user")
       << " " << params.at("dbname");

    auto sql = ReadingPipe(ss.str()).read();
    REQUIRE(!sql.empty(), "can not load schema : " << schemaName);
    boost::replace_all(sql, "SET row_security = off;", "");
    return sql;
}

bool isSchemaExists(
    pgpool3::Pool& pool,
    const std::string& schemaName)
{
    return common::retryDuration([&] {
        auto work = pool.masterReadOnlyTransaction();
        auto r = work->exec(
            "SELECT 1"
            " FROM information_schema.schemata"
            " WHERE schema_name='" + schemaName + "'");
        return !r.empty();
    });
}

bool createSchemaIfNotExists(
    pgpool3::Pool& pool,
    const revision::Branch& branch,
    SchemaNameType type)
{
    REQUIRE(branch.type() != revision::BranchType::Trunk,
            "Wrong branch type " << branch.type());

    const auto schemaName = vrevisionsSchemaName(branch.id());

    if (isSchemaExists(pool, schemaName)) {
        return false;
    }

    const auto trunkSchemaName = vrevisionsSchemaName(revision::TRUNK_BRANCH_ID);
    auto sql = getSchemaSql(cfg()->poolCore(), trunkSchemaName);

    const auto& newSchemaName = type == SchemaNameType::Temporary
        ? SCHEMANAME_TMP
        : schemaName;
    boost::replace_all(sql, trunkSchemaName, newSchemaName);
    if (type == SchemaNameType::Temporary) {
        sql.insert(0, DROP_SCHEMANAME_TMP);
    }

    common::execCommitWithRetries(
        pool,
        "create schema " + newSchemaName,
        {}, // searchPath
        sql + "; SET SEARCH_PATH=public",
        [&] (pqxx::transaction_base& work) {
            work.exec(
                "SELECT vrevisions_stable.prepare_schema(" +
                work.quote(newSchemaName) + "," +
                std::to_string(branch.id()) + ")");
        }
    );

    INFO() << newSchemaName << " created";
    return true;
}

} // namespace

bool createViewSchemaIfNotExists(
    const revision::Branch& branch,
    SchemaNameType type)
{
    return createSchemaIfNotExists(cfg()->poolView(branch.id()), branch, type);
}

bool createLabelsSchemaIfNotExists(
    const revision::Branch& branch,
    SchemaNameType type)
{
    return createSchemaIfNotExists(cfg()->poolLabels(branch.id()), branch, type);
}

} // namespace sync
} // namespace wiki
} // namespace maps

