#include "sync_attrs.h"
#include "db_helpers.h"
#include "lock_helpers.h"
#include <maps/wikimap/mapspro/services/editor/src/objectvisitor.h>
#include "maps/wikimap/mapspro/services/editor/src/srv_attrs/calc.h"

#include <maps/wikimap/mapspro/services/editor/src/views/objects_c.h>
#include <maps/wikimap/mapspro/services/editor/src/views/objects_p.h>
#include <maps/wikimap/mapspro/services/editor/src/views/objects_l.h>
#include <maps/wikimap/mapspro/services/editor/src/views/objects_a.h>
#include <maps/wikimap/mapspro/services/editor/src/views/objects_r.h>

#include <maps/wikimap/mapspro/services/editor/src/objects/object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/point_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/areal_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/line_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/complex_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/relation_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/attr_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/model_object.h>

#include <maps/wikimap/mapspro/libs/views/include/magic_strings.h>
#include <yandex/maps/wiki/common/pg_utils.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/revision/branch.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <set>

namespace maps {
namespace wiki {
namespace sync {

namespace {

const std::string EMPTY_HSTORE = "''::hstore";

class ObjectsViewLocker
{
public:
    ObjectsViewLocker(
            Transaction& work,
            const std::set<std::string>& branchViews)
        : work_(work)
        , branchViews_(branchViews)
    {
    }

    void lockOidsInView(const GeoObjectCollection& collection)
    {
        if (branchViews_.empty()) {
            return;
        }

        for (const auto& objPtr : collection) {
            data_[objPtr->tablename()].insert(objPtr->id());
        }

        for (const auto& tableName : views::VIEW_TABLES) {
            if (tableName != views::TABLE_OBJECTS_R) {
                lockObjects(tableName);
            }
        }
    }

private:
    void lockObjects(const std::string& tableName)
    {
        auto it = data_.find(tableName);
        if (it == data_.end() || it->second.empty()) {
            return;
        }

        if (!branchViews_.count(tableName)) {
            return;
        }

        auto query = "SELECT id FROM vrevisions_stable." + tableName
                + " WHERE " + common::whereClause(ID, it->second)
                + " ORDER BY id, branch_mask_id"
                + " FOR UPDATE";

        work_.exec(query);
    }

    Transaction& work_;
    std::set<std::string> branchViews_;
    std::unordered_map<std::string, TOIds> data_;
};

class UpdateSrvAttrsProcessor : public ObjectProcessor
{
public:
    UpdateSrvAttrsProcessor(
            Transaction& work,
            TBranchId branchId,
            std::set<std::string> branchViews)
        : work_(work)
        , branchId_(branchId)
        , branchViews_(std::move(branchViews))
    {
    }

    void processGeoObject(GeoObject* obj) override
    {
        THROW_WIKI_INTERNAL_ERROR("Impossible object " << obj->id());
    }

    void processArealObject(ArealObject* obj) override
    {
        update<ArealObject::ViewType>(*obj);
    }

    void processJunction(Junction* obj) override
    {
        update<Junction::ViewType>(*obj);
    }

    void processLinearElement(LinearElement* obj) override
    {
        update<LinearElement::ViewType>(*obj);
    }

    void processLineObject(LineObject* obj) override
    {
        update<LineObject::ViewType>(*obj);
    }

    void processPointObject(PointObject* obj) override
    {
        update<PointObject::ViewType>(*obj);
    }

    void processComplexObject(ComplexObject* obj) override
    {
        update<ComplexObject::ViewType>(*obj);
    }

    void processAttrObject(AttrObject* obj) override
    {
        THROW_WIKI_INTERNAL_ERROR("Impossible object " << obj->id());
    }

    void processRelationObject(RelationObject* obj) override
    {
        THROW_WIKI_INTERNAL_ERROR("Impossible object " << obj->id());
    }

    void processModel3dObject(Model3dObject* obj) override
    {
        THROW_WIKI_INTERNAL_ERROR("Impossible object " << obj->id());
    }

    template<typename T>
    void update(GeoObject& object)
    {
        auto srvAttrsValue = getSrvAttrsValue(object);

        bool updateZminZmax = false;
        if (!object.geom().isNull()) {
            TZoom oldZmin = object.zmin();
            TZoom oldZmax = object.zmax();
            object.generalize(work_);
            updateZminZmax = (object.zmin() != oldZmin || object.zmax() != oldZmax);
        }
        if (object.cache().policy().serviceAttributesLoadPolicy ==
                        ServiceAttributesLoadPolicy::Load
                && !updateZminZmax
                && object.original()
                && srvAttrsValue == getSrvAttrsValue(*object.original())) {
            return;
        }

        if (branchViews_.count(T::tableName)) {
            query_ += "SELECT " + T::updateSrvAttrsFunction + "("
                + std::to_string(branchId_) + ","
                + std::to_string(object.id()) + ","
                + std::to_string(object.zmin()) + ","
                + std::to_string(object.zmax()) + ","
                + srvAttrsValue + ");\n";
        } else {
            query_ += "UPDATE " + T::tableName
                + " SET service_attrs=" + srvAttrsValue + ","
                + "   zmin=" + std::to_string(object.zmin()) + ","
                + "   zmax=" + std::to_string(object.zmax()) +
                + " WHERE id=" + std::to_string(object.id())
                + ";\n";
        }
    }

    void execute()
    {
        if (!query_.empty()) {
            work_.exec(query_);
        }
    }

private:
    std::string getSrvAttrsValue(const GeoObject& object) const
    {
        const auto& srvAttrs = object.serviceAttributes();
        bool hasData = false;
        for (const auto& pair : srvAttrs) {
            if (!pair.second.empty()) {
                hasData = true;
                break;
            }
        }
        return hasData
            ? common::attributesToHstore(work_, srvAttrs)
            : EMPTY_HSTORE;
    }

    Transaction& work_;
    TBranchId branchId_;
    std::set<std::string> branchViews_;

    std::string query_;
};

} // namespace

void updateAttrs(ObjectsCache& cache, ObjectPredicate filter)
{
    auto completeFilter = [&] (const GeoObject* object) -> bool
    {
        return filter(object) &&
               object->revision().valid() &&
               !object->isDeleted() &&
               !is<RelationObject>(object) && // no srv attrs, no visibility
               object->syncView();
    };
    GeoObjectCollection collection = cache.find(completeFilter);
    srv_attrs::CalcSrvAttrs srvAttrsCalculator(cache);
    Transaction& work = cache.workView();

    const auto& branch = cache.branchContext().branch;
    auto branchViews = findBranchViews(work, branch.id());

    if (!branchViews.empty()) {
        lockViews(work, LockType::Shared);
    }

    ObjectsViewLocker(work, branchViews).lockOidsInView(collection);

    UpdateSrvAttrsProcessor updateProcessor(work, branch.id(), std::move(branchViews));

    for (const auto& objPtr : collection) {
        srvAttrsCalculator.process(objPtr);
        objPtr->applyProcessor(updateProcessor);
    }

    updateProcessor.execute();
}

} // namespace sync
} // namespace wiki
} // namespace maps
