#include "sync_objects.h"
#include "db_helpers.h"
#include "dependent_info.h"
#include "sync_processor.h"
#include "target_tables.h"
#include "tasks.h"
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/magic_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/srv_attrs/registry.h>
#include <maps/wikimap/mapspro/services/editor/src/observers/renderer.h>
#include <maps/wikimap/mapspro/services/editor/src/rendererpool.h>

#include <maps/wikimap/mapspro/services/editor/src/revision_meta/update_queue.h>
#include <maps/wikimap/mapspro/services/editor/src/revision_meta/bbox_update_task.h>

#include <maps/wikimap/mapspro/libs/views/include/magic_strings.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>

namespace maps {
namespace wiki {
namespace sync {

// https://wiki.yandex-team.ru/maps/dev/core/wikimap/mapspro/synch-tds

namespace rf = revision::filters;

namespace {

const size_t BULK_LOADING_COMMITS_SIZE = 1000;
const size_t MAX_PARTIAL_UPDATE_LABELS_THRESHOLD = 500000;
const size_t BIG_DATA_COMMITS_LIMIT = 100;
const size_t MAX_DEPENDENT_INFO_THREADS = 3;

void
logCategories(const std::string& message, const StringSet& categoryIds)
{
    if (categoryIds.empty()) {
        INFO() << message << "<empty>";
        return;
    }
    INFO() << message
           << common::join(categoryIds, ' ');
}

template <typename Loader>
void
bulkLoad(const TOIds& commitIds, Loader loader)
{
    TOIds bulkCommitIds;
    for (auto commitId : commitIds) {
        bulkCommitIds.insert(commitId);
        if (bulkCommitIds.size() >= BULK_LOADING_COMMITS_SIZE) {
            loader(bulkCommitIds);
            bulkCommitIds.clear();
        }
    }
    if (!bulkCommitIds.empty()) {
        loader(bulkCommitIds);
    }
}

} // namespace

SyncObjects::SyncObjects(
        SyncParams&& params,
        BranchLocker& branchLocker)
    : params_(std::move(params))
    , branchLocker_(branchLocker)
    , branch_(params_.branch())
{
    branchLocker_.lockBranchPersistently(params_.branch(), params_.branchLockType());

    branch_ = params_.updateBranch(branchLocker_.work());

    targetTables_.reset(new TargetTables(params_));
    if (branch_.type() != revision::BranchType::Trunk) {
        targetTables_->createSchemaIfNotExists();
    }

    INFO() << "SyncObjects::INIT"
           << " threads: " << params_.threads()
           << " branch: " << branch_.id()
                << "," << branch_.type() << "(" << branch_.state() << ")"
           << " default batch-size: " << params_.batchSize();
}

SyncObjects::~SyncObjects() = default;
SyncObjects::SyncObjects(SyncObjects&&) = default;

void
SyncObjects::run(CustomIds customIds)
{
    params_.updateHeadCommitId(branchLocker_.work());

    initCustomIds(std::move(customIds));

    if (params_.setProgressState() == SetProgressState::Yes) {
        auto work = masterCoreTransaction(AccessMode::ReadWrite);
        branch_.setState(*work, revision::BranchState::Progress);
        work->commit();
    }

    rebuildData();

    if (params_.setProgressState() == SetProgressState::Yes) {
        auto& mainTxn = branchLocker_.work();
        branch_.setState(
            mainTxn, params_.executionState()->fail
                ? revision::BranchState::Unavailable
                : revision::BranchState::Normal);
    }
}

void
SyncObjects::initCustomIds(CustomIds customIds)
{
    if (customIds.empty()) {
        return;
    }

    customObjectIds_ = std::move(customIds.objectIds);
    if (customIds.commitIds.empty()) {
        return;
    }

    customCommitIds_ = std::move(customIds.commitIds);

    auto headCommitId = params_.headCommitId();
    for (auto commitId : *customCommitIds_) {
        REQUIRE(commitId <= headCommitId,
            "commit id: " << commitId << " out of range [1," << headCommitId << "]");
    }

    revision::RevisionsGateway gateway(branchLocker_.work(), branch_);
    auto snapshot = gateway.snapshot(headCommitId);
    auto loader = [&] (const TOIds& commitIds) {
            auto filter = revision::filters::CommitAttr::id().in(commitIds);
            for (const auto& revId : snapshot.revisionIdsByFilter(filter)) {
                customObjectIds_->insert(revId.objectId());
            }
        };

    bulkLoad(*customCommitIds_, loader);
}

void
SyncObjects::rebuildData()
{
    const bool isAllObjects = !customObjectIds_;
    if (isAllObjects) {
        clearTargetTablesByFlags();
    }

    if (!params_.headCommitId()) {
        return;
    }

    const SyncFlags& flags = params_.syncFlags();
    if (flags.view() || flags.attrs() || flags.labels() || flags.suggest()) {
        if (isAllObjects) {
            rebuildDataForAllObjectIds();
        } else {
            rebuildDataForCustomObjectIds();
        }
    }

    if (flags.bboxAny()) {
        if (isAllObjects) {
            rebuildBBoxes(boost::none);
        } else if (customCommitIds_) {
            rebuildBBoxes(customCommitIds_);
        } else {
            INFO() << "SyncObjects: bboxes rebuild for custom objects is not allowed";
        }
    }

    INFO() << "SyncObjects:"
           << " head-commit-id: " << params_.headCommitId()
           << " stages: " << flags;
}

void
SyncObjects::clearTargetTablesByFlags()
{
    bool needClearAllLabels = false;
    auto branchId = targetTables_->branch().id();

    auto processView = [&, this] () {
        auto workView = cfg()->poolView(branchId).masterWriteableTransaction();

        const SyncFlags& syncFlags = params_.syncFlags();
        if (syncFlags.labels()) {
            auto allLabeledCategories = cfg()->rendererPool()->labeledCategories();
            labeledCategories_ = params_.objectsCategories(allLabeledCategories);
            if (labeledCategories_ == allLabeledCategories) {
                needClearAllLabels = true;
            } else {
                auto filter = headObjectIdsFilter(labeledCategories_);
                filter &= rf::Geom::defined();

                labeledObjectIds_ = tryLoadHeadObjectIds(
                    branchLocker_.work(),
                    params_.headCommitId(),
                    branch_,
                    filter,
                    MAX_PARTIAL_UPDATE_LABELS_THRESHOLD);

                needClearAllLabels =
                    !labeledObjectIds_ &&
                    params_.branchLockType() == revision::Branch::LockType::Exclusive;
            }
        }
        if (syncFlags.attrs() || syncFlags.suggest()) {
            targetTables_->clearSuggestData(*workView);
        }
        if (syncFlags.view()) {
            targetTables_->clearView(*workView);
        }
        workView->commit(); // long time clearing objects data in 'progress' mode
    };

    processView();
    if (needClearAllLabels) {
        auto workLabels = cfg()->poolLabels(branchId).masterWriteableTransaction();
        targetTables_->clearAllLabels(*workLabels);
        workLabels->commit();
    }
}

void
SyncObjects::rebuildDataForCustomObjectIds() const
{
    ASSERT(!!customObjectIds_);

    const TOIds& oids = *customObjectIds_;
    if (oids.empty() || !params_.executionState()->isOk()) {
        return;
    }

    INFO() << "rebuild data for custom object ids : " << oids.size();

    SyncProcessor processor(*targetTables_);

    auto taskInfoView = Tasks::passView(params_.syncFlags());
    if (taskInfoView) {
        processor.processObjects(oids, *taskInfoView);
        processor.wait();
    }

    auto taskInfoAttrs = Tasks::passAttrs(params_.syncFlags());
    auto taskInfoSuggest = Tasks::passSuggest(params_.syncFlags());
    auto taskInfoLabels = Tasks::passLabels(params_.syncFlags());
    if (!taskInfoAttrs && !taskInfoLabels) {
        if (taskInfoSuggest) {
            processor.processObjects(oids, *taskInfoSuggest);
            processor.wait();
        }
        return;
    }

    DependentInfo dependentInfo;
    if (taskInfoAttrs) {
        dependentInfo.enableCollectSrvAttrsObjectIds();
    }
    if (taskInfoSuggest) {
        dependentInfo.enableCollectSuggestObjectIds();
    }
    if (taskInfoLabels) {
        dependentInfo.enableCollectLabeledObjectIds();
    }
    {
        auto threads = std::min(MAX_DEPENDENT_INFO_THREADS, params_.threads());
        INFO() << "SyncObjects: building dependent info with threads: " << threads;
        SyncProcessor processor(*targetTables_, threads);
        processor.processObjects(oids, Tasks::passDependentInfo(dependentInfo));
        processor.wait();
    }

    if (taskInfoAttrs) {
        processor.processObjects(dependentInfo.loadSrvAttrsObjectIds(), *taskInfoAttrs);
        processor.wait();
    }
    if (taskInfoSuggest) {
        processor.processObjects(dependentInfo.loadSuggestObjectIds(), *taskInfoSuggest);
        processor.wait();
    }

    if (!taskInfoLabels || !params_.executionState()->isOk()) {
        return;
    }

    auto labeledOids = dependentInfo.loadLabeledObjectIds();
    INFO() << "revision.object ids for labeler : " << labeledOids.size();
    if (labeledOids.size() > MAX_PARTIAL_UPDATE_LABELS_THRESHOLD &&
        params_.branchLockType() == revision::Branch::LockType::Exclusive)
    {
        labeledOids.clear();
        rebuildAllObjectsLabels();
    } else {
        processor.processObjects(labeledOids, *taskInfoLabels);
        labeledOids.clear();
        processor.wait();
    }
}

void
SyncObjects::rebuildDataForAllObjectIds() const
{
    rebuildAllObjectsView();
    rebuildAllObjectsAttrs();
    rebuildAllObjectsSuggest();
    rebuildAllObjectsLabels();
}

void
SyncObjects::rebuildAllObjectsView() const
{
    auto taskInfoView = Tasks::passView(params_.syncFlags());
    if (!taskInfoView) {
        return;
    }

    const auto& categories = cfg()->editor()->categories();
    auto allCategories = categories.idsByFilter(Categories::All);
    auto objectsCategories = params_.objectsCategories(allCategories);

    rebuildAllObjectsViewForCategories(objectsCategories);

    if (params_.hasCustomViewTableNames()) {
        if (!params_.hasCustomViewTable(views::TABLE_OBJECTS_R)) {
            INFO() << "sync view: objects_r table is excluded";
            return;
        }
        if (params_.isCustomCategoryIds()) {
            objectsCategories = params_.customCategories();
        } else {
            objectsCategories = allCategories;
        }
    }

    rebuildAllRelationsViewForCategories(objectsCategories);
}

void SyncObjects::rebuildAllObjectsViewForCategories(const StringSet& objectsCategories) const
{
    if (objectsCategories.empty()) {
        INFO() << "sync view: empty categories list";
        return;
    }

    if (!params_.executionState()->isOk()) {
        return;
    }

    const auto& categories = cfg()->editor()->categories();

    StringSet objectsViewCategories;
    for (const auto& categoryId : objectsCategories) {
        const auto& category = categories[categoryId];
        if (category.syncView() && category.templateId() != "rel") {
            objectsViewCategories.insert(categoryId);
        }
    }
    logCategories("sync view categories: ", objectsViewCategories);

    auto taskInfoView = Tasks::passView(params_.syncFlags());
    ASSERT(taskInfoView);

    SyncProcessor processor(*targetTables_);

    for (const auto& categoryId : objectsViewCategories) {
        if (!params_.executionState()->isOk()) {
            return;
        }
        auto filter = headObjectIdsFilter(categoryId);

        auto oids = headObjectIds(
            branchLocker_.work(), params_.headCommitId(), branch_, filter);
        INFO() << "revision.object view ids (" << categoryId << ") : " << oids.size();

        auto taskInfo = *taskInfoView;
        taskInfo.name += " (objects, " + categoryId + ")";
        processor.processObjects(oids, taskInfo);
    }

    processor.wait();
}

void SyncObjects::rebuildAllRelationsViewForCategories(const StringSet& objectsCategories) const
{
    if (objectsCategories.empty()) {
        INFO() << "sync view relations: empty categories list";
        return;
    }

    if (!params_.executionState()->isOk()) {
        return;
    }

    const auto& categories = cfg()->editor()->categories();
    auto allCategories = categories.idsByFilter(Categories::All);

    auto relationsCategories =
        sync::relationsCategories(objectsCategories, allCategories);

    logCategories("sync view relations, master categories: ", relationsCategories.master);
    logCategories("sync view relations, slave categories: ", relationsCategories.slave);

    if (relationsCategories.master.empty() && relationsCategories.slave.empty()) {
        return;
    }

    auto filter = headRelationIdsFilter(
        objectsCategories,
        relationsCategories,
        SlaveRole::nonSyncViewRoles());

    auto oids = headObjectIds(
        branchLocker_.work(), params_.headCommitId(), branch_, filter);
    INFO() << "revision.relations view ids: " << oids.size();

    auto taskInfoView = Tasks::passView(params_.syncFlags());
    ASSERT(taskInfoView);

    auto taskInfo = *taskInfoView;
    taskInfo.name += " (relations)";
    SyncProcessor processor(*targetTables_);
    processor.processObjects(oids, taskInfo);
    processor.wait();
}

void
SyncObjects::rebuildAllObjectsAttrs() const
{
    auto taskInfoAttrs = Tasks::passAttrs(params_.syncFlags());
    if (!taskInfoAttrs) {
        return;
    }
    auto attrsCategories = params_.objectsCategories(
            srv_attrs::ServiceAttributesRegistry::get().registeredCategories());
    if (attrsCategories.empty()) {
        INFO() << "sync attrs: empty categories list";
        return;
    }

    logCategories("sync attrs categories: ", attrsCategories);

    SyncProcessor processor(*targetTables_);

    for (const auto& categoryId : attrsCategories) {
        if (!params_.executionState()->isOk()) {
            return;
        }
        auto filter = headObjectIdsFilter(categoryId);

        auto oids = headObjectIds(
            branchLocker_.work(), params_.headCommitId(), branch_, filter);
        INFO() << "revision.object attrs ids (" << categoryId << ") : " << oids.size();

        auto taskInfo = *taskInfoAttrs;
        taskInfo.name += " (" + categoryId + ")";
        processor.processObjects(oids, taskInfo);
    }
    processor.wait();
}

void
SyncObjects::rebuildAllObjectsSuggest() const
{
    auto taskInfoSuggest = Tasks::passSuggest(params_.syncFlags());
    if (!taskInfoSuggest) {
        return;
    }
    auto suggestCategories = params_.objectsCategories(
        cfg()->editor()->categories().idsByFilter(Categories::Suggest));
    if (suggestCategories.empty()) {
        INFO() << "sync suggest: empty categories list";
        return;
    }

    logCategories("sync suggest categories: ", suggestCategories);

    SyncProcessor processor(*targetTables_);

    for (const auto& categoryId : suggestCategories) {
        if (!params_.executionState()->isOk()) {
            return;
        }
        auto filter = headObjectIdsFilter(categoryId);

        auto oids = headObjectIds(
            branchLocker_.work(), params_.headCommitId(), branch_, filter);
        INFO() << "revision.object suggest ids (" << categoryId << ") : " << oids.size();

        auto taskInfo = *taskInfoSuggest;
        taskInfo.name += " (" + categoryId + ")";
        processor.processObjects(oids, taskInfo);
    }
    processor.wait();
}


void
SyncObjects::rebuildAllObjectsLabels() const
{
    auto taskInfoLabels = Tasks::passLabels(params_.syncFlags());
    if (!taskInfoLabels || !params_.executionState()->isOk()) {
        return;
    }
    if (labeledObjectIds_) {
        INFO() << "sync labels size: " << labeledObjectIds_->size();
        SyncProcessor processor(*targetTables_);
        processor.processObjects(*labeledObjectIds_, *taskInfoLabels);
        processor.wait();
    } else if (params_.branchLockType() == revision::Branch::LockType::Shared) {
        if (labeledCategories_.empty()) {
            INFO() << "sync labels: empty categories list";
            return;
        }

        logCategories("sync labels categories: ", labeledCategories_);

        SyncProcessor processor(*targetTables_);
        for (const auto& categoryId : labeledCategories_) {
            if (!params_.executionState()->isOk()) {
                return;
            }
            auto filter = headObjectIdsFilter(categoryId);
            filter &= rf::Geom::defined();

            auto oids = headObjectIds(
                branchLocker_.work(), params_.headCommitId(), branch_, filter);
            INFO() << "revision.object labels ids (" << categoryId << ") : " << oids.size();

            auto taskInfo = *taskInfoLabels;
            taskInfo.name += " (" + categoryId + ")";
            processor.processObjects(oids, taskInfo);
        }
        processor.wait();
    } else {
        auto branchId = branch_.id();
        auto workLabels = BranchContextFacade::acquireWorkWriteLabelsOnly(branchId);
        targetTables_->clearAllLabels(*workLabels);
        INFO() << "Sync all labels started";
        auto workView = BranchContextFacade::acquireWorkWriteViewOnly(branchId);
        RendererObserver::placeAllLabels(*workView, *workLabels, branch_.id());
        INFO() << "Sync all labels done";
        workLabels->commit();
    }
}

namespace {

Categories::Filter
buildBBoxCategoriesFilter(
    const Categories::Filter& commonFilter,
    const SyncParams& params)
{
    if (!params.isCustomCategoryIds()) {
        return commonFilter;
    }

    StringSet categories = params.objectsCategories(
        cfg()->editor()->categories().idsByFilter(commonFilter));

    return [=] (const Category& cat) { return categories.count(cat.id()); };
}

BBoxUpdateAction
bboxUpdateActionFromFlags(const SyncFlags& flags)
{
    if (flags.bboxAll()) {
        return BBoxUpdateAction::Both;
    }
    if (flags.bboxHistory()) {
        return BBoxUpdateAction::History;
    }
    if (flags.bboxView()) {
        return BBoxUpdateAction::View;
    }
    REQUIRE(false, "Can not determine bbox update action from flags: " << flags);
}

} // namespace

void
SyncObjects::rebuildBBoxes(boost::optional<TCommitIds> customCommitIds) const
{
    auto progressFlags = params_.executionState();
    if (!progressFlags->isOk()) {
        return;
    }

    auto& work = branchLocker_.work();
    revision_meta::CommitsQueue queue(work, branch_.id());

    bool canOffloadBBoxProcessingToQueue =
        branch_.type() == revision::BranchType::Trunk
        && !params_.isCustomCategoryIds()
        && params_.bigData() == BigData::No
        && (customCommitIds_ && customCommitIds_->size() < BIG_DATA_COMMITS_LIMIT);

    if (!canOffloadBBoxProcessingToQueue) {
        branchLocker_.lockWaitExclusive(branch_, revision_meta::BRANCH_LOCK_ID);
    } else if (!branchLocker_.tryLockExclusive(branch_, revision_meta::BRANCH_LOCK_ID)) {
        for (auto commitId : *customCommitIds) {
            queue.push(commitId);
        }
        INFO() << "Sync bboxes skipped, "
               << customCommitIds->size() << " commits added into queue";
        return;
    }

    const TCommitId headCommitId = params_.headCommitId();
    BBoxUpdateAction action = customCommitIds
        ? BBoxUpdateAction::Both
        : bboxUpdateActionFromFlags(params_.syncFlags());

    TCommitId startCommitId = 1;
    if (customCommitIds) {
        ASSERT(!customCommitIds->empty());
        auto commitId = findNonServiceStartCommitId(work, *customCommitIds);
        startCommitId = commitId ? commitId : headCommitId;

        while (!customCommitIds->empty() && *customCommitIds->begin() < startCommitId) {
            customCommitIds->erase(customCommitIds->begin());
        }
    } else if (action == BBoxUpdateAction::View) {
        startCommitId = headCommitId;
    }

    TCommitIds commitIds;
    auto bboxCategoriesFilter = Categories::CachedGeom;
    if (!customCommitIds) {
        bboxCategoriesFilter =
            buildBBoxCategoriesFilter(Categories::CachedGeom, params_);
        if (params_.syncFlags().bboxHistory()) {
            commitIds = queue.lockCommits(headCommitId, boost::none);
        }
    }

    ComputeBBoxesParams bboxParams{
        action,
        branch_.id(),
        startCommitId,
        headCommitId,
        customCommitIds ? std::move(*customCommitIds) : TCommitIds{},
        params_.commitRangeBatchSize(),
        cfg()->editor()->categories().idsByFilter(bboxCategoriesFilter),
        params_.threads(),
        progressFlags
    };

    ComputeBBoxTask task(bboxParams);

    INFO() << "Sync bboxes started";
    task.exec(work);

    if (!commitIds.empty() && progressFlags->isOk()) {
        queue.deleteCommits(commitIds);
    }
    INFO() << "Sync bboxes done"
           << ", canceled: " << progressFlags->cancel
           << ", failed: " << progressFlags->fail;
}

} // namespace sync
} // namespace wiki
} // namespace maps
