#include "sync_params.h"

#include "db_helpers.h"
#include <maps/wikimap/mapspro/services/editor/src/factory.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/categories.h>

#include <maps/wikimap/mapspro/libs/views/include/magic_strings.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/configs/editor/categories.h>
#include <yandex/maps/wiki/configs/editor/category_groups.h>
#include <yandex/maps/wiki/configs/editor/category_template.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/branch.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

namespace maps {
namespace wiki {
namespace sync {

namespace {

bool
isAll(const StringSet& stages)
{
    return stages.empty() || (stages.size() == 1 && *stages.begin() == STAGES_ALL);
}

bool
hasParam(const StringSet& stages, const std::string& param)
{
    if (isAll(stages)) {
        return true;
    }
    if (!stages.count(param)) {
        return false;
    }
    DEBUG() << "sync: +" << param;
    return true;
}

void
printFlag(std::ostream& os, bool flag, const std::string& name)
{
    if (flag) {
        os << " " << name;
    }
}

void
checkBranchNotDeleted(const revision::Branch& branch)
{
    REQUIRE(branch.type() != revision::BranchType::Deleted,
        "can not rebuild data for " << branch.type() <<
        " branch, id: " << branch.id());
}

revision::Branch
getBranch(TBranchId branchId)
{
    auto work = masterCoreTransaction(AccessMode::ReadOnly);
    revision::BranchManager branchManager(*work);
    auto branch = branchManager.load(branchId);
    checkBranchNotDeleted(branch);
    return branch;
}

revision::Branch
getBranch(const std::string& branchStr)
{
    ASSERT(!branchStr.empty());
    auto work = masterCoreTransaction(AccessMode::ReadOnly);
    revision::BranchManager branchManager(*work);
    auto branch = branchManager.loadByString(branchStr);
    checkBranchNotDeleted(branch);
    return branch;
}

size_t
calcThreadsCount(size_t threads, TBranchId branchId)
{
    if (threads != AUTO_THREAD_COUNT) {
        return threads;
    }

    //reserve main txn for main thread
    return maxThreadsCount(branchId) - 1;
}

} // namespace

SyncFlags::SyncFlags(const std::string& stagesStr)
{
    init(splitCast<StringSet>(stagesStr, ','));
}

SyncFlags::SyncFlags(const StringSet& stages)
{
    init(stages);
}

void SyncFlags::init(const StringSet& stages)
{
    view_ = hasParam(stages, STAGES_VIEW);
    attrs_ = hasParam(stages, STAGES_ATTRS);
    suggest_ = !attrs_ && hasParam(stages, STAGES_SUGGEST);
    labels_ = hasParam(stages, STAGES_LABELS);
    bboxHistory_ = hasParam(stages, STAGES_BBOX) || hasParam(stages, STAGES_BBOX_HISTORY);
    bboxView_ = hasParam(stages, STAGES_BBOX) || hasParam(stages, STAGES_BBOX_VIEW);
    if (isAll(stages)) {
        DEBUG() << "sync: " << STAGES_ALL;
    }
    REQUIRE(view() || attrs() || suggest() || labels() || bboxAny(),
            "Invalid stages: '" << common::join(stages, ',') << "'");
}

std::ostream& operator << (std::ostream& out, const SyncFlags& flags)
{
    if (flags.all()) {
        out << STAGES_ALL;
        return out;
    }

    std::ostringstream os;
    printFlag(os, flags.view(), STAGES_VIEW);
    printFlag(os, flags.attrs(), STAGES_ATTRS);
    printFlag(os, flags.suggest(), STAGES_SUGGEST);
    printFlag(os, flags.labels(), STAGES_LABELS);
    printFlag(os, flags.bboxAll(), STAGES_BBOX);
    if (!flags.bboxAll()) {
        printFlag(os, flags.bboxHistory(), STAGES_BBOX_HISTORY);
        printFlag(os, flags.bboxView(), STAGES_BBOX_VIEW);
    }
    out << os.str();
    return out;
}

SyncParams::SyncParams(
        const ExecutionStatePtr& executionState,
        SyncFlags syncFlags,
        TBranchId branchId,
        size_t threads)
    : executionState_(executionState)
    , syncFlags_(syncFlags)
    , branch_(getBranch(branchId))
    , branchLockType_(revision::Branch::LockType::Shared)
    , setProgressState_(SetProgressState::No)
    , batchSize_(DEFAULT_BATCH_SIZE)
    , threads_(calcThreadsCount(threads, branch_.id()))
    , bigData_(BigData::No)
    , commitRangeBatchSize_(DEFAULT_COMMIT_RANGE_BATCH_SIZE)
{
    ASSERT(executionState_);
    executionState_->clear();

    initCustomCategories(STR_ALL_CATEGORY_GROUPS, {});
}

SyncParams::SyncParams(
        const ExecutionStatePtr& executionState,
        TBranchId branchId,
        SetProgressState setProgressState,
        size_t threads)
    : executionState_(executionState)
    , syncFlags_(STAGES_ALL)
    , branch_(getBranch(branchId))
    , branchLockType_(revision::Branch::LockType::Shared)
    , setProgressState_(setProgressState)
    , batchSize_(DEFAULT_BATCH_SIZE)
    , threads_(calcThreadsCount(threads, branch_.id()))
    , bigData_(BigData::Yes)
    , commitRangeBatchSize_(DEFAULT_COMMIT_RANGE_BATCH_SIZE)
{
    ASSERT(executionState_);
    executionState_->clear();

    initCustomCategories(STR_ALL_CATEGORY_GROUPS, {});
}

SyncParams::SyncParams(
        const ExecutionStatePtr& executionState,
        const std::string& stages,
        const std::string& categoryGroups,
        const std::string& categories,
        const std::string& viewTables,
        const std::string& branchStr,
        revision::Branch::LockType lockType,
        SetProgressState setProgressState,
        size_t batchSize,
        size_t threads,
        BigData bigData)
    : executionState_(executionState)
    , syncFlags_(stages)
    , branch_(getBranch(branchStr))
    , branchLockType_(lockType)
    , setProgressState_(setProgressState)
    , batchSize_(batchSize)
    , threads_(calcThreadsCount(threads, branch_.id()))
    , bigData_(bigData)
    , commitRangeBatchSize_(DEFAULT_COMMIT_RANGE_BATCH_SIZE)
{
    ASSERT(executionState_);
    executionState_->clear();

    initCustomViewTables(viewTables);
    initCustomCategories(categoryGroups, categories);

    if (customViewTableNames_) {
        REQUIRE(!syncFlags_.suggest() && !syncFlags_.labels() && !syncFlags_.bboxAny(),
            "Do not use view tables param for suggest, labels or bbox synchronization");
    }
}

revision::Branch
SyncParams::updateBranch(Transaction& work)
{
    revision::BranchManager branchManager(work);
    branch_ = branchManager.load(branch_.id());
    checkBranchNotDeleted(branch_);
    return branch_;
}

void
SyncParams::updateHeadCommitId(Transaction& work)
{
    headCommitId_ = revision::RevisionsGateway(work, branch_).headCommitId();
    INFO() << "detect head-commit: " << *headCommitId_
           << " for branch id: " << branch_.id();
}

TCommitId
SyncParams::headCommitId() const
{
    ASSERT(headCommitId_);
    return *headCommitId_;
}

void SyncParams::initCustomViewTables(const std::string& viewTables)
{
    if (viewTables.empty() || viewTables == STR_ALL_VIEW_TABLES) {
        return;
    }
    auto viewTableNames = splitCast<StringSet>(viewTables, ',');
    for (const auto& name : viewTableNames) {
        REQUIRE(views::VIEW_TABLES.count(name), "Wrong view table name " << name);
    }
    if (viewTableNames != views::VIEW_TABLES) {
        customViewTableNames_ = std::move(viewTableNames);
    }
}

void
SyncParams::initCustomCategories(
    const std::string& categoryGroupsStr,
    const std::string& categoriesStr)
{
    auto categoryGroupIds = splitCast<StringSet>(categoryGroupsStr, ',');
    if (categoryGroupIds.size() == 1 && *categoryGroupIds.begin() == STR_ALL_CATEGORY_GROUPS) {
        if (categoriesStr.empty()) {
            return;
        }
        categoryGroupIds.clear();
    }
    const auto& categoryGroups = cfg()->editor()->categoryGroups();
    StringSet availableCategoryIds;
    for (const auto& group : categoryGroups.allGroups()) {
        const auto& categories = group.second->categoryIds();
        availableCategoryIds.insert(categories.begin(), categories.end());
    }
    customCategoryIds_ = categoryGroups.categoryIdsByGroups(categoryGroupIds);

    for (const auto& categoryId : splitCast<StringSet>(categoriesStr, ',')) {
        REQUIRE(availableCategoryIds.contains(categoryId),
                "Strange category: " << categoryId);
        customCategoryIds_->emplace(categoryId);
    }

    if (*customCategoryIds_ == availableCategoryIds) {
        customCategoryIds_.reset();
    }
}

StringSet
SyncParams::objectCategoriesFromViewTables() const
{
    ASSERT(customViewTableNames_);

    StringSet resultCategories;
    for (const auto& pair : cfg()->editor()->categories()) {
        const auto& category = pair.second;
        if (!category.syncView()) {
            continue;
        }
        const auto& tableName = GeoObjectFactory::objectClass(category.id()).tableName;
        if (!tableName.empty() && hasCustomViewTable(tableName)) {
            resultCategories.emplace(category.id());
        }
    }
    return resultCategories;
}

StringSet
SyncParams::objectsCategories(const StringSet& availableCategories) const
{
    if (!customCategoryIds_ && !customViewTableNames_) {
        return availableCategories;
    }

    StringSet filteredCategoriesFromViewTables;
    if (customViewTableNames_) {
        auto categoriesFromViewTables = objectCategoriesFromViewTables();
        std::set_intersection(
            categoriesFromViewTables.begin(), categoriesFromViewTables.end(),
            availableCategories.begin(), availableCategories.end(),
            std::inserter(filteredCategoriesFromViewTables, filteredCategoriesFromViewTables.end()));
    }

    if (customViewTableNames_ && !customCategoryIds_) {
        return filteredCategoriesFromViewTables;
    }

    StringSet filteredCustomCategories;
    if (customCategoryIds_) {
        std::set_intersection(
            (*customCategoryIds_).begin(), (*customCategoryIds_).end(),
            availableCategories.begin(), availableCategories.end(),
            std::inserter(filteredCustomCategories, filteredCustomCategories.end()));
    }

    if (customCategoryIds_ && !customViewTableNames_) {
        return filteredCustomCategories;
    }

    StringSet resultCategories;
    std::set_intersection(
        filteredCategoriesFromViewTables.begin(), filteredCategoriesFromViewTables.end(),
        filteredCustomCategories.begin(), filteredCustomCategories.end(),
        std::inserter(resultCategories, resultCategories.end()));
    return resultCategories;
}

bool SyncParams::hasCustomViewTable(const std::string& tableName) const
{
    if (!customViewTableNames_) {
        return false;
    }
    return customViewTableNames_->count(tableName) > 0;
}

StringSet SyncParams::customCategories() const
{
    ASSERT(customCategoryIds_);
    return *customCategoryIds_;

}

} // namespace sync
} // namespace wiki
} // namespace maps
