#include "sync_processor.h"
#include "target_tables.h"
#include "sync_params.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/profiletimer.h>
#include <yandex/maps/wiki/revision/branch.h>

#include <sstream>
#include <vector>

namespace maps {
namespace wiki {
namespace sync {

namespace {

const size_t WAIT_POOL_MAX_BATCHES_PER_THREAD = 100;

class SyncBatchData
{
public:
    SyncBatchData(
            const TargetTables& targetTables,
            std::vector<TOid>& oids,
            const Tasks::TaskInfo& taskInfo,
            size_t batchIndex,
            size_t batchNumber)
        : targetTables_(targetTables)
        , taskInfo_(taskInfo)
        , batchIndex_(batchIndex)
        , batchNumber_(batchNumber)
    {
        oids.swap(oids_);
    }

    void operator() () const
    {
        try {
            if (targetTables_.syncParams().executionState()->isOk()) {
                ProfileTimer pt;
                run();
                INFO() << "SYNC BATCH PROCESSED " << batchInfo()
                       << " : " << pt.getElapsedTime();
            } else {
                INFO() << "SYNC BATCH SKIPPED " << batchInfo();
            }
        }
        catch (const maps::Exception& e) {
            std::ostringstream ss;
            ss << e;
            handleError(ss.str().c_str());
        }
        catch (const pqxx::failure& e) {
            handleError(e.what());
        }
        catch (const std::exception& e) {
            handleError(e.what());
        }
        catch (...) {
            handleError("unknown error");
        }
    }

private:
    void handleError(const char* errorMessage) const
    {
        targetTables_.syncParams().executionState()->fail = true;
        ERROR() << "SYNC BATCH FAILURE " << batchInfo()
                << " Error: " << errorMessage;
    }

    std::string batchLogContext() const
    {
        return std::to_string(batchIndex_) + "/" + std::to_string(batchNumber_);
    }

    std::string batchInfo() const
    {
        return batchLogContext() + " " + taskInfo_.name;
    }

    void run() const
    {
        TOIds oids{oids_.begin(), oids_.end()};
        Tasks tasks(targetTables_, oids, batchLogContext());
        (taskInfo_.method)(tasks);
    }

private:
    std::vector<TOid> oids_;
    const TargetTables& targetTables_;
    const Tasks::TaskInfo taskInfo_;
    const size_t batchIndex_;
    const size_t batchNumber_;
};

} // namespace

SyncProcessor::SyncProcessor(const TargetTables& targetTables, size_t threads)
    : targetTables_(targetTables)
{
    if (threads) {
        threadPool_ = make_unique<ThreadPool>(threads);
    }
}

void
SyncProcessor::wait()
{
    if (threadPool_) {
        if (targetTables_.syncParams().executionState()->isOk()) {
            threadPool_->shutdown();
        }
        threadPool_.reset();
    }
}

void
SyncProcessor::processObjects(
    const TOIds& oids,
    const Tasks::TaskInfo& taskInfo)
{
    if (oids.empty()) {
        return;
    }

    const auto& params = targetTables_.syncParams();
    if (!params.executionState()->isOk()) {
        INFO() << "SYNC " << taskInfo.name << " SKIPPED: " << oids.size();
        return;
    }

    const auto batchSize = params.batchSize()
        ? std::min(params.batchSize(), taskInfo.maxBatchSize)
        : taskInfo.maxBatchSize;

    const size_t batchNumber = (oids.size() + batchSize - 1) / batchSize;

    if (batchNumber > (WAIT_POOL_MAX_BATCHES_PER_THREAD * params.threads())) {
        wait();
    }

    INFO() << "SYNC " << taskInfo.name
           << " TO PROCESS: " << oids.size()
           << " BATCHES: " << batchNumber << " BATCH SIZE: " << batchSize;

    size_t batchIndex = 0;
    auto batchPusher = [&] (std::vector<TOid>& batchOfOids)
    {
        SyncBatchData batch(
            targetTables_,
            batchOfOids,
            taskInfo, ++batchIndex, batchNumber);

        if (!threadPool_) {
            threadPool_ = make_unique<ThreadPool>(params.threads());
        }
        threadPool_->push(batch);
    };

    std::vector<TOid> batchOfOids;
    batchOfOids.reserve(batchSize);
    for (auto oid : oids) {
        batchOfOids.push_back(oid);
        if (batchOfOids.size() >= batchSize) {
            batchPusher(batchOfOids);
            batchOfOids.clear();
            batchOfOids.reserve(batchSize);
        }
    }
    if (!batchOfOids.empty()) {
        batchPusher(batchOfOids);
    }
}

} // namespace sync
} // namespace wiki
} // namespace maps
