#include "sync_view.h"
#include "db_helpers.h"
#include "lock_helpers.h"
#include <maps/wikimap/mapspro/services/editor/src/objectvisitor.h>

#include <maps/wikimap/mapspro/services/editor/src/views/objects_c.h>
#include <maps/wikimap/mapspro/services/editor/src/views/objects_p.h>
#include <maps/wikimap/mapspro/services/editor/src/views/objects_l.h>
#include <maps/wikimap/mapspro/services/editor/src/views/objects_a.h>
#include <maps/wikimap/mapspro/services/editor/src/views/objects_r.h>

#include <maps/wikimap/mapspro/services/editor/src/objects/object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/point_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/areal_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/line_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/complex_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/relation_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/attr_object.h>

#include <maps/wikimap/mapspro/libs/views/include/magic_strings.h>
#include <yandex/maps/wiki/common/pg_utils.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/configs/editor/categories.h>
#include <yandex/maps/wiki/revision/branch.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <unordered_set>
#include <unordered_map>

namespace maps {
namespace wiki {
namespace sync {

namespace {

using TOidSet = std::unordered_set<TOid>;

struct SyncViewContext
{
    Transaction& work;
    TBranchId branchId;
    TOidSet oidsInView;
    std::set<std::string> branchViews;

    bool hasBranchView(const std::string& tableName) const
    {
        return branchViews.count(tableName) > 0;
    }
};

class ObjectsViewChecker
{
public:
    ObjectsViewChecker(
            Transaction& work,
            const std::set<std::string>& branchViews,
            bool checkViewExists)
        : work_(work)
        , branchViews_(branchViews)
        , checkViewExists_(checkViewExists)
    {
    }

    TOidSet&& getOrLockOidsInView(const GeoObjectCollection& collection) &&
    {
        if (checkViewExists_ || !branchViews_.empty()) {
            for (const auto& objPtr : collection) {
                data_[objPtr->tablename()].insert(objPtr->id());
            }

            for (const auto& tableName : views::VIEW_TABLES) {
                checkObjects(tableName);
            }
        }

        return std::move(oidsInView_);
    }

private:
    void checkObjects(const std::string& tableName)
    {
        auto it = data_.find(tableName);
        if (it == data_.end() || it->second.empty()) {
            return;
        }

        std::string query;
        if (branchViews_.count(tableName)) {
            query = "SELECT id FROM vrevisions_stable." + tableName
                + " WHERE " + common::whereClause(ID, it->second)
                + " ORDER BY id, branch_mask_id"
                + " FOR UPDATE";
        } else if (checkViewExists_) {
            query = "SELECT id FROM " + tableName
                + " WHERE " + common::whereClause(ID, it->second)
                + " ORDER BY id"
                + " FOR UPDATE";
        }

        if (!query.empty()) {
            for (auto row : work_.exec(query)) {
                oidsInView_.insert(row[0].as<TOid>());
            }
        }
    }

    Transaction& work_;
    std::set<std::string> branchViews_;
    bool checkViewExists_;

    std::unordered_map<std::string, TOIds> data_;
    TOidSet oidsInView_;
}; // ObjectsViewChecker

template<typename T>
struct ObjectsCollection
{
    explicit ObjectsCollection(const SyncViewContext& context)
        : context_(context)
    {
    }

    template<typename U>
    void addObject(const U* obj)
    {
        if (context_.oidsInView.count(obj->id())) {
            objectsToUpdate_.emplace_back(context_.work, obj);
        } else {
            objectsToInsert_.emplace_back(context_.work, obj);
        }
    }

    std::string generateUpdateQuery() const
    {
        std::string query;
        if (objectsToUpdate_.empty()) {
            return query;
        }
        if (context_.hasBranchView(T::tableName)) {
            for (const auto& obj : objectsToUpdate_) {
                query += "SELECT " + T::updateFunction + "("
                    + std::to_string(context_.branchId) + ","
                    + obj.valueStr() + ");\n";
            }
        } else {
            for (const auto& obj : objectsToUpdate_) {
                query += "UPDATE " + T::tableName
                    + " SET " + obj.setStr()
                    + " WHERE id=" + std::to_string(obj.id())
                    + ";\n";
            }
        }
        return query;
    }

    std::string generateInsertQuery() const
    {
        std::string query;
        if (objectsToInsert_.empty()) {
            return query;
        }
        if (context_.hasBranchView(T::tableName)) {
            for (const auto& obj : objectsToInsert_) {
                query += "SELECT " + T::insertFunction + "("
                    + std::to_string(context_.branchId) + ","
                    + obj.valueStr() + ");\n";
            }
        } else {
            query = "INSERT INTO " + T::tableName + T::fields
                + " VALUES "
                + common::join(objectsToInsert_,
                    [](const T& obj) { return "(" + obj.valueStr() + ")"; },
                    ',')
                + ";\n";
        }
        return query;
    }

    const SyncViewContext& context_;

    std::vector<T> objectsToInsert_;
    std::vector<T> objectsToUpdate_;
}; // ObjectsCollection

class InsertUpdateVisitor : public ObjectVisitor
{
public:
    explicit InsertUpdateVisitor(const SyncViewContext& context)
        : context_(context)
        , objectsP_(context)
        , objectsL_(context)
        , objectsA_(context)
        , objectsC_(context)
        , objectsR_(context)
    {
    }

    void visitGeoObject(const GeoObject* obj) override
    {
        THROW_WIKI_INTERNAL_ERROR("Impossible object " << obj->id());
    }

    void visitArealObject(const ArealObject* obj) override
    {
        objectsA_.addObject(obj);
    }

    void visitJunction(const Junction* obj) override
    {
        objectsP_.addObject(obj);
    }

    void visitLinearElement(const LinearElement* obj) override
    {
        objectsL_.addObject(obj);
    }

    void visitLineObject(const LineObject* obj) override
    {
        objectsL_.addObject(obj);
    }

    void visitPointObject(const PointObject* obj) override
    {
        objectsP_.addObject(obj);
    }

    void visitComplexObject(const ComplexObject* obj) override
    {
        objectsC_.addObject(obj);
    }

    void visitAttrObject(const AttrObject* obj) override
    {
        THROW_WIKI_INTERNAL_ERROR("Impossible object " << obj->id());
    }

    void visitRelationObject(const RelationObject* obj) override
    {
        objectsR_.addObject(obj);
    }

    void execute() const
    {
        std::string query;

        query += objectsP_.generateUpdateQuery();
        query += objectsL_.generateUpdateQuery();
        query += objectsA_.generateUpdateQuery();
        query += objectsC_.generateUpdateQuery();
        query += objectsR_.generateUpdateQuery();

        query += objectsP_.generateInsertQuery();
        query += objectsL_.generateInsertQuery();
        query += objectsA_.generateInsertQuery();
        query += objectsC_.generateInsertQuery();
        query += objectsR_.generateInsertQuery();

        if (!query.empty()) {
            context_.work.exec(query);
        }
    }

private:
    const SyncViewContext& context_;

    ObjectsCollection<views::PointViewObject> objectsP_;
    ObjectsCollection<views::LinearViewObject> objectsL_;
    ObjectsCollection<views::ArealViewObject> objectsA_;
    ObjectsCollection<views::ComplexViewObject> objectsC_;
    ObjectsCollection<views::RelationViewObject> objectsR_;
}; // InsertUpdateVisitor

class DeleteVisitor : public ObjectVisitor
{
public:
    explicit DeleteVisitor(const SyncViewContext& context)
        : context_(context)
    {
    }

    void visitGeoObject(const GeoObject* obj) override
    {
        THROW_WIKI_INTERNAL_ERROR("Impossible object " << obj->id());
    }

    void visitArealObject(const ArealObject* obj) override
    {
        deleteObject<ArealObject::ViewType>(*obj);
    }

    void visitJunction(const Junction* obj) override
    {
        deleteObject<Junction::ViewType>(*obj);
    }

    void visitLinearElement(const LinearElement* obj) override
    {
        deleteObject<LinearElement::ViewType>(*obj);
    }

    void visitLineObject(const LineObject* obj) override
    {
        deleteObject<LineObject::ViewType>(*obj);
    }

    void visitPointObject(const PointObject* obj) override
    {
        deleteObject<PointObject::ViewType>(*obj);
    }

    void visitComplexObject(const ComplexObject* obj) override
    {
        deleteObject<ComplexObject::ViewType>(*obj);
    }

    void visitAttrObject(const AttrObject* obj) override
    {
        THROW_WIKI_INTERNAL_ERROR("Impossible object " << obj->id());
    }

    void visitRelationObject(const RelationObject* obj) override
    {
        deleteObject<RelationObject::ViewType>(*obj);
    }

    template<typename T>
    void deleteObject(const GeoObject& object)
    {
        if (context_.oidsInView.count(object.id())) {
            if (context_.hasBranchView(T::tableName)) {
                std::string query = "SELECT " + T::deleteFunction + "("
                    + std::to_string(context_.branchId) + ","
                    + std::to_string(object.id()) + ");";
                    context_.work.exec(query);
            } else {
                context_.work.exec(
                    "DELETE FROM " + object.tablename() +
                    " WHERE id=" + std::to_string(object.id()));
            }
        }
        if (object.category().suggest()) {
            context_.work.exec("DELETE FROM suggest_data WHERE object_id = " + std::to_string(object.id()));
        }
    }

private:
    const SyncViewContext& context_;
}; // DeleteVisitor

} // namespace

void updateViews(ObjectsCache& cache, ObjectPredicate filter, bool checkViewExists)
{
    auto completeFilter = [&] (const GeoObject* object) -> bool
    {
        return filter(object)
            && object->revision().valid()
            && object->syncView();
    };
    GeoObjectCollection collection = cache.find(completeFilter);
    Transaction& work = cache.workView();

    const auto& branch = cache.branchContext().branch;
    auto branchViews = findBranchViews(work, branch.id());

    if (!branchViews.empty()) {
        lockViews(work, LockType::Shared);
    }

    TOidSet oidsInView =
        ObjectsViewChecker(work, branchViews, checkViewExists)
            .getOrLockOidsInView(collection);

    SyncViewContext context{
        work,
        branch.id(),
        std::move(oidsInView),
        std::move(branchViews)};

    InsertUpdateVisitor insertUpdateVisitor(context);
    DeleteVisitor deleteVisitor(context);

    for (const auto& objPtr : collection) {
        if (objPtr->isDeleted()) {
            objPtr->applyVisitor(deleteVisitor);
        } else {
            objPtr->generalize(work);
            objPtr->applyVisitor(insertUpdateVisitor);
        }
    }
    insertUpdateVisitor.execute();
}

} // namespace sync
} // namespace wiki
} // namespace maps
