#include "target_tables.h"

#include "create_schema.h"
#include "sync_params.h"
#include "db_helpers.h"
#include "lock_helpers.h"
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>

#include <maps/wikimap/mapspro/services/editor/src/views/objects_c.h>
#include <maps/wikimap/mapspro/services/editor/src/views/objects_p.h>
#include <maps/wikimap/mapspro/services/editor/src/views/objects_l.h>
#include <maps/wikimap/mapspro/services/editor/src/views/objects_a.h>
#include <maps/wikimap/mapspro/services/editor/src/views/objects_r.h>

#include <maps/wikimap/mapspro/libs/views/include/magic_strings.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/configs/editor/categories.h>
#include <maps/libs/common/include/exception.h>

namespace maps {
namespace wiki {
namespace sync {

namespace {

const size_t DELETE_BATCH_SIZE = 1000;

void
truncateTable(Transaction& work, const std::string& tableName)
{
    work.exec("TRUNCATE " + tableName);
    INFO() << "cleared all records from " << tableName;
}

void
clearTable(Transaction& work, const std::string& tableName, const std::string& filter)
{
    REQUIRE(!filter.empty(), "Invalid (empty) vrevisions filter");

    TOIds oids;

    std::string idStr = boost::algorithm::ends_with(tableName, TABLE_SUGGEST_DATA)
        ? "object_id"
        : "id";

    auto lockQuery = "SELECT " + idStr + " FROM " + tableName
        + " WHERE " + filter
        + " ORDER BY " + idStr + " FOR UPDATE";
    for (const auto& row : work.exec(lockQuery)) {
        oids.emplace(row[0].as<TOid>());
    }

    auto deleteOids = [&](const TOIds& batch) {
        auto deleteQuery = "DELETE FROM " + tableName
            + " WHERE " + idStr + " IN (" + common::join(batch, ',') + ")";
        work.exec(deleteQuery);
    };

    TOIds batchOIds;
    for (auto id : oids) {
        batchOIds.insert(id);
        if (batchOIds.size() >= DELETE_BATCH_SIZE) {
            deleteOids(batchOIds);
            batchOIds.clear();
        }
    }
    if (!batchOIds.empty()) {
        deleteOids(batchOIds);
    }

    INFO() << "cleared " << oids.size() << " records from " << tableName
        << ", filtering expr: " << filter;
}

class ViewDeleter
{
public:
    ViewDeleter(
            Transaction& work,
            TBranchId branchId,
            std::string schemaName)
        : work_(work)
        , branchId_(branchId)
        , schemaName_(std::move(schemaName))
        , branchViews_(findBranchViews(work_, branchId_))
    {
    }

    void truncateAll()
    {
        if (!branchViews_.empty()) {
            lockViews(work_, LockType::Exclusive);
            work_.exec("SELECT vrevisions_stable.delete_branch(" + std::to_string(branchId_) + ")");
        } else {
            truncateTable(work_, schemaName_ + "." + views::TABLE_OBJECTS);
            truncateTable(work_, schemaName_ + "." + views::TABLE_OBJECTS_R);
        }
        INFO() << "delete branch " << branchId_;
    }

    void deleteFromTable(
        const std::string& tableName,
        const std::string& filter) const
    {
        if (filter == STR_FALSE) {
            return;
        }
        auto viewTableName = schemaName_ + "." + tableName;
        if (filter == STR_TRUE) {
            truncateTable(work_, viewTableName);
        } else {
            clearTable(work_, viewTableName, filter);
        }
    }

    template<typename T>
    void deleteFromTable(const std::string& filter)
    {
        if (filter == STR_FALSE) {
            return;
        }
        if (!branchViews_.count(T::tableName)) {
            deleteFromTable(T::tableName, filter);
            return;
        }
        lockViews(work_, LockType::Shared);
        auto query = "SELECT " + T::deleteFunction + "(" + std::to_string(branchId_) + ", id)"
            + " FROM " + schemaName_ + "." + T::tableName
            + " WHERE " + filter
            + " ORDER BY id"
            + " FOR UPDATE";
        auto r = work_.exec(query);
        INFO() << "cleared " << r.affected_rows() << " relations, filtering expr: " << filter;
    }

    Transaction& work_;
    TBranchId branchId_;
    std::string schemaName_;
    std::set<std::string> branchViews_;
};

} // namespace

TargetTables::TargetTables(const SyncParams& params)
    : branch_(params.branch())
    , params_(params)
    , schemaName_(vrevisionsSchemaName(branch_.id()))
    , viewEmpty_(false)
    , suggestEmpty_(false)
    , labelsEmpty_(false)
{}

void
TargetTables::createSchemaIfNotExists()
{
    auto result = createViewSchemaIfNotExists(branch_, SchemaNameType::Permanent);
    if (result) {
        viewEmpty_ = suggestEmpty_ = true;
        if (!isSplittedViewLabels(branch_.id())) {
            labelsEmpty_ = true;
            return;
        }
    }
    if (createLabelsSchemaIfNotExists(branch_, SchemaNameType::Permanent)) {
        labelsEmpty_ = true;
    }
}

void
TargetTables::clearView(Transaction& work)
{
    if (viewEmpty_) {
        return;
    }

    viewEmpty_ = true;

    ViewDeleter deleter(work, branch_.id(), schemaName_);

    if (params_.isCustomCategoryIds()) {
        const auto& categories = cfg()->editor()->categories();
        auto allCategories = categories.idsByFilter(Categories::All);

        auto objectsCategories = params_.objectsCategories(allCategories);
        if (!objectsCategories.empty()) {
            deleter.deleteFromTable(views::TABLE_OBJECTS, objectsVRevisionsFilter(objectsCategories));
        }

        if (params_.hasCustomViewTableNames()) {
            if (!params_.hasCustomViewTable(views::TABLE_OBJECTS_R)) {
                INFO() << "clear view: objects_r table is excluded";
                return;
            }
            objectsCategories = params_.customCategories();
        }

        auto relCategories = relationsCategories(objectsCategories, allCategories);
        deleter.deleteFromTable<views::RelationViewObject>(
            relationsVRevisionsFilter(objectsCategories, relCategories));
        return;
    }

    if (!params_.hasCustomViewTableNames()) {
        deleter.truncateAll();
        return;
    }

    if (params_.hasCustomViewTable(views::ComplexViewObject::tableName)) {
        deleter.deleteFromTable<views::ComplexViewObject>(STR_TRUE);
    }
    if (params_.hasCustomViewTable(views::PointViewObject::tableName)) {
        deleter.deleteFromTable<views::PointViewObject>(STR_TRUE);
    }
    if (params_.hasCustomViewTable(views::LinearViewObject::tableName)) {
        deleter.deleteFromTable<views::LinearViewObject>(STR_TRUE);
    }
    if (params_.hasCustomViewTable(views::ArealViewObject::tableName)) {
        deleter.deleteFromTable<views::ArealViewObject>(STR_TRUE);
    }
    if (params_.hasCustomViewTable(views::RelationViewObject::tableName)) {
        deleter.deleteFromTable<views::RelationViewObject>(STR_TRUE);
    }
}

void
TargetTables::clearAllLabels(Transaction& work)
{
    if (labelsEmpty_) {
        return;
    }
    ViewDeleter(work, branch_.id(), schemaName_).deleteFromTable(TABLE_LABELS, STR_TRUE);
    labelsEmpty_ = true;
}

void
TargetTables::clearSuggestData(Transaction& work)
{
    if (suggestEmpty_) {
        return;
    }
    ViewDeleter deleter(work, branch_.id(), schemaName_);
    if (!params_.isCustomCategoryIds()) {
        deleter.deleteFromTable(TABLE_SUGGEST_DATA, STR_TRUE);
        suggestEmpty_ = true;
        return;
    }
    auto suggestCategories =
        cfg()->editor()->categories().idsByFilter(Categories::Suggest);
    auto categories = params_.objectsCategories(suggestCategories);
    if (!categories.empty()) {
        deleter.deleteFromTable(TABLE_SUGGEST_DATA, suggestVRevisionsFilter(categories));
    }
    suggestEmpty_ = true;
}

} // namespace sync
} // namespace wiki
} // namespace maps

