#include "tablevalues.h"
#include "objects/object.h"
#include "objects/attr_object.h"
#include "objects_cache.h"
#include "relations_manager.h"
#include "relations_processor.h"
#include "configs/config.h"
#include "utils.h"

#include <yandex/maps/wiki/configs/editor/categories.h>

namespace maps {
namespace wiki {

namespace {

const TableValues s_emptyTable{};

} // namespace

std::string
dump(const std::multimap<std::string, std::string>& m)
{
    std::ostringstream os;
    os << "[";
    for (const auto& a: m) {
        os << "(" << a.first;
        os << "," << a.second;
        os << ")";
    }
    os << "]";
    return os.str();
}

StringMultiMap
TableValues::keyColumnsValues(size_t row, const AttrDefPtr& attrDefPtr) const
{
    auto allValues = values(row);
    const auto keyColumns = attrDefPtr->keyColumns();
    const auto allColumns = attrDefPtr->columns();
    for (const auto& column : allColumns) {
        if (!keyColumns.contains(column)) {
            allValues.erase(column);
        }
    }
    return allValues;
}

bool
TableValues::contains(const TableValues& other) const
{
    if (numRows() < other.numRows()) {
        return false;
    }
    for (const auto& otherRow : other.rows_) {
        if (std::find(rows_.begin(), rows_.end(), otherRow)
            == rows_.end()) {
            return false;
        }
    }
    return true;
}

void
TableAttributesValues::add(const std::string& attrId, TableValues&& tableValues)
{
    tableAttributesValues_.insert(std::make_pair(attrId, std::move(tableValues)));
}

const TableValues&
TableAttributesValues::find(const std::string& attrId) const
{
    auto it = tableAttributesValues_.find(attrId);
    return it == tableAttributesValues_.end()
        ? s_emptyTable
        : it->second;
}

StringSet
TableAttributesValues::attrNames() const
{
    return readKeys(tableAttributesValues_);
}

void
readTableAttributesFromSlaveInfos(ObjectsCache& cache, TOid objectId)
{
    ObjectPtr obj = cache.getExisting(objectId);
    obj->tableAttributes().clear();
    const auto& range = obj->relations(RelationType::Slave).range(
        obj->category().slaveRoleIds(roles::filters::IsTable));
    for (auto attrDefPtr : obj->attributes().definitions()) {
        if (!attrDefPtr->table()) {
            continue;
        }
        TableValues tableValues;
        //load attribute from slaves
        for (const auto& slaveInfo : range) {
            if (slaveInfo.categoryId() != attrDefPtr->rowObjCategory()) {
                continue;
            }
            ObjectPtr attrObj = cache.getExisting(slaveInfo.id());
            StringMultiMap row;
            for (const auto& attr : attrObj->attributes()) {
                if (!attrDefPtr->columns().count(attr.name())) {
                    continue;
                }
                for (const auto& value : attr.values()) {
                    row.insert({attr.name(), value});
                    DEBUG() << "TABLE: " << attr.name() << ":" << value;
                }
            }
            if (!attrDefPtr->roleAttr().empty()) {
                row.insert({attrDefPtr->roleAttr(), slaveInfo.roleId()});
            }
            tableValues.addRow(std::move(row));
        }
        obj->tableAttributes().add(attrDefPtr->id(), std::move(tableValues));
    }
}

namespace
{
TOid
findBoundRowObject(ObjectPtr obj,
                   std::string role,
                   const StringMultiMap& attrVals,
                   const StringSet& attrsSubSet)
{
    for (const auto& rel : obj->slaveRelations().range(role)) {
        if (AttrObject::matchObjectAttributes(attrVals, rel.relative(), attrsSubSet)) {
            return rel.id();
        }
    }
    return 0;
}

struct NewBinding
{
    TOid master;
    TOid slave;
    std::string role;
};
typedef std::vector<NewBinding> NewBindings;

void
updateRealtionsFromNewBindings(const ObjectPtr& obj, ObjectsCache& cache, const NewBindings& newBindings)
{
    RelationsProcessor(cache).deleteRelations(RelationType::Slave, obj.get(),
        obj->category().slaveRoleIds(roles::filters::IsTable));
    for (const auto& binding : newBindings) {
        cache.relationsManager().createRelation(
            binding.master, binding.slave, binding.role);
    }
}
}//namespace

void
writeTableAttributesToSlaveInfos(ObjectsCache& cache, TOid objectId)
{
    ObjectPtr obj = cache.getExisting(objectId);
    StringSet attrs = obj->tableAttributes().attrNames();
    NewBindings newBindings;
    for (const auto& attrId : attrs) {
        const auto& tabAttrDef = cfg()->editor()->attribute(attrId);
        if (!tabAttrDef->table()) {
            THROW_WIKI_INTERNAL_ERROR("Not a table attribute: " << tabAttrDef->id());
        }
        const TableValues& tableValues = obj->tableAttributes().find(tabAttrDef->id());
        StringSet attrsSubset = tabAttrDef->columns();
        attrsSubset.erase(tabAttrDef->roleAttr());
        std::set<StringMultiMap> rowsKeyCoulmnValues;
        for (size_t row = 0; row < tableValues.numRows(); ++row) {
            const auto keyColumnsValues = tableValues.keyColumnsValues(row, tabAttrDef);
            WIKI_REQUIRE(!rowsKeyCoulmnValues.contains(keyColumnsValues),
                ERR_ATTRIBUTE_VALUE_DUPLICATE,
                "Duplicate values in attribute: " << attrId);
            rowsKeyCoulmnValues.insert(keyColumnsValues);
            //create/read object for row values
            std::string role = tabAttrDef->roleAttr().empty()
                ? tabAttrDef->rowObjRole()
                : tableValues.value(row, tabAttrDef->roleAttr());

            const auto rowValues = tableValues.values(row);
            TOid rowObjId = findBoundRowObject(obj, role, rowValues, attrsSubset);
            if (rowObjId) {
                newBindings.push_back({obj->id(), rowObjId, role});
                continue;
            }
            ObjectPtr attrObj = tabAttrDef->uniqueRows()
                ? AttrObject::createUniqueObject(cache,
                    tabAttrDef->rowObjCategory(),
                    rowValues,
                    attrsSubset)
                : AttrObject::createObject(cache,
                    tabAttrDef->rowObjCategory(),
                    rowValues);
            newBindings.push_back({obj->id(), attrObj->id(), role});
        }
    }
    updateRealtionsFromNewBindings(obj, cache, newBindings);
}

bool
areTableAttrsEqual(ObjectPtr obj1, ObjectPtr obj2)
{
    if (obj1 == nullptr || obj2 == nullptr
        || obj1->categoryId() != obj2->categoryId()) {
        return false;
    }
    if (obj1 == obj2) {
        return true;
    }
    const auto& tableAttributes1 = obj1->tableAttributes();
    const auto& tableAttributes2 = obj2->tableAttributes();
    auto attrNames1 = tableAttributes1.attrNames();
    auto attrNames2 = tableAttributes2.attrNames();
    if (attrNames1 != attrNames2) {
        return false;
    }
    for (const auto& attrName : attrNames1) {
        if (!tableAttributes1.find(attrName).equal(
                tableAttributes2.find(attrName))) {
            return false;
        }
    }
    return true;
}

void
validateTableValuesSize(
    const AttrDefPtr& attrDef, const TableValues& tableValues,
    const std::string& attributeId)
{
    const auto numRows = tableValues.numRows();
    WIKI_REQUIRE(
        (!attrDef->minValues() || numRows >= attrDef->minValues()) &&
        (!attrDef->maxValues() || numRows <= attrDef->maxValues()),
        ERR_ATTRIBUTE_VALUE_INVALID,
        " Table attribute " << attributeId << " rows count invalid ("
            << tableValues.numRows() << ")");
}

} // namespace wiki
} // namespace maps
