#pragma once

#include "db_helpers.h"
#include "helpers.h"
#include "observers.h"
#include "tests_common.h"

#include <maps/wikimap/mapspro/services/editor/src/serialize/formatter.h>
#include <maps/wikimap/mapspro/services/editor/src/serialize/formatter_context.h>
#include <maps/wikimap/mapspro/services/editor/src/serialize/save_object_parser.h>
#include <maps/wikimap/mapspro/services/editor/src/actions/save_object/saveobject.h>

#include <yandex/maps/wiki/social/feedback/task.h>

#include <string>

namespace maps {
namespace wiki {
namespace tests {

class TestFormatterContext : public FormatterContext
{
public:
    virtual bool needsExpansion(const GeoObject*) { return true; }
};

TOid
getObjectID(const std::string& getObjectResultJSON);


template<typename Controller, typename... Args>
auto
performRequest(ObserverCollection& observers, Args&&... args)
{
    using Request = typename Controller::Request;
    const Request request{std::forward<Args>(args)...};
    return Controller(observers, request)();
}

template<typename Controller, typename... Args>
auto
performRequest(const ObserverCollection& observers, Args&&... args)
{
    using Request = typename Controller::Request;
    const Request request{std::forward<Args>(args)...};
    return Controller(observers, request)();
}

template<typename Controller, typename... Args>
auto
performRequest(ObserverCollection&& observers, Args&&... args)
{
    using Request = typename Controller::Request;
    const Request request{std::forward<Args>(args)...};
    return Controller(std::move(observers), request)();
}

template<typename Controller, typename... Args>
auto
performRequest(Args&&... args)
{
    using Request = typename Controller::Request;
    const Request request{std::forward<Args>(args)...};
    return Controller(request)();
}

std::string
performSaveObjectRequestJsonStr(
    const std::string& jsonRequestBody,
    const ObserverCollection& observers = makeObservers<>(),
    TUid uid = TESTS_USER);

/// Populates DB with data from a file. The file format is guessed by its
/// extension.
///
/// @return Response text to be sent to the frontend.
std::string
performSaveObjectRequest(
    const std::string& requestBodyFilePath,
    const ObserverCollection& observers = makeObservers<>(),
    TUid uid = TESTS_USER);

void
catchLogicException(std::function<void()> f, const std::string& status);

template<typename Response>
json::Value
convertToJson(std::unique_ptr<Response>&& response)
{
    return json::Value::fromString(
        (*Formatter::create(common::FormatType::JSON))(*response)
    );
}

template<typename Controller, typename... Args>
json::Value
performAndValidateJson(Args&&... args)
{
    auto request = typename Controller::Request{std::forward<Args>(args)...};
    auto formatter = Formatter::create(common::FormatType::JSON);
    auto result = (*formatter)(*Controller(makeObservers<>(), request)());
    validateJsonResponse(result, Controller::taskName());
    return json::Value::fromString(result);
}

template<typename Controller, typename... Args>
std::string
performJsonGetRequest(Args&&... args)
{
   auto request = typename Controller::Request{std::forward<Args>(args)...};
   Controller controller(request);
   auto formatter = Formatter::create(common::FormatType::JSON);
   return (*formatter)(*controller());
}

template<typename Controller, typename... Args>
json::Value
performAndValidateJsonGetRequest(Args&&... args)
{
    auto result = performJsonGetRequest<Controller>(std::forward<Args>(args)...);
    validateJsonResponse(result, Controller::taskName());
    return json::Value::fromString(result);
}

template<typename Controller, typename... Args>
std::string
performXmlGetRequest(Args&&... args)
{
   auto request = typename Controller::Request{std::forward<Args>(args)...};
   Controller controller(request);
   auto formatter = Formatter::create(common::FormatType::XML);
   return (*formatter)(*controller());
}

void performObjectsImport(const std::string& jsonPath, const std::string& connectionString);

void setModerationRole(TUid uid, social::ModerationMode moderationMode, TOid aoiId);

social::feedback::Task addFeedbackTask();
social::feedback::Task acquireFeedbackTask(social::TId, social::TUid uid);
void bindCommitToFeedbackTask(TId commitId, TId feedbackTaskId);
void populateACLPermissionsTree();
TUid createRandomUser();
void setUserPermissions(TUid uid, std::list<std::list<std::string>>&& paths);

} // namespace tests
} // namespace wiki
} // namespace maps
