#pragma once

#include <maps/wikimap/mapspro/services/editor/src/tests/helpers/actions/revert_commit.h>
#include <maps/wikimap/mapspro/services/editor/src/tests/helpers/actions/save_object.h>
#include <maps/wikimap/mapspro/services/editor/src/tests/helpers/actions/tasks.h>
#include <maps/wikimap/mapspro/services/editor/src/tests/helpers/controller_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/tests/helpers/objects_creator/objects_creator.h>
#include <maps/wikimap/mapspro/services/editor/src/tests/helpers/tests_common.h>

#include <maps/wikimap/mapspro/services/editor/src/observers/observer.h>

#include <yandex/maps/wiki/social/moderation.h>
#include <yandex/maps/wiki/social/task.h>

#include <maps/libs/enum_io/include/enum_io_fwd.h>


namespace maps::wiki::tests {

enum class UserType {Common, Moderator, Cartographer};
DECLARE_ENUM_IO(UserType);

const auto TRUSTED_USERS = std::array{
    UserType::Moderator,
    UserType::Cartographer
};

enum class RevertMode {Commit, Resolve, Close};
DECLARE_ENUM_IO(RevertMode);

const auto ALL_REVERT_MODES = enum_io::enumerateValues<RevertMode>();


template <typename... Observers>
struct ExtendedEditorFixture: public EditorTestFixture {
    ExtendedEditorFixture()
        : EditorTestFixture()
        , user(createRandomUser())
        , moderator(createRandomUser())
        , cartographer(createRandomUser())
        , userTypeToUid({
              {UserType::Common, user},
              {UserType::Moderator, moderator},
              {UserType::Cartographer, cartographer}
          })
    {
        (observers.add(std::make_shared<Observers>()), ...);

        setUserPermissions(user, {{"mpro"}});
        setUserPermissions(moderator, {{"mpro"}});
        setUserPermissions(cartographer, {{"mpro"}});

        defaultAoiId = saveObject(cartographer, aoi(), observers).objectId;

        setModerationRole(moderator, social::ModerationMode::Moderator, defaultAoiId);
        setModerationRole(cartographer, social::ModerationMode::Supervisor, defaultAoiId);
    }

    social::TId acquire(TUid uid, const social::EventFilter& eventFilter)
    {
        auto txn = db.pool().masterWriteableTransaction();
        const auto result = acquireTask(
            social::Gateway(*txn), uid, eventFilter, cfg()->moderationTimeIntervals()
        );
        txn->commit();
        return result;
    }

    social::TId acquire(TUid uid, TCommitId commitId)
    {
        return acquire(uid, social::EventFilter().commitIds({commitId}));
    }

    auto resolveTask(TUid uid, const social::EventFilter& eventFilter, social::ResolveResolution resolveResolution)
    {
        const auto taskId = acquire(uid, eventFilter);
        return performRequest<SocialModerationTasksResolve>(
            observers, UserContext(uid, {}), defaultAoiId, resolveResolution, social::TaskIds{taskId}
        );
    }

    auto resolveCommitTask(TUid uid, TCommitId commitId, social::ResolveResolution resolveResolution)
    {
        return resolveTask(uid, social::EventFilter().commitIds({commitId}), resolveResolution);
    }

    auto closeTask(TUid uid, const social::EventFilter& eventFilter, social::CloseResolution closeResolution)
    {
        const auto taskId = acquire(uid, eventFilter);
        return performRequest<SocialModerationTasksClose>(
            observers, UserContext(uid, {}), defaultAoiId, closeResolution, social::TaskIds{taskId}
        );
    }

    auto closeCommitTask(TUid uid, TCommitId commitId, social::CloseResolution closeResolution)
    {
        return closeTask(uid, social::EventFilter().commitIds({commitId}), closeResolution);
    }

    RevertResult revert(RevertMode mode, TUid uid, TCommitId commitId)
    {
        switch (mode) {
            case RevertMode::Commit:
                return revertCommit(uid, commitId, observers);

            case RevertMode::Resolve:
                return convertToJson(
                    resolveCommitTask(uid, commitId, social::ResolveResolution::Revert)
                );

            case RevertMode::Close:
                return convertToJson(
                    closeCommitTask(uid, commitId, social::CloseResolution::Revert)
                );
        }
    }

    ObserverCollection observers;
    TOid defaultAoiId;
    const TUid user;
    const TUid moderator;
    const TUid cartographer;
    const std::unordered_map<UserType, TUid> userTypeToUid;
};

} // namespace maps::wiki::tests
