#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/revision_meta/bbox_update_task.h>

#include <maps/wikimap/mapspro/services/editor/src/tests/helpers/controller_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/tests/helpers/helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/tests/helpers/tests_common.h>

#include <yandex/maps/wiki/common/geom.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>

#include <geos/geom/CoordinateArraySequenceFactory.h>
#include <geos/geom/CoordinateSequenceFactory.h>
#include <geos/geom/GeometryFactory.h>
#include <geos/geom/LineString.h>

namespace maps::wiki::tests {

namespace {

const size_t BRANCH_ID = 0;
const size_t HYDRO_PARTS = 30;
const double RADIUS = 100.0;

/**
Emulates import commits.
Revisions (objectId - commitId):
1 - 1 - hydro
2 - 2 - hydro_fc_el
3 - 3 - relation: hydro_fc_el #2 to hydro_fc #62
4 - 4 - hydro_fc_el
5 - 5 - relation: hydro_fc_el #4 to hydro_fc #62
...
60 - 60 - hydro_fc_el
61 - 61 - relation: hydro_fc_el #60 to hydro_fc #62
62 - 62 - hydro_fc
63 - 63 - relation: hydro_fc #62 to hydro #1
2 - 64 - change geometry to inrease bbox
2 - 65 - change geometry to not change bbox
*/
void prepareContourData()
{
    auto work = cfg()->poolCore().masterWriteableTransaction();
    revision::RevisionsGateway gateway(*work);

    revision::DBID lastObjectId = 0;

    auto pair = revision::RevisionsGateway::NewRevisionData(
        revision::RevisionID::createNewID(++lastObjectId),
        revision::ObjectRevision::Data(
            revision::Attributes{{"cat:hydro", "1"}}));
    auto commit = gateway.createCommit({ pair }, TESTS_USER, { { "test", "1" } });
    UNIT_ASSERT_EQUAL(commit.id(), lastObjectId);

    //round lake
    for (size_t i = 0; i < HYDRO_PARTS; i++)
    {
        double startAngle = 2.0 * M_PI * i / HYDRO_PARTS;
        double endAngle = 2.0 * M_PI * (i + 1) / HYDRO_PARTS;

        auto cs = geos::geom::DefaultCoordinateSequenceFactory().create(2);
        cs->setAt(geos::geom::Coordinate(cos(startAngle) * RADIUS, sin(startAngle) * RADIUS), 0);
        cs->setAt(geos::geom::Coordinate(cos(endAngle) * RADIUS, sin(endAngle) * RADIUS), 1);

        common::GeometryPtr lineStringPtr(geos::geom::GeometryFactory::getDefaultInstance()->createLineString(std::move(cs)));

        pair = revision::RevisionsGateway::NewRevisionData(
            revision::RevisionID::createNewID(++lastObjectId),
            revision::ObjectRevision::Data(
                revision::Attributes{{"cat:hydro_fc_el", "1"}},
                std::nullopt,
                common::wkb(lineStringPtr.get())));
        commit = gateway.createCommit({pair}, TESTS_USER, {{"test", "1"}});
        UNIT_ASSERT_EQUAL(commit.id(), i * 2 + 2);

        pair = revision::RevisionsGateway::NewRevisionData(
            revision::RevisionID::createNewID(++lastObjectId),
            revision::ObjectRevision::Data(
                revision::Attributes{
                    {"rel:role", "part"},
                    {"rel:master", "hydro_fc"},
                    {"rel:slave", "hydro_fc_el"}},
                std::nullopt,
                std::nullopt,
                revision::RelationData(HYDRO_PARTS * 2 + 2, i * 2 + 2)));
        commit = gateway.createCommit({pair}, TESTS_USER, {{"test", "1"}});
        UNIT_ASSERT_EQUAL(commit.id(), i * 2 + 3);
    }

    UNIT_ASSERT_EQUAL(lastObjectId, HYDRO_PARTS * 2 + 1);

    pair = revision::RevisionsGateway::NewRevisionData(
        revision::RevisionID::createNewID(++lastObjectId),
        revision::ObjectRevision::Data(
            revision::Attributes{{"cat:hydro_fc", "1"}}));
    commit = gateway.createCommit({pair}, TESTS_USER, {{"test", "1"}});
    UNIT_ASSERT_EQUAL(commit.id(), HYDRO_PARTS * 2 + 2);

    pair = revision::RevisionsGateway::NewRevisionData(
        revision::RevisionID::createNewID(++lastObjectId),
        revision::ObjectRevision::Data(
            revision::Attributes{
                {"rel:role", "fc_part"},
                {"rel:master", "hydro"},
                {"rel:slave", "hydro_fc"}},
            std::nullopt,
            std::nullopt,
            revision::RelationData(1, HYDRO_PARTS * 2 + 2)));
    commit = gateway.createCommit({pair}, TESTS_USER, {{"test", "1"}});
    UNIT_ASSERT_EQUAL(commit.id(), HYDRO_PARTS * 2 + 3);

    work->commit();
}

revision::DBID changeFcEl(revision::RevisionID oldRevisionId, const std::string& wkb)
{
    auto work = cfg()->poolCore().masterWriteableTransaction();
    revision::RevisionsGateway gateway(*work);

    auto pair = revision::RevisionsGateway::NewRevisionData(
        oldRevisionId,
        revision::ObjectRevision::Data(
            revision::Attributes{{"cat:hydro_fc_el", "1"}},
            std::nullopt,
            wkb));
    auto commit = gateway.createCommit({pair}, TESTS_USER, {{"test", "1"}});
    //UNIT_ASSERT_EQUAL(commit.id(), HYDRO_PARTS * 2 + 4);

    work->commit();

    return commit.id();
}

void changeFcElToIncreaseBBox()
{
    double startAngle = 2.0 * M_PI * 0 / HYDRO_PARTS;
    double endAngle = 2.0 * M_PI * (0 + 1) / HYDRO_PARTS;

    auto cs = geos::geom::DefaultCoordinateSequenceFactory().create(3);
    cs->setAt(geos::geom::Coordinate(cos(startAngle) * RADIUS, sin(startAngle) * RADIUS), 0);
    cs->setAt(geos::geom::Coordinate(cos(startAngle) * RADIUS * 5.0, sin(startAngle) * RADIUS * 5.0), 1);
    cs->setAt(geos::geom::Coordinate(cos(endAngle) * RADIUS, sin(endAngle) * RADIUS), 2);

    common::GeometryPtr lineStringPtr(geos::geom::GeometryFactory::getDefaultInstance()->createLineString(std::move(cs)));

    auto commitId = changeFcEl(revision::RevisionID(2, 2), common::wkb(lineStringPtr.get()));
    UNIT_ASSERT_EQUAL(commitId, HYDRO_PARTS * 2 + 4);
}

void changeFcElToNotChangeBBox()
{
    double startAngle = 2.0 * M_PI * 0 / HYDRO_PARTS;
    double endAngle = 2.0 * M_PI * (0 + 1) / HYDRO_PARTS;

    auto cs = geos::geom::DefaultCoordinateSequenceFactory().create(4);
    cs->setAt(geos::geom::Coordinate(cos(startAngle) * RADIUS, sin(startAngle) * RADIUS), 0);
    cs->setAt(geos::geom::Coordinate(cos(startAngle) * RADIUS * 5.0, sin(startAngle) * RADIUS * 5.0), 1);
    cs->setAt(geos::geom::Coordinate(0.0, 0.0), 2);
    cs->setAt(geos::geom::Coordinate(cos(endAngle) * RADIUS, sin(endAngle) * RADIUS), 3);

    common::GeometryPtr lineStringPtr(geos::geom::GeometryFactory::getDefaultInstance()->createLineString(std::move(cs)));

    auto commitId = changeFcEl(revision::RevisionID(2, 64), common::wkb(lineStringPtr.get()));
    UNIT_ASSERT_EQUAL(commitId, HYDRO_PARTS * 2 + 5);
}

void checkBBoxCount(TCommitIds customCommitIds, size_t expectedBBoxCount)
{
    auto work = cfg()->poolCore().masterWriteableTransaction();

    auto startCommitId = customCommitIds.empty() ? 1 : *customCommitIds.begin();

    ComputeBBoxesParams bboxParams{
        BBoxUpdateAction::History,
        BRANCH_ID,
        startCommitId,
        revision::RevisionsGateway(*work).headCommitId(),
        std::move(customCommitIds),
        COMMIT_RANGE_BATCH_SIZE,
        cfg()->editor()->categories().idsByFilter(Categories::CachedGeom),
        10, //thread pool size
        make_unique<ExecutionState>()
    };

    ComputeBBoxTask task(bboxParams);
    task.exec(*work);

    auto r = work->exec("SELECT count(*) FROM revision_meta.geometry");
    UNIT_ASSERT_EQUAL(r[0][0].as<size_t>(), expectedBBoxCount);

    work->commit();
}

void clearBBoxes()
{
    auto work = cfg()->poolCore().masterWriteableTransaction();
    work->exec("TRUNCATE revision_meta.geometry");
    work->commit();
}

}
Y_UNIT_TEST_SUITE(revision_meta)
{
WIKI_FIXTURE_TEST_CASE(test_revision_meta, EditorTestFixture)
{
    prepareContourData();

    INFO() << "Try process all commits";
    checkBBoxCount({}, 1);
    clearBBoxes();

    INFO() << "Try custom commit 2";
    checkBBoxCount({2}, 1);
    clearBBoxes();

    INFO() << "Try custom commit 3";
    checkBBoxCount({3}, 1);
    clearBBoxes();

    changeFcElToIncreaseBBox();

    INFO() << "Process all commits";
    checkBBoxCount({}, 2);
    clearBBoxes();

    INFO() << "Try custom commit 2";
    checkBBoxCount({2}, 2);
    clearBBoxes();

    INFO() << "Try custom commit 3";
    checkBBoxCount({3}, 2);
    clearBBoxes();

    INFO() << "Try custom commit 64";
    checkBBoxCount({64}, 1);

    changeFcElToNotChangeBBox();

    INFO() << "Try custom commit 65";
    checkBBoxCount({65}, 1);
}
} // Y_UNIT_TEST_SUITE

} // namespace maps::wiki::tests
