#include "stable_branch_builder.h"
#include <maps/wikimap/mapspro/services/editor/src/sync/branch_mask.h>
#include <maps/wikimap/mapspro/services/editor/src/sync/sync_objects.h>
#include <maps/wikimap/mapspro/services/editor/src/sync/sync_params.h>
#include <maps/wikimap/mapspro/services/editor/src/sync/db_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/approved_commits/approver.h>
#include <maps/wikimap/mapspro/services/editor/src/approved_commits/processor.h>
#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/pool_sections.h>

#include <maps/wikimap/mapspro/libs/views/include/magic_strings.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/revision/exception.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>

#include <boost/lexical_cast.hpp>
#include <boost/program_options.hpp>

#include <iostream>
#include <fstream>
#include <string>
#include <vector>

namespace po = boost::program_options;
namespace rev = maps::wiki::revision;
namespace mwt = maps::wiki::tool;
namespace mws = maps::wiki::sync;

namespace {

const std::string OPT_HELP = "help";
const std::string OPT_CONFIG = "config";
const std::string OPT_LOG_LEVEL = "log-level";
const std::string OPT_BRANCH = "branch";
const std::string OPT_ALL_OBJECTS = "all-objects";
const std::string OPT_COMMITS = "commits";
const std::string OPT_COMMITS_FILE = "commits-file";
const std::string OPT_IDS = "ids";
const std::string OPT_IDS_FILE = "ids-file";
const std::string OPT_CATEGORY_GROUPS = "category-groups";
const std::string OPT_CATEGORIES = "categories";
const std::string OPT_VIEW_TABLES = "view-tables";
const std::string OPT_STAGES = "stages";
const std::string OPT_BATCH_SIZE = "batch-size";
const std::string OPT_THREADS = "threads";
const std::string OPT_ACTION = "action";
const std::string OPT_USER_ID = "user-id";
const std::string OPT_BIG_DATA = "big-data";
const std::string OPT_SET_PROGRESS_STATE = "set-progress-state";
const std::string OPT_BRANCH_EXCLUSIVE_LOCK = "branch-exclusive-lock";
const std::string OPT_SINGLE_COMMITS_MODE = "single-commits-mode";

const std::string CONFIG_POOL_NAME = "editor-tool";
const std::string DEFAULT_CONFIG_PATH = "/etc/yandex/maps/wiki/services/services.xml";
const std::string DEFAULT_CATEGORY_GROUPS = mws::STR_ALL_CATEGORY_GROUPS;
const std::string DEFAULT_VIEW_TABLES = mws::STR_ALL_VIEW_TABLES;
const size_t DEFAULT_ALL_OBJECTS = 0;
const size_t DEFAULT_SET_PROGRESS_STATE = 0;
const size_t DEFAULT_BRANCH_EXCLUSIVE_LOCK = 0;

const int EXIT_CODE_SUCCESS             = 0;
const int EXIT_CODE_CONCURRENCY_ERROR   = 1;
const int EXIT_CODE_ERROR               = 2;

inline std::string trunkBranchAsString()
{
    return boost::lexical_cast<std::string>(rev::BranchType::Trunk);
}

std::string toPrettyList(const std::string& name, const std::vector<std::string>& options)
{
    return name + '\n' + maps::wiki::common::join(options, '\n');
}

} // namespace

namespace maps::wiki::tool {

namespace {

const std::string ACTION_REFRESH = "refresh";
const std::string ACTION_CREATE_STABLE = "create-stable";
const std::string ACTION_ADD_FOR_APPROVE = "add-for-approve";
const std::string ACTION_PROCESS_APPROVED_COMMITS_QUEUE = "process-approved-commits-queue";
const std::string ACTION_PROCESS_APPROVED_LABELS_QUEUE = "process-approved-labels-queue";
const std::string ACTION_PROCESS_APPROVED_BBOXES_QUEUE = "process-approved-bboxes-queue";
const std::string ACTION_CLEANUP_BRANCH_MASK = "cleanup-branch-mask";

const std::vector<std::string> ACTIONS_LIST {
    ACTION_REFRESH,
    ACTION_CREATE_STABLE,
    ACTION_ADD_FOR_APPROVE,
    ACTION_PROCESS_APPROVED_COMMITS_QUEUE,
    ACTION_PROCESS_APPROVED_LABELS_QUEUE,
    ACTION_PROCESS_APPROVED_BBOXES_QUEUE,
    ACTION_CLEANUP_BRANCH_MASK,
};

const std::string& DEFAULT_ACTION = mwt::ACTION_REFRESH;

const std::vector<std::string> STAGES_LIST {
    mws::STAGES_ALL,
    mws::STAGES_VIEW,
    mws::STAGES_ATTRS,
    mws::STAGES_SUGGEST,
    mws::STAGES_LABELS,
    mws::STAGES_BBOX,
    mws::STAGES_BBOX_VIEW,
    mws::STAGES_BBOX_HISTORY,
};

const std::string& DEFAULT_STAGES = mws::STAGES_ALL;


StringSet allowedExtraPoolSections(
    const std::string& action,
    const std::string& branchStr)
{
    if (action == ACTION_ADD_FOR_APPROVE) {
        return { POOL_SECTION_NAME_SOCIAL };
    }

    auto isBranchTrunk = [&] {
        return
            branchStr == trunkBranchAsString() ||
            branchStr == std::to_string(revision::TRUNK_BRANCH_ID);
    };

    if (action == ACTION_CLEANUP_BRANCH_MASK ||
        action == ACTION_PROCESS_APPROVED_COMMITS_QUEUE ||
        action == ACTION_PROCESS_APPROVED_BBOXES_QUEUE ||
        action == ACTION_CREATE_STABLE ||
        action == ACTION_PROCESS_APPROVED_LABELS_QUEUE ||
        !isBranchTrunk()) // ACTION_REFRESH !trunk
    {
        return {
            POOL_SECTION_NAME_VIEW_STABLE,
            POOL_SECTION_NAME_LABELS_STABLE
        };
    }

    // ACTION_REFRESH trunk
    return {
        POOL_SECTION_NAME_VIEW_TRUNK,
        POOL_SECTION_NAME_LABELS_TRUNK
    };
}


class ExecutionStateCheckBranch : public ExecutionState
{
public:
    ExecutionStateCheckBranch(TBranchId branchId)
        : branchId_(branchId)
    {}

    bool checkOk(Transaction& workCore) override
    {
        if (isOk()) {
            auto branch = revision::BranchManager(workCore).load(branchId_);
            if (branch.state() == revision::BranchState::Normal) {
                return true;
            }
            if (branch.state() == revision::BranchState::Progress
                && hasStabeBranchCreationAttribute(branch)) {
                return true;
            }
            cancel = true;
        }
        return false;
    }

private:
    const TBranchId branchId_;
};

template <class T>
T
loadParam(
    const po::variables_map& vm,
    const std::string& name,
    const T& defaultValue = T())
{
    if (vm.count(name)) {
        return vm[name].as<T>();
    } else {
        return defaultValue;
    }
}

TCommitIds
loadCommitIds(const po::variables_map& vm)
{
    auto commitIds = splitCast<TCommitIds>(
        loadParam<std::string>(vm, OPT_COMMITS), ',');

    auto commitsFile = loadParam<std::string>(vm, OPT_COMMITS_FILE);
    if (!commitsFile.empty()) {
        std::ifstream file(commitsFile);
        for (TCommitId id; file >> id; ) {
            commitIds.insert(id);
        }
    }
    return commitIds;
}

TOIds
loadObjectIds(const po::variables_map& vm)
{
    auto objectIds = splitCast<TOIds>(loadParam<std::string>(vm, OPT_IDS), ',');

    auto objectIdsFile = loadParam<std::string>(vm, OPT_IDS_FILE);
    if (!objectIdsFile.empty()) {
        std::ifstream file(objectIdsFile);
        for (TOid id; file >> id; ) {
            objectIds.insert(id);
        }
    }
    return objectIds;
}

void
processActionAddForApprove(const po::variables_map& vm)
{
    auto uid = loadParam<TUid>(vm, OPT_USER_ID);
    REQUIRE(uid, "must provide " << OPT_USER_ID << " param");
    auto commits = loadCommitIds(vm);
    REQUIRE(!commits.empty(), "must provide " <<
            OPT_COMMITS << " or " << OPT_COMMITS_FILE << " param");

    auto branchCtx = BranchContextFacade().acquireWrite();
    auto user = acl::ACLGateway(branchCtx.txnCore()).user(uid);
    approved_commits::CommitsApprover approver(branchCtx, user, commits);
    auto closedTaskIds = approver.closeTasks(social::CloseResolution::Approve);

    INFO() << "pushed commits to pre approve queue: "
           << common::join(approver.commitIds(), ',');
    INFO() << "closed moderation tasks: "
           << common::join(closedTaskIds, ',');

    branchCtx.commit();
}

ExecutionStatePtr
createStableBranch(const po::variables_map& vm)
{
    auto executionState = std::make_shared<ExecutionState>();

    const auto uid = loadParam<TUid>(vm, OPT_USER_ID);
    REQUIRE(uid, "must provide " << OPT_USER_ID << " param");

    auto threads = loadParam<size_t>(vm, OPT_THREADS, sync::AUTO_THREAD_COUNT);

    StableBranchBuilder builder(executionState, uid, threads);
    builder.run();

    return executionState;
}

ExecutionStatePtr
processApprovedCommitsQueue(
    const po::variables_map& vm, revision_meta::ApprovedQueueMode mode)
{
    sync::BranchLocker branchLocker(sync::masterCoreConnectionString());

    auto& work = branchLocker.work();
    revision::BranchManager branchManager(work);

    auto branches = branchManager.load({{revision::BranchType::Approved, 1}});
    if (branches.empty()) {
        INFO() << "approved branch not exists";
        return std::make_shared<ExecutionState>();
    }

    const auto& branch = branches.front();

    auto executionState =
        std::make_shared<ExecutionStateCheckBranch>(branch.id());

    approved_commits::Processor processor(
        branchLocker,
        branch,
        loadParam<size_t>(vm, OPT_THREADS, sync::AUTO_THREAD_COUNT),
        sync::DEFAULT_COMMIT_RANGE_BATCH_SIZE);
    if (processor.tryProcess(mode, executionState)) {
        work.commit();
    }
    return executionState;
}

sync::CustomIds
createCustomIds(const po::variables_map& vm)
{
    sync::CustomIds customIds{loadCommitIds(vm), loadObjectIds(vm)};

    bool allObjectsMode = loadParam<size_t>(vm, OPT_ALL_OBJECTS) == 1;
    if (allObjectsMode) {
        REQUIRE(customIds.empty(), "--" << OPT_ALL_OBJECTS << "=1 conflict with ids/commits options");
    } else {
        REQUIRE(!customIds.empty(), "None of ids/commits options is specified. "
            << "Use --" << OPT_ALL_OBJECTS << " to rebuild all data");
    }

    return customIds;
}

ExecutionStatePtr
processActionRefresh(const po::variables_map& vm, const std::string& branchStr)
{
    auto executionState = std::make_shared<ExecutionState>();
    auto runner = [&](sync::CustomIds customIds)
    {
        sync::SyncParams syncParams(
            executionState,
            loadParam<std::string>(vm, OPT_STAGES, mwt::DEFAULT_STAGES),
            loadParam<std::string>(vm, OPT_CATEGORY_GROUPS, DEFAULT_CATEGORY_GROUPS),
            loadParam<std::string>(vm, OPT_CATEGORIES),
            loadParam<std::string>(vm, OPT_VIEW_TABLES, DEFAULT_VIEW_TABLES),
            branchStr,
            loadParam<size_t>(vm, OPT_BRANCH_EXCLUSIVE_LOCK) > 0
                ? revision::Branch::LockType::Exclusive
                : revision::Branch::LockType::Shared,
            loadParam<size_t>(vm, OPT_SET_PROGRESS_STATE) > 0
                ? sync::SetProgressState::Yes
                : sync::SetProgressState::No,
            loadParam<size_t>(vm, OPT_BATCH_SIZE, sync::DEFAULT_BATCH_SIZE),
            loadParam<size_t>(vm, OPT_THREADS, sync::AUTO_THREAD_COUNT),
            loadParam<bool>(vm, OPT_BIG_DATA)
                ? sync::BigData::Yes
                : sync::BigData::No);
        sync::BranchLocker branchLocker(sync::masterCoreConnectionString());
        sync::SyncObjects controller(std::move(syncParams), branchLocker);
        controller.run(std::move(customIds));
        branchLocker.work().commit();
    };

    auto customIds = createCustomIds(vm);

    if (loadParam<bool>(vm, OPT_SINGLE_COMMITS_MODE)) {
        REQUIRE(customIds.objectIds.empty(), "object ids must be empty");
        REQUIRE(!customIds.commitIds.empty(), "commit ids must be non-empty");
        for (auto commitId : customIds.commitIds) {
            ASSERT(commitId);
            runner(TCommitIds{commitId});
        }
    } else {
        runner(std::move(customIds));
    }
    return executionState;
}

void
run(const po::variables_map& vm)
{
    auto configPath = loadParam<std::string>(vm, OPT_CONFIG, DEFAULT_CONFIG_PATH);
    auto logLevel = loadParam<std::string>(vm, OPT_LOG_LEVEL);
    if (!logLevel.empty()) {
        log8::setLevel(logLevel);
    }

    const auto branchStr = loadParam<std::string>(vm, OPT_BRANCH, trunkBranchAsString());
    const auto action = loadParam<std::string>(vm, OPT_ACTION, mwt::DEFAULT_ACTION);

    ConfigScope config(
        configPath,
        POOL_SECTION_NAME_LONG_READ,
        CONFIG_POOL_NAME,
        allowedExtraPoolSections(action, branchStr),
        LongReadAccess::Try);
    cfg()->onLoad();
    if (logLevel.empty()) {
        log8::setLevel(cfg()->logLevel());
    }

    REQUIRE(
        std::count(ACTIONS_LIST.begin(), ACTIONS_LIST.end(), action),
        "Invalid action " << action
    );

    if (action == ACTION_ADD_FOR_APPROVE) {
        processActionAddForApprove(vm);
        return;
    }

    if (action == ACTION_CLEANUP_BRANCH_MASK) {
        sync::cleanUpBranchMaskTable();
        return;
    }

    ExecutionStatePtr executionState;

    if (action == ACTION_CREATE_STABLE) {
        executionState = createStableBranch(vm);
    } else if (action == ACTION_PROCESS_APPROVED_COMMITS_QUEUE) {
        executionState = processApprovedCommitsQueue(vm, revision_meta::ApprovedQueueMode::ViewAttrs);
    } else if (action == ACTION_PROCESS_APPROVED_LABELS_QUEUE) {
        executionState = processApprovedCommitsQueue(vm, revision_meta::ApprovedQueueMode::Labels);
    } else if (action == ACTION_PROCESS_APPROVED_BBOXES_QUEUE) {
        executionState = processApprovedCommitsQueue(vm, revision_meta::ApprovedQueueMode::Bboxes);
    } else { //ACTION_REFRESH
        executionState = processActionRefresh(vm, branchStr);
    }

    REQUIRE(!executionState->fail,
        "execution state"
            << ", canceled: " << executionState->cancel
            << ", failed: " << executionState->fail
    );
}

} // namespace
} // namespace maps::wiki::tool

int
main(int argc, char *argv[])
{
    using namespace maps::wiki;

    po::options_description desc("Allowed options");
    desc.add_options()
        (OPT_HELP.c_str(),
            "produce help message")
        (OPT_CONFIG.c_str(), po::value<std::string>(),
            "config path (default: path to services.xml)")
        (OPT_LOG_LEVEL.c_str(), po::value<std::string>(),
            "log level (default: info, recommended: warn)")
        (OPT_BRANCH.c_str(), po::value<std::string>()->default_value(trunkBranchAsString()),
            "revision branch (trunk|approved|stable|<id>)")
        (OPT_ALL_OBJECTS.c_str(), po::value<size_t>()->default_value(DEFAULT_ALL_OBJECTS),
            "clear and rebuild all data (1|0)")
        (OPT_COMMITS.c_str(), po::value<std::string>(),
            "comma separated commit id list")
        (OPT_COMMITS_FILE.c_str(), po::value<std::string>(),
            "file with commits id list")
        (OPT_SINGLE_COMMITS_MODE.c_str(), po::value<bool>()->default_value(false),
            "sync view by one commit")
        (OPT_IDS.c_str(), po::value<std::string>(),
            "comma separated object id list")
        (OPT_IDS_FILE.c_str(), po::value<std::string>(),
            "file with objects id list")
        (OPT_BATCH_SIZE.c_str(), po::value<size_t>(),
            "batch size (default: view:1000, attrs:500, labels:100")
        (OPT_THREADS.c_str(), po::value<size_t>(),
            "working threads (default: max write connections)")
        (OPT_CATEGORY_GROUPS.c_str(), po::value<std::string>()->default_value(DEFAULT_CATEGORY_GROUPS),
            "comma delimited category groups list for syncing (rd_group,ad_group,service_group ...)")
        (OPT_CATEGORIES.c_str(), po::value<std::string>(),
            "comma delimited categories list for syncing (rd,rd_el,rd_jc,aoi ...)")
        (OPT_VIEW_TABLES.c_str(), po::value<std::string>()->default_value(DEFAULT_VIEW_TABLES),
            ("view tables for syncing (" + common::join(views::VIEW_TABLES, ',') + ")").c_str())
        (OPT_ACTION.c_str(), po::value<std::string>()->default_value(mwt::DEFAULT_ACTION),
            toPrettyList("Possible actions:", mwt::ACTIONS_LIST).c_str())
        (OPT_USER_ID.c_str(), po::value<TUid>(),
            ("using with " + OPT_ACTION + "=" + mwt::ACTION_CREATE_STABLE + "|" +
                                                mwt::ACTION_ADD_FOR_APPROVE).c_str())
        (OPT_STAGES.c_str(), po::value<std::string>()->default_value(mwt::DEFAULT_STAGES),
            toPrettyList("Sync stages:", mwt::STAGES_LIST).c_str())
        (OPT_BIG_DATA.c_str(), po::value<bool>()->default_value(false),
            "big data (1 - force wait lock for bbox processing)")
        (OPT_SET_PROGRESS_STATE.c_str(),
            po::value<size_t>()->default_value(DEFAULT_SET_PROGRESS_STATE),
            "set progress state for active branch (1|0)")
        (OPT_BRANCH_EXCLUSIVE_LOCK.c_str(),
            po::value<size_t>()->default_value(DEFAULT_BRANCH_EXCLUSIVE_LOCK),
            "lock active branch in exclusive mode (1|0)");

    try {
        po::variables_map vm;
        po::parsed_options parsed = po::parse_command_line(argc, argv, desc);
        po::store(parsed, vm);
        po::notify(vm);

        auto unrecognized =  po::collect_unrecognized(parsed.options, po::include_positional);

        if (argc > 1 && !vm.count(OPT_HELP) && unrecognized.empty()) {
            tool::run(vm);
            return EXIT_CODE_SUCCESS;
        }

        if (!unrecognized.empty()) {
            std::cerr << "Unrecognized parameters: ["
                << common::join(unrecognized, ", ")
                << "]" << std::endl;
        }
        std::cerr << "Usage: wiki-editor-tool <options>" << std::endl;
        std::cerr << desc << std::endl;
    }
    catch (const rev::BranchAlreadyLockedException& e) {
        std::cerr << e << std::endl;
        return EXIT_CODE_CONCURRENCY_ERROR;
    }
    catch (const maps::Exception& e) {
        std::cerr << e << std::endl;
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
    return EXIT_CODE_ERROR;
}
