#include "stable_branch_builder.h"
#include "copy_schema.h"
#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/sync/create_schema.h>
#include <maps/wikimap/mapspro/services/editor/src/approved_commits/processor.h>

#include <yandex/maps/wiki/common/pg_retry_helpers.h>
#include <yandex/maps/wiki/common/retry_duration.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/commit_manager.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <maps/wikimap/mapspro/libs/revision_meta/include/commit_regions.h>

#include <maps/libs/log8/include/log8.h>

namespace maps::wiki::tool {

namespace {

void setApprovedBranchToNormalState(revision::Branch& approvedBranch)
{
    common::execCommitWithRetries(
        cfg()->poolCore(),
        "set normal state for approved branch",
        {},
        {},
        [&] (pqxx::transaction_base& txn) {
            approvedBranch.concatenateAttributes(txn, {{STABLE_BRANCH_CREATION_ATTR, "0"}});
            approvedBranch.setState(txn, revision::BranchState::Normal);
        }
    );
}

void setApprovedBranchToProgressState(revision::Branch& approvedBranch)
{
    common::execCommitWithRetries(
        cfg()->poolCore(),
        "set progress state for approved branch",
        {},
        {},
        [&] (pqxx::transaction_base& txn) {
            approvedBranch.concatenateAttributes(txn, {{STABLE_BRANCH_CREATION_ATTR, "1"}});
            if (approvedBranch.state() == revision::BranchState::Normal) {
                approvedBranch.setState(txn, revision::BranchState::Progress);
            }
        }
    );
}

revision::Branch prepareApprovedBranch()
{
    auto branches = common::retryDuration([&] {
        auto work = sync::masterCoreTransaction(AccessMode::ReadOnly);

        revision::BranchManager branchManager(*work);
        return branchManager.load({
            {revision::BranchType::Approved, 1},
            {revision::BranchType::Stable, 1}});
    });

    REQUIRE(!branches.empty() && branches.front().type() == revision::BranchType::Approved,
        "approved branch not found");
    REQUIRE(branches.back().type() != revision::BranchType::Stable,
        "stable branch already exists");

    auto approvedBranch = branches.front();
    REQUIRE(approvedBranch.state() != revision::BranchState::Unavailable,
        "approved branch id:" << approvedBranch.id()
        << " is not normal state: " << approvedBranch.state());

    setApprovedBranchToProgressState(approvedBranch);

    return approvedBranch;
}

} // namespace

StableBranchBuilder::StableBranchBuilder(
        const ExecutionStatePtr& executionState,
        TUid uid,
        size_t threads)
    : executionState_(executionState)
    , uid_(uid)
    , threads_(threads)
{
}

void StableBranchBuilder::run() const
{
    // Check branch existence and set attribute to disable sync
    auto approvedBranch = prepareApprovedBranch();

    auto threads = threads_ ? threads_ : sync::maxThreadsCount(approvedBranch.id());

    auto stableBranch = syncApprovedAndPrepareStableBranch(approvedBranch, threads);
    ASSERT(stableBranch.state() == revision::BranchState::Unavailable);

    copyDataFromBranch(executionState_, approvedBranch, stableBranch, threads);
    ASSERT(stableBranch.state() != revision::BranchState::Progress);

    setApprovedBranchToNormalState(approvedBranch);
}

revision::Branch StableBranchBuilder::syncApprovedAndPrepareStableBranch(
    revision::Branch& approvedBranch,
    size_t threads) const
{
    sync::BranchLocker branchLocker(sync::masterCoreConnectionString());

    // Sync approved branch (lock TO_APPROVE_LOCK_ID is set)
    approved_commits::Processor processor(
        branchLocker, approvedBranch, threads, sync::DEFAULT_COMMIT_RANGE_BATCH_SIZE);
    processor.waitAndProcess(revision_meta::ApprovedQueueMode::ViewAttrs);
    processor.waitAndProcess(revision_meta::ApprovedQueueMode::Labels);
    processor.waitAndProcess(revision_meta::ApprovedQueueMode::Bboxes);

    branchLocker.lockBranchPersistently(approvedBranch, revision::Branch::LockType::Exclusive);

    // Renew branch after lock
    revision::BranchManager branchManager(branchLocker.work());
    approvedBranch = branchManager.loadApproved();

    ASSERT(approvedBranch.state() == revision::BranchState::Progress);
    ASSERT(approvedBranch.type() != revision::BranchType::Deleted);

    // Create stable branch, merge commits and create view schema
    return prepareStableBranch(approvedBranch, branchLocker);
}

revision::Branch StableBranchBuilder::prepareStableBranch(
    const revision::Branch& approvedBranch,
    const sync::BranchLocker& branchLocker) const
{
    auto work = sync::masterCoreTransaction(AccessMode::ReadWrite);

    revision::RevisionsGateway gateway(*work, approvedBranch);
    auto headCommitId = gateway.headCommitId();

    revision::BranchManager branchManager(*work);
    auto stableBranch = branchManager.createStable(uid_, {});
    INFO() << "Created stable branch, id: " << stableBranch.id();

    revision::CommitManager commitManager(*work);
    auto mergedCommits = commitManager.mergeApprovedToStable(headCommitId);
    INFO() << "Merged approved to stable branch: " << mergedCommits.size()
        << " commits, head-commit-id: " << headCommitId;
    ASSERT(stableBranch.state() == revision::BranchState::Unavailable);

    revision_meta::CommitRegions(*work).toStable(stableBranch, mergedCommits);

    ASSERT(createViewSchemaIfNotExists(stableBranch, sync::SchemaNameType::Temporary));
    ASSERT(createLabelsSchemaIfNotExists(stableBranch, sync::SchemaNameType::Temporary));

    branchLocker.lockBranchPersistently(stableBranch, revision::Branch::LockType::Exclusive);

    work->commit(); //create branch and merge commits
    return stableBranch;
}

} // namespace maps::wiki::tool
