#include "split_linear_element_callback.h"

#include <maps/wikimap/mapspro/services/editor/src/collection.h>
#include <maps/wikimap/mapspro/services/editor/src/objects_cache.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>
#include <maps/wikimap/mapspro/services/editor/src/topo_storage.h>
#include <maps/wikimap/mapspro/services/editor/src/relations_processor.h>
#include "topo_relations_processor.h"

#include <yandex/maps/wiki/common/geom.h>

#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/closest_point.h>

#include <boost/optional/optional_io.hpp>

#include <algorithm>
#include <sstream>


namespace maps {
namespace wiki {

void
SplitLinearElementCallback::processRequest(topo::SplitEdgeRequest& request)
    const
{
    DEBUG() << "Split edge request, edge id: " << request.sourceId();
    for (size_t i = 0; i < request.splitPoints().size(); ++i) {
        auto geoPoint = common::mercatorToGeodetic(
            request.splitPoints()[i]->geom.x(),
            request.splitPoints()[i]->geom.y()
        );
        DEBUG() << "SplitPoint: [" << geoPoint.x() << ", " << geoPoint.y() << "] ";
    }

    const topo::SplitPolylinePtrVector& splitPolylines = request.splitPolylines();

    auto index = boost::make_optional<topo::EdgeID>(false, {});
    double maxLength = 0;

    // Here realy tricky algorithm of choosing part to keep id.
    // We try to choose the longest part,
    // but shared one is less preferable than part which is not shared.
    for (size_t i = 0; i < request.splitPolylines().size(); ++i) {
        const topo::SplitPolylinePtr& current = splitPolylines[i];
        if (current->edgeId) { // edge is reserved already
            continue;
        }
        const double length = geolib3::length(current->geom);
        if (!index
                || (!current->isShared && splitPolylines[*index]->isShared)
                || (current->isShared == splitPolylines[*index]->isShared && maxLength < length))
        {
            maxLength = length;
            index = i;
        }
    }

    // Possible that all parts are reserved (edge is fully overlapped by other one).
    if (index) {
        request.selectPartToKeepID(*index);
    }
}

void
SplitLinearElementCallback::processEvent(const topo::SplitEdgeEvent& event) const
{
    REQUIRE(
        event.partToKeepID() || cache_.getExisting(event.sourceId()),
        "It must be guaranteed that some part keeps id or edge exists."
        // Namely, if edge doesn't exist then it is edited.
        // Implementation of the topolib and request processing guarantees that
        // there is edited part of edge which keeps id.
    );

    DEBUG() << "Split edge event, edge id: " << event.sourceId();
    if (event.partToKeepID()) {
        DEBUG() << "Part keeping id: " << event.partToKeepID();
    } else {
        DEBUG() << "Edge is merged, no part keeping id";
    }

    bool isNewAndPrimary = false;
    bool sourceIsPrimary = false;
    if (!event.sourceExists()) {
        const auto it = linearElementsData_.find(event.sourceId());
        REQUIRE(it != linearElementsData_.end(),
            "New linear element data not found, source edge id " << event.sourceId());
        isNewAndPrimary = it->second.isPrimary;
    }
    sourceIsPrimary =
        isNewAndPrimary ||
        (event.sourceExists() &&
        cache_.getExisting(event.sourceId())->primaryEdit());

    double totalLength = 0.0;
    std::vector<GeoObject*> parts;
    std::vector<LinearElement*> leParts;
    parts.reserve(event.edgeIds().size());
    leParts.reserve(event.edgeIds().size());
    for (size_t i = 0; i < event.edgeIds().size(); ++i) {
        const TOid leId = event.edgeIds()[i];
        DEBUG() << "Edge part, index: " << i << ", id:" << leId;
        LinearElement* le = as<LinearElement>(cache_.getExisting(leId));
        if (isNewAndPrimary) {
            le->primaryEdit(i == event.partToKeepID());
        } else {
            le->setAffectedBySplitId(
                event.partToKeepID()
                ? event.edgeIds()[*event.partToKeepID()]
                : 0);
        }
        if (sourceIsPrimary) {
            le->setPartOfPrimary();
        }
        parts.push_back(le);
        leParts.push_back(le);
        totalLength += le->geom().realLength();
    }

    ObjectPtr source = event.sourceExists()
        ? cache_.getExisting(event.sourceId())
        : ObjectPtr();
    Attributes attrs = source
        ? source->attributes()
        : leParts[*event.partToKeepID()]->attributes();

    double offsetFromA = 0.0;
    double offsetFromB = totalLength;
    size_t totalParts = leParts.size();
    for (size_t partNum = 0; partNum < totalParts; ++partNum) {
        auto& le = leParts[partNum];
        double curLength = le->geom().realLength();
        offsetFromB -= curLength;
        le->setJunctionSpecificAttributes(offsetFromA, offsetFromB, attrs, partNum, totalParts);
        offsetFromA += curLength;
    }

    TOIds mastersToUpdate;
    if (event.sourceExists()) {
        for (const auto& rInfo : source->masterRelations().range()) {
            mastersToUpdate.insert(rInfo.id());
        }
    }

    RelationsProcessor processor(cache_);
    if (source) {
        processor.replaceWithParts(
            RelationType::Master, source.get(), parts
        );
    } else {
        processor.replaceWithParts(
            RelationType::Master, leParts[*event.partToKeepID()], parts
        );
    }

    if (!mastersToUpdate.empty()) {
        TopoRelationsProcessor topoProcessor(cache_);
        GeoObjectCollection toUpdateSeqNums = cache_.get(mastersToUpdate);
        for (const auto& objPtr : toUpdateSeqNums) {
            topoProcessor.updateSeqNums(objPtr.get());
        }
    }
}

} // namespace wiki
} // namespace maps
