#include "topo_cache.h"

#include <maps/wikimap/mapspro/services/editor/src/objects_cache.h>
#include <maps/wikimap/mapspro/services/editor/src/revisions_facade.h>
#include <maps/wikimap/mapspro/services/editor/src/relations_manager.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>

#include <yandex/maps/wiki/configs/editor/topology_groups.h>
#include <yandex/maps/wiki/topo/editor.h>

#include <maps/libs/geolib/include/contains.h>
#include <maps/libs/geolib/include/intersection.h>

#include <iomanip>

namespace maps {
namespace wiki {

namespace {

std::string
bboxToStr(const geolib3::BoundingBox& bbox)
{
    std::ostringstream os;
    os << std::setprecision(DOUBLE_FORMAT_PRECISION);
    os << "[" << bbox.minX() << ", " << bbox.minY()
       << ", " << bbox.maxX() << ", " << bbox.maxY() << "]";
    return os.str();
}

} // namespace

TopoCache::TopoCache(
        const TopologyGroup& topoGroup,
        ObjectsCache& cache,
        geolib3::BoundingBox region)
    : topoGroup_(topoGroup)
    , jcToLeRoles_({topoGroup_.startJunctionRole(), topoGroup_.endJunctionRole()})
    , cache_(cache)
{
    tolerance_ = std::max(
        topoGroup_.maxJunctionGravity(),
        topoGroup_.tolerance()
    );

    region_ = geolib3::resizeByValue(
        region,
        topo::Editor::maxInteractionDistance(tolerance_) + tolerance_);

    DEBUG() << "Preload region: " << bboxToStr(region_);
    preload();
    DEBUG() << "Preload region: " << bboxToStr(region_) << " end";
}

void
TopoCache::preload()
{
    auto categoryIds = topoGroup_.linearElementsCategories();
    auto op = revision::filters::GeomFilterExpr::Operation::IntersectsLinestrings;

    const auto& jcCategoryId = topoGroup_.junctionsCategory();
    const auto& jcCategory = cfg()->editor()->categories()[jcCategoryId];

    auto jcHasAdditionalMasters =
        jcCategory.masterRoleIds(roles::filters::All) != jcToLeRoles_;
    if (jcHasAdditionalMasters) {
        categoryIds.insert(jcCategoryId);
        op = op | revision::filters::GeomFilterExpr::Operation::IntersectsPoints;
    }

    TOIds linearElementIds;
    {
        auto ids = RevisionsFacade::getIdsByBBox(
            cache_.revisionsFacade().snapshot(), region_, categoryIds, op);

        for (const auto& obj : cache_.get(ids)) {
            if (obj->geom().isNull()) {
                continue;
            }
            cachedObjects_.insert(
                {obj->id(),
                 {obj->categoryId(), boundingBox(obj->geom()), /* bool deleted = */ false}});
            if (is<LinearElement>(obj)) {
                linearElementIds.insert(obj->id());
            }
        }
    }

    cache_.relationsManager().loadRelations(
        RelationType::Slave, linearElementIds, jcToLeRoles_);

    if (jcHasAdditionalMasters) {
        return;
    }
    auto elements = cache_.get(linearElementIds);
    for (auto& el : elements) {
        LinearElement* le = as<LinearElement>(el);
        for (const auto& jc: cache_.get(TOIds{le->junctionAId(), le->junctionBId()})) {
            cachedObjects_.insert(
                {jc->id(),
                 {jc->categoryId(), boundingBox(jc->geom()), /* bool deleted = */ false}});
        }
    }
}

TOIds
TopoCache::nodeIdsByBBox(const geolib3::BoundingBox& bbox) const
{
    return idsByBBox(bbox, {topoGroup_.junctionsCategory()});
}

TOIds
TopoCache::edgeIdsByBBox(const geolib3::BoundingBox& bbox) const
{
    return idsByBBox(bbox, topoGroup_.linearElementsCategories());
}

TOIds
TopoCache::idsByBBox(
    const geolib3::BoundingBox& bbox,
    const StringSet& categoryIds) const
{
    REQUIRE(
        geolib3::contains(region_, bbox),
        "Requested bbox " << bboxToStr(bbox) << " is not preloaded");

    TOIds result;
    auto expBBox = geolib3::resizeByValue(bbox, tolerance_);
    for (const auto& cachedObj : cachedObjects_) {
        const auto& data = cachedObj.second;
        if (data.deleted || !categoryIds.count(data.categoryId)) {
            continue;
        }
        if (geolib3::intersects(expBBox, data.bbox)) {
            result.insert(cachedObj.first);
        }
    }
    return result;
}

void
TopoCache::addObject(TOid id)
{
    auto obj = cache_.getExisting(id);
    REQUIRE(
        cachedObjects_.insert(
            {obj->id(),
            {
                obj->categoryId(),
                boundingBox(obj->geom()),
                /* bool deleted = */ false
            }}
        ).second,
        "Object with id " << id << " already exists in topo cache");
}

void
TopoCache::updateObject(TOid id, const Geom& newGeom)
{
    auto it = cachedObjects_.find(id);
    if (it != cachedObjects_.end()) {
        it->second.bbox = boundingBox(newGeom);
    }
}

void
TopoCache::deleteObject(TOid id)
{
    auto it = cachedObjects_.find(id);
    if (it != cachedObjects_.end()) {
        it->second.deleted = true;
    }
}

} // namespace wiki
} // namespace maps
