#include "topo_relations_processor.h"

#include <maps/wikimap/mapspro/services/editor/src/collection.h>
#include <maps/wikimap/mapspro/services/editor/src/objects_cache.h>
#include <maps/wikimap/mapspro/services/editor/src/relations_manager.h>
#include <maps/wikimap/mapspro/services/editor/src/relations_processor.h>
#include <maps/wikimap/mapspro/services/editor/src/topo_storage.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>

#include <yandex/maps/wiki/configs/editor/categories.h>
#include <yandex/maps/wiki/topo/cache.h>

#include <utility>

namespace maps {
namespace wiki {

TopoRelationsProcessor::TopoRelationsProcessor(ObjectsCache& cache)
    : cache_(cache)
{}

void
TopoRelationsProcessor::addLinearElementToJunction(
    Junction *jc, LinearElement* le, const std::string& roleId)
{
    cache_.relationsManager().createRelation(le->id(), jc->id(), roleId);
}

void
TopoRelationsProcessor::removeLinearElementFromJunction(
    Junction *jc, LinearElement* le, const std::string& roleId)
{
    cache_.relationsManager().deleteRelation(le->id(), jc->id(), roleId);
}

void
TopoRelationsProcessor::updateSeqNums(GeoObject* obj)
{
    if (!obj->category().slavesTopologicalyContinuous() ||
        !obj->isModifiedLinksToSlaves())
    {
        return;
    }

    std::map<std::string, std::list<RelationInfo>> roleSlavesWithGeom;
    RelationInfos::Range slaveInfos = obj->slaveRelations().range(
        obj->category().slaveRoleIds(roles::filters::IsSequentialGeom));
    for (const auto& slaveInfo : slaveInfos) {
        if (!slaveInfo.geom().isNull()) {
            roleSlavesWithGeom[slaveInfo.roleId()].push_back(slaveInfo);
        }
    }
    if (roleSlavesWithGeom.empty()) {
        return;
    }
    for (auto& kv : roleSlavesWithGeom) {
        auto& slaves = kv.second;
        if (slaves.size() == 1) {
            slaves.front().setSeqNum(0);
        } else {
            computeSeqNums(obj, slaves);
        }
    }
}

/// TODO Assumes only linear elements have sequential roles
/// Slaves with geom size > 1
namespace {

bool
isWithinSmallDistance(const Geom& geom1, const Geom& geom2)
{
#if GEOS_VERSION_MAJOR*100+GEOS_VERSION_MINOR*10+GEOS_VERSION_PATCH < 342
    return const_cast<geos::geom::Geometry*>(geom1.geosGeometryPtr())->
        isWithinDistance(geom2.geosGeometryPtr(),
            CALCULATION_TOLERANCE);
#else
  return geom1->isWithinDistance(geom2.geosGeometryPtr(),
    CALCULATION_TOLERANCE);
#endif
}

bool
isSequnceReversed(
    const std::list<RelationInfo>& orderedSlaves,
    const GeoObject* containingObject)
{
    if (orderedSlaves.size() < 2) {
        return false;
    }
    auto geomSlaveRoles = containingObject->category().geomSlaveRoleIds();
    if (geomSlaveRoles.size() < 2) {
        return false;
    }
    auto orderedSlavesRole = orderedSlaves.back().roleId();

    auto it = std::find(geomSlaveRoles.begin(), geomSlaveRoles.end(), orderedSlavesRole);
    ASSERT (it != geomSlaveRoles.end());
    bool jointRoleIsLeading = (it != geomSlaveRoles.begin());
    const auto& jointRole =
        jointRoleIsLeading
        ? *(it - 1)
        : *(it + 1);
    for (const auto& slaveInfo: containingObject->slaveRelations().range(jointRole)) {
        const Geom& geom = slaveInfo.geom();
        ASSERT(!geom.isNull());
        if (jointRoleIsLeading) {
            if (isWithinSmallDistance(geom, orderedSlaves.back().geom())) {
                return true;
            }
        } else {
            if (isWithinSmallDistance(geom, orderedSlaves.front().geom())) {
                return true;
            }
        }
    }
    return false;
}
}//namespace

void
TopoRelationsProcessor::computeSeqNums(
    GeoObject* obj, const std::list<RelationInfo>& slavesWithGeom)
{
    topo::EdgeIDSet edgeIds;
    for (const auto& slaveInfo : slavesWithGeom) {
        DEBUG() << "Edge id: " << slaveInfo.id();
        edgeIds.insert(slaveInfo.id());
    }

    auto topoGroup = cfg()->editor()->topologyGroups().findGroup(
        slavesWithGeom.front().categoryId());
    REQUIRE(topoGroup, "Topology group is not set, category id " <<
        slavesWithGeom.front().categoryId());
    TopoStorage storage(*topoGroup, cache_);

    topo::Cache topoCache(storage, CALCULATION_TOLERANCE);
    boost::optional<topo::Circuit> circuitSeq = topoCache.circuit(edgeIds, false);
    if (!circuitSeq) {
        if (obj->primaryEdit()) {
            THROW_WIKI_LOGIC_ERROR(ERR_TOPO_NOT_CONTINUOUS,
                "Objects are not topologically continuous");
        }
        return;
    }
    std::list<RelationInfo> orderedSlaves;
    for (const auto& id : (*circuitSeq).edges) {
        auto it = std::find_if(slavesWithGeom.begin(), slavesWithGeom.end(),
            [id] (const RelationInfo& link) -> bool
            {
                return link.relative()->id() == id;
            });
        if (it == slavesWithGeom.end()) {
            THROW_WIKI_INTERNAL_ERROR("Object " << id <<
                " not found in topologically sorted slaves");
        }
        orderedSlaves.push_back(*it);
    }
    if (isSequnceReversed(orderedSlaves, obj)) {
        orderedSlaves.reverse();
    }
    size_t i = 0;
    TOIds modifiedRelationIds;
    for (auto& slaveInfo : orderedSlaves) {
        slaveInfo.setSeqNum(i);
        DEBUG() << BOOST_CURRENT_FUNCTION
            << " master_id: " << obj->id()
            << " slave_id: " << slaveInfo.id()
            << " seq_num: " << i;
        ++i;
    }
}

} // namespace wiki
} // namespace maps
