#include "topological_callbacks.h"

#include "topo_relations_processor.h"
#include <maps/wikimap/mapspro/services/editor/src/collection.h>
#include <maps/wikimap/mapspro/services/editor/src/objects_cache.h>
#include <maps/wikimap/mapspro/services/editor/src/relations_manager.h>
#include <maps/wikimap/mapspro/services/editor/src/relations_processor.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/object_update_data.h>

#include <yandex/maps/wiki/configs/editor/topology_groups.h>

namespace maps {
namespace wiki {

void
TopoAddLinearElementCallback::process(const topo::AddEdgeEvent& event) const
{
    LinearElement* newLe = as<LinearElement>(cache_.getExisting(event.id()));
    if (event.sourceExists()) {
        LinearElement* sourceLe = as<LinearElement>(cache_.getExisting(event.sourceId()));
        sourceLe->cloneNoGeom(newLe);
    } else {
        auto newLEDataIt = linearElementsData_.find(event.sourceId());
        REQUIRE(newLEDataIt != linearElementsData_.end(),
            "New linear element data not found, sourceId " << event.sourceId());
        const NewLinearElementData& data = newLEDataIt->second;
        const ObjectUpdateData& objectData = *data.updateData;
        newLe->initAttributes(objectData.categoryId(),
            objectData.attributes(), objectData.tableAttributes());
        newLe->setRichContent(objectData.richContent());

        for (const auto& roleDiff : data.mastersDiff) {
            const auto& roleId = roleDiff.first;
            const TOIds& masterIds = roleDiff.second;
            for (auto masterId : masterIds) {
                REQUIRE(masterId, "All LE master objects should have already been processed");
                cache_.relationsManager().createRelation(masterId, newLe->id(), roleId);
            }
        }
        DEBUG() << "New linear element " << newLe->id() << ", is primary: " << data.isPrimary;
        newLe->primaryEdit(data.isPrimary);
    }
    auto topoGroup = cfg()->editor()->topologyGroups().findGroup(newLe->categoryId());
    REQUIRE(topoGroup, "Topology group not set for linear element, id: " << newLe->id());
    Junction* start = as<Junction>(cache_.getExisting(event.startNode()));
    Junction* end = as<Junction>(cache_.getExisting(event.endNode()));
    TopoRelationsProcessor topoProcessor(cache_);
    topoProcessor.addLinearElementToJunction(start, newLe, (*topoGroup).startJunctionRole());
    topoProcessor.addLinearElementToJunction(end, newLe, (*topoGroup).endJunctionRole());
}

void
TopoMoveLinearElementCallback::process(const topo::MoveEdgeEvent& event) const
{
    LinearElement* le = as<LinearElement>(cache_.getExisting(event.id()));
    auto topoGroup = cfg()->editor()->topologyGroups().findGroup(le->categoryId());
    REQUIRE(topoGroup, "Topology group not set for linear element, id: " << le->id());
    TopoRelationsProcessor topoProcessor(cache_);
    Junction* oldStart = as<Junction>(cache_.getExisting(event.oldStartNode()));
    Junction* oldEnd = as<Junction>(cache_.getExisting(event.oldEndNode()));
    if (event.newStartNode()) {
        topoProcessor.removeLinearElementFromJunction(oldStart, le, (*topoGroup).startJunctionRole());
    }
    if (event.newEndNode()) {
        topoProcessor.removeLinearElementFromJunction(oldEnd, le, (*topoGroup).endJunctionRole());
    }
    if (event.newStartNode()) {
        Junction* newStart = as<Junction>(cache_.getExisting(*event.newStartNode()));
        topoProcessor.addLinearElementToJunction(newStart, le, (*topoGroup).startJunctionRole());
    }
    if (event.newEndNode()) {
        Junction* newEnd = as<Junction>(cache_.getExisting(*event.newEndNode()));
        topoProcessor.addLinearElementToJunction(newEnd, le, (*topoGroup).endJunctionRole());
    }
}

void
TopoDeleteLinearElementCallback::process(const topo::DeleteEdgeEvent& event) const
{
    LinearElement* le = as<LinearElement>(cache_.getExisting(event.id()));
    auto topoGroup = cfg()->editor()->topologyGroups().findGroup(le->categoryId());
    REQUIRE(topoGroup, "Topology group not set for linear element, id: " << le->id());
    Junction* jcA = as<Junction>(cache_.getExisting(le->junctionAId()));
    Junction* jcB = as<Junction>(cache_.getExisting(le->junctionBId()));

    TopoRelationsProcessor processor(cache_);

    processor.removeLinearElementFromJunction(jcA, le, (*topoGroup).startJunctionRole());
    processor.removeLinearElementFromJunction(jcB, le, (*topoGroup).endJunctionRole());
}

} // namespace wiki
} // namespace maps
