#pragma once

#include "common.h"
#include "geom.h"

#include <maps/libs/xml/include/xml.h>
#include <maps/libs/tile/include/coord.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/serialization.h>

#include <maps/libs/json/include/value.h>
#include <geos/geom/Coordinate.h>
#include <geos/geom/Envelope.h>

#include <string>
#include <istream>

namespace maps{
namespace wiki{

GeometryPtr createGeom(const pqxx::field& dbfield);
GeometryPtr createGeom(double xMin, double yMin, double xMax, double yMax,
                       SpatialRefSystem spatialRefSystem);
GeometryPtr createGeom(const geos::geom::Envelope& envelope,
                       SpatialRefSystem spatialRefSystem);
GeometryPtr createGeom(const geolib3::BoundingBox& bbox,
                       SpatialRefSystem spatialRefSystem);

GeometryPtr createPoint(std::vector<geos::geom::Coordinate> *points,
                        SpatialRefSystem spatialRefSystem);

GeometryPtr createGeomFromJson(const json::Value& json);
GeometryPtr createGeomFromJsonStr(const std::string& jsonText);
TGeoPoint createGeoPointFromJsonStr(const std::string& jsonText);
TGeoPoint createGeoPointFromJson(const json::Value& json);

std::vector<TGeoPoint> createGeoPointCollectionFromJson(const json::Value& json);
std::vector<TGeoPoint> createGeoPointCollectionFromJsonStr(const std::string& jsonText);

std::vector<Geom> createGeomCollectionFromJson(const json::Value& json);
std::vector<Geom> createGeomCollectionFromJsonStr(const std::string& jsonText);

std::optional<TGeoPoint> findGeomError(const Geom& geom);

GeometryPtr createPoint(double x, double y, SpatialRefSystem spatialRefSystem);

GeometryPtr createPolygon(
    std::vector<geos::geom::Coordinate> *points,
    std::vector<GeosGeometryPtr> *holes,
    SpatialRefSystem spatialRefSystem,
    AntiMeridianAdjustPolicy adjustPolicy);

GeometryPtr createPolyline(
    std::vector<geos::geom::Coordinate> *points,
    SpatialRefSystem spatialRefSystem,
    AntiMeridianAdjustPolicy adjustPolicy);

enum class PolygonBufferPolicy
{
    Self,
    Rings
};

std::list<Geom> createPolygonBuffers(
    const Geom& geom,
    double width,
    PolygonBufferPolicy policy);

std::string json(const geos::geom::Envelope& envelope);
std::string json(const geolib3::BoundingBox& bbox);

geos::geom::Envelope createEnvelope(const std::string& boundingBox,
                                    SpatialRefSystem spatialRefSystem);
geolib3::BoundingBox createBbox(
        const std::string& boundingBox,
        SpatialRefSystem spatialRefSystem);

/*
 * Decodes base64 ignoring whitespace characters
 */
std::string decodeBase64(const std::string& base64data);

template <class MapType>
std::set<typename MapType::key_type>
readKeys(const MapType& map)
{
    std::set<typename MapType::key_type> ret;
    for (typename MapType::const_iterator it = map.begin(); it != map.end(); ++it) {
        ret.insert(it->first);
    }
    return ret;
}

class ReadingPipe : public boost::noncopyable
{
public:
    explicit ReadingPipe(const std::string& command);
    ~ReadingPipe();

    std::string read() const;
    int wait();

private:
    FILE* f_;
};

} //namespace wiki
} //namespace maps

namespace geos{
namespace geom{

inline std::istream& operator >> (std::istream& is, Coordinate& point)
{
    is>>point.x>>point.y;
    return is;
}

} //namespace geom
} //namespace geos
