#include <maps/wikimap/mapspro/services/gdpr/src/lib/utils.h>

#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>

#include <contrib/libs/fmt/include/fmt/format.h>

#include <chrono>
#include <iostream>

namespace maps::wiki::gdpr {
namespace {

using namespace std::chrono;
using namespace std::chrono_literals;

std::optional<std::string> performCheck(
    pqxx::transaction_base& txn, std::optional<days> age)
{
    if (!age) {
        return std::nullopt;
    }

    auto takeouts = getTakeoutDataWithThreshold(txn, *age);
    if (takeouts.empty()) {
        return std::nullopt;
    }

    const auto& oldest = takeouts.front();
    auto oldestAge = duration_cast<days>(
        system_clock::now() - oldest.requestTime);

    return fmt::format(
        "{} takeouts older than {} days.\n"
        "Oldest takeout [id={}] is {} days old",
        takeouts.size(), age->count(),
        oldest.takeoutId, oldestAge.count());
}

void run(common::ExtendedXmlDoc& config, std::optional<days> warnAge, std::optional<days> critAge)
{
    REQUIRE(!critAge || warnAge < critAge, "warn age must be less than crit age if both defined");
    maps::wiki::common::PoolHolder core(config, "core", "grinder");
    auto txn = core.pool().slaveTransaction();
    auto crit = performCheck(*txn, critAge);
    if (crit) {
        std::cout << "2; " << *crit;
        return;
    }

    auto warn = performCheck(*txn, warnAge);
    if (warn) {
        std::cout << "1; " << *warn;
        return;
    }

    std::cout << "0; No unprocessed takeouts\n";
}

} // anonymous namespace
} // namespace maps::wiki::gdpr

int main(int argc, char* argv[]) try
{
    using namespace maps::wiki::gdpr;

    // juggler events must have clean stderr
    maps::log8::setLevel(maps::log8::Level::FATAL);

    maps::cmdline::Parser argsParser{"Unprocessed GDPR-requests monitoring"};

    auto configPath = argsParser.file("config")
        .help("path to services config");

    auto warnAgeArg = argsParser.num("warn-age")
        .help("Age value (days) to start to warn at");

    auto critAgeArg = argsParser.num("crit-age")
        .help("Age value (days) to start to crit at");

    argsParser.parse(argc, argv);

    std::optional<std::chrono::days> warnAge;
    if (warnAgeArg.defined()) {
        warnAge.emplace(warnAgeArg);
    }
    std::optional<std::chrono::days> critAge;
    if (warnAgeArg.defined()) {
        critAge.emplace(critAgeArg);
    }

    auto configPtr = configPath.defined()
        ? std::make_unique<maps::wiki::common::ExtendedXmlDoc>(configPath)
        : maps::wiki::common::loadDefaultConfig();

    run(*configPtr, warnAge, critAge);
    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    std::cout << "2; Exception caught: " << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    std::cout << "2; Exception caught: " << e.what();
    return EXIT_FAILURE;
}
