#include <library/cpp/resource/resource.h>

#include <maps/infra/yacare/include/metrics.h>
#include <maps/infra/yacare/include/yacare.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>

#include <maps/wikimap/mapspro/services/maintenance/src/libs/sql_metrics.h>

#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>

namespace {

std::unique_ptr<maps::wiki::maintenance::SqlMetric> g_sqlRevisionMetaApprovedBboxesQueueSizeChecker;
std::unique_ptr<maps::wiki::maintenance::SqlMetric> g_sqlRevisionQueryTimeChecker;
std::unique_ptr<maps::wiki::maintenance::SqlMetric> g_sqlSocialQueryTimeChecker;
std::unique_ptr<maps::wiki::maintenance::SqlMetric> g_sqlSocialUserActivityAlertsChecker;

const std::string CONFIG_PATH = "/etc/yandex/maps/wiki/services/services.xml";
const std::chrono::seconds TTL(60);

const std::string CORE_DB_ID = "core";
const std::string GRINDER_POOL_ID = "grinder";
const std::string SOCIAL_DB_ID = "social";

} // namespace

YCR_METRICS_GENERATOR([]() {
    ASSERT(g_sqlRevisionMetaApprovedBboxesQueueSizeChecker);
    return maps::yasm_metrics::YasmMetrics().addMetric(
        maps::yasm_metrics::NumericalYasmMetric(
            "revision_meta_approved_bboxes_queue_size_axxx",
            g_sqlRevisionMetaApprovedBboxesQueueSizeChecker->getValue()));
})

YCR_METRICS_GENERATOR([]() {
    ASSERT(g_sqlRevisionQueryTimeChecker);
    return maps::yasm_metrics::YasmMetrics().addMetric(
        maps::yasm_metrics::NumericalYasmMetric(
            "revision_query_time_axxx", g_sqlRevisionQueryTimeChecker->getValue()));
})

YCR_METRICS_GENERATOR([]() {
    ASSERT(g_sqlSocialQueryTimeChecker);
    return maps::yasm_metrics::YasmMetrics().addMetric(
        maps::yasm_metrics::NumericalYasmMetric(
            "social_query_time_axxx", g_sqlSocialQueryTimeChecker->getValue()));
})

YCR_METRICS_GENERATOR([]() {
    ASSERT(g_sqlSocialUserActivityAlertsChecker);
    return maps::yasm_metrics::YasmMetrics().addMetric(
        maps::yasm_metrics::NumericalYasmMetric(
            "social_user_activity_alerts_axxx", g_sqlSocialUserActivityAlertsChecker->getValue()));
})

YCR_MAIN(argc, argv) {
    maps::cmdline::Parser parser;
    auto workerConfig = parser.string("config")
        .help("path to worker configuration")
        .defaultValue(CONFIG_PATH);
    auto ttl = parser.option<std::chrono::seconds>("ttl")
        .help("time duration between sql checking queries")
        .defaultValue(TTL);
    parser.parse(argc, argv);

    auto configDocPtr = workerConfig.defined()
        ? std::make_unique<maps::wiki::common::ExtendedXmlDoc>(workerConfig)
        : maps::wiki::common::loadDefaultConfig();

    maps::wiki::common::PoolHolder corePoolHolder(*configDocPtr, CORE_DB_ID, GRINDER_POOL_ID);
    maps::wiki::common::PoolHolder socialPoolHolder(*configDocPtr, SOCIAL_DB_ID, GRINDER_POOL_ID);

    g_sqlRevisionMetaApprovedBboxesQueueSizeChecker = std::make_unique<maps::wiki::maintenance::SqlMetric>(
        corePoolHolder.pool(),
        "approved bboxes queue size",
        ttl,
        NResource::Find("revision_meta_approved_bboxes_queue_size.sql"),
        std::make_shared<maps::wiki::maintenance::SqlMetricStrategyQueryValue>()
    );
    g_sqlRevisionQueryTimeChecker = std::make_unique<maps::wiki::maintenance::SqlMetric>(
        corePoolHolder.pool(), "revision worker", ttl, NResource::Find("revision_worker_query.sql"),
        std::make_shared<maps::wiki::maintenance::SqlMetricStrategyQueryTime>()
    );
    g_sqlSocialQueryTimeChecker = std::make_unique<maps::wiki::maintenance::SqlMetric>(
        socialPoolHolder.pool(), "social worker", ttl, NResource::Find("social_worker_query.sql"),
        std::make_shared<maps::wiki::maintenance::SqlMetricStrategyQueryTime>()
    );

    g_sqlSocialUserActivityAlertsChecker = std::make_unique<maps::wiki::maintenance::SqlMetric>(
        socialPoolHolder.pool(),
        "social user activity alerts",
        ttl,
        NResource::Find("social_user_activity_alerts_query.sql"),
        std::make_shared<maps::wiki::maintenance::SqlMetricStrategyQueryValue>()
    );

    yacare::run(yacare::RunSettings{.useSystemDefaultLocale = true});
    return 0;
}
