#include <maps/wikimap/mapspro/services/maintenance/src/libs/sql_metrics.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

namespace maps::wiki::maintenance {

namespace {

using std::chrono::duration_cast;
using std::chrono::milliseconds;

} // anonymous namespace

SqlMetric::SqlMetric(
        pgpool3::Pool& pool,
        std::string workerName,
        std::chrono::seconds ttl,
        std::string query,
        std::shared_ptr<SqlMetricStrategy> strategy)
    : pool_(pool)
    , workerName_(std::move(workerName))
    , query_(std::move(query))
    , strategy_(std::move(strategy))
    , worker_([this](){ update(); }, ttl)
{
    ASSERT(strategy_);
    worker_.start();
}

void SqlMetric::update()
{
    try {
        auto txn = pool_.slaveTransaction();

        auto result = strategy_->getValue(*txn, query_);
        INFO() << workerName_ << " has updated lastValue: " << result;

        lastValue_.store(result, std::memory_order_relaxed);
    } catch (const std::exception& ex) {
        ERROR() << "Error on update " << workerName_ << " : " << ex.what();
    }
}

MetricValue SqlMetric::getValue()
{
    return lastValue_.load(std::memory_order_relaxed);
}


MetricValue SqlMetricStrategyQueryTime::getValue(
    pqxx::transaction_base& txn, const std::string& query) const
{
    auto firstTimestamp = std::chrono::steady_clock::now();
    auto rows = txn.exec(query);
    auto secondTimestamp = std::chrono::steady_clock::now();

    auto timeDuration = secondTimestamp - firstTimestamp;
    return duration_cast<milliseconds>(timeDuration).count();
}

MetricValue SqlMetricStrategyQueryValue::getValue(
    pqxx::transaction_base& txn, const std::string& query) const
{
    return txn.exec1(query)[0].as<MetricValue>();
}

} // namespace maps::wiki::maintenance
