#include "result_handler.h"

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/mds_path.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/queued_photo_id_gateway.h>

#include <maps/libs/chrono/include/time_point.h>

#include <maps/libs/geolib/include/serialization.h>
#include <yandex/maps/geolib3/sproto.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/opencv.h>

#include <chrono>
#include <optional>
#include <cmath>

namespace maps::mrc::uploader {

using namespace signal_queue;

namespace {

template<typename T>
mds::Key makeMdsKey(const T& result)
{
    return mds::Key(result.mdsGroupId().get(), result.mdsPath().get());
}

template<typename T>
bool hasMdsKey(const T& result)
{
    return result.mdsGroupId() && result.mdsPath();
}

/**
 * Posts JPEG data to MDS and sets mdsGroupId and mdsPath of result object
 */
template<typename T>
void saveImageToMds(T& result, const std::string& path, mds::Mds& mds)
{
    auto resp = mds.post(path, result.data().image());
    result.mdsGroupId() = resp.key().groupId;
    result.mdsPath() = resp.key().path;
}

std::optional<ugc_event_logger::UserInfo>
tryMakeUgcEventUserInfo(const AssignmentImage& data)
{
    if (!data.sourceIp()) {
        return {};
    }
    ugc_event_logger::UserInfo result{.ip = *data.sourceIp()};
    if (data.sourcePort()) {
        result.port = *data.sourcePort();
    }

    if (data.userId()) {
        result.uid = *data.userId();
    }
    return result;
}

std::optional<ugc_event_logger::UserInfo>
tryMakeUgcEventUserInfo(const RideImage& data)
{
    if (!data.sourceIp()) {
        return {};
    }
    ugc_event_logger::UserInfo result{.ip = *data.sourceIp()};
    if (data.sourcePort()) {
        result.port = *data.sourcePort();
    }

    result.uid = data.userId();
    return result;
}

bool sameFeatureExists(
    const db::Feature& feature,
    pqxx::transaction_base& txn)
{
    return db::FeatureGateway{txn}.exists(
        db::table::Feature::sourceId == feature.sourceId() &&
        db::table::Feature::date == feature.timestamp());
}

} // anonymous namespace

db::Feature ResultHandler::makeFeature(
        const sresults::Image& resultImage,
        const std::string& sourceId,
        db::Dataset dataset)
{
    REQUIRE(!resultImage.image().empty(),
            "Empty image from sourceId " << sourceId);

    auto feature = sql_chemistry::GatewayAccess<db::Feature>::construct()
        .setTimestamp(
            chrono::TimePoint(
                std::chrono::milliseconds(resultImage.created())
            )
        )
        .setSourceId(sourceId)
        .setDataset(dataset)
        .setUploadedAt(chrono::TimePoint::clock::now());

    cv::Mat image = common::decodeImage(resultImage.image());
    feature.setSize(image.cols, image.rows);

    if (resultImage.estimatedPosition()) {
        const auto& location = resultImage.estimatedPosition().get();

        feature.setGeodeticPos(geolib3::sproto::decode(location.point()));

        if (location.heading()) {
            feature.setHeading(geolib3::Heading(location.heading().get()));
        }
    }

    return feature;
}

void ResultHandler::saveToDatabase(AssignmentImage& result)
{
    std::string sourceId = result.sourceId() ? *result.sourceId()
            : std::to_string(result.assignmentId());

    if (result.data().image().empty()) {
        WARN() << "Discard empty image from sourceId " << sourceId;
        return;
    }

    auto feature =
        makeFeature(result.data(), sourceId, db::Dataset::Agents)
            .setAssignmentId(static_cast<db::TId>(result.assignmentId()));

    if (sameFeatureExists(feature, *poolHolder_.pool().masterReadOnlyTransaction())) {
        WARN() << "Same feature exists: sourceId = " << feature.sourceId()
            << ", date = " << chrono::formatIsoDateTime(feature.timestamp());
        return;
    }

    auto txn = poolHolder_.pool().masterWriteableTransaction();
    db::FeatureGateway{*txn}.insert(feature);

    if (!hasMdsKey(result)) {
        std::string mdsPath = common::makeMdsPath(
            common::MdsObjectSource::Ugc,
            common::MdsObjectType::Image,
            feature.id());
        saveImageToMds(result, mdsPath, mds_);
    }
    feature.setMdsKey({result.mdsGroupId().get(), result.mdsPath().get()});

    db::FeatureGateway{*txn}.update(feature, db::UpdateFeatureTxn::No);

    txn->commit();

    if (auto ugcEventUserInfo = tryMakeUgcEventUserInfo(result)) {
        ugcEventLogger_.logEvent(
            ugcEventUserInfo.value(),
            ugc_event_logger::Action::Create,
            ugc_event_logger::Photo{.id = std::to_string(feature.id())}
        );
    }
}

void ResultHandler::saveToDatabase(RideImage& result)
{
    if (result.data().image().empty()) {
        WARN() << "Discard empty image from sourceId " << result.sourceId();
        return;
    }

    db::Dataset dataset = db::Dataset::Agents;
    common::MdsObjectSource mdsObjectSource;

    if (result.userId() != db::feature::YANDEX_DRIVE_UID) {
        dataset = db::Dataset::Rides;
        mdsObjectSource = common::MdsObjectSource::Ride;
    } else {
        dataset = db::Dataset::Drive;
        mdsObjectSource = common::MdsObjectSource::Drive;
    }

    auto feature = makeFeature(result.data(), result.sourceId(), dataset)
                       .setUserId(result.userId());
    if (result.clientRideId()) {
        feature.setClientRideId(result.clientRideId().get());
    }
    if (sameFeatureExists(feature, *poolHolder_.pool().masterReadOnlyTransaction())) {
        WARN() << "Same feature exists: sourceId = " << feature.sourceId()
            << ", date = " << chrono::formatIsoDateTime(feature.timestamp());
        return;
    }

    auto txn = poolHolder_.pool().masterWriteableTransaction();
    db::FeatureGateway{*txn}.insert(feature);

    if (!hasMdsKey(result)) {
        std::string mdsPath = common::makeMdsPath(
            mdsObjectSource,
            common::MdsObjectType::Image,
            feature.id());
        saveImageToMds(result, mdsPath, mds_);
    }

    feature.setMdsKey({result.mdsGroupId().get(), result.mdsPath().get()});
    db::FeatureGateway{*txn}.update(feature, db::UpdateFeatureTxn::No);

    // rides.queued_photo_id:
    // - used by ride_ispector
    // - designed for rides photos only
    if (db::Dataset::Rides == feature.dataset()) {
        db::rides::QueuedPhotoId photoId{feature.id(),
                                         chrono::TimePoint::clock::now()};
        db::QueuedPhotoIdGateway{*txn}.insert(photoId);
    }

    txn->commit();

    if (auto ugcEventUserInfo = tryMakeUgcEventUserInfo(result)) {
        ugcEventLogger_.logEvent(
            ugcEventUserInfo.value(),
            ugc_event_logger::Action::Create,
            ugc_event_logger::Photo{.id = std::to_string(feature.id())}
        );
    }
}

ResultHandler::ResultHandler(const mrc::common::Config& cfg, ugc_event_logger::Logger& ugcEventLogger)
    : poolHolder_(cfg.makePoolHolder())
    , mds_([&] {
        using namespace std::literals::chrono_literals;
        auto configuration = cfg.makeMdsConfiguration();
        return configuration.setTimeout(1min);
    }())
    , resultsQueue_(cfg.signalsUploader().queuePath())
    , ugcEventLogger_(ugcEventLogger)
{}

namespace {

template <typename T>
boost::optional<T>
nonThrowingPop(ResultsQueue& queue) noexcept(true) try {
    return queue.pop<T>();
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return boost::none;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return boost::none;
}

}

template <typename... Ts>
std::vector<std::variant<Ts...>> ResultHandler::pop()
{
        std::vector<std::variant<Ts...>> result;
        result.reserve(sizeof...(Ts));

        auto move = [](auto elem, std::vector<std::variant<Ts...>>& to) {
            if (elem) {
                to.push_back(std::move(*elem));
            }
        };

        (move(nonThrowingPop<Ts>(resultsQueue_), result), ...);
        return result;
}

template std::vector<std::variant<AssignmentImage, RideImage>>
ResultHandler::pop<AssignmentImage, RideImage>();
// The instantiations below are used in the tests
template std::vector<std::variant<AssignmentImage>> ResultHandler::pop<AssignmentImage>();
template std::vector<std::variant<RideImage>> ResultHandler::pop<RideImage>();

template<typename T>
void ResultHandler::process(T result) try {

    try {
        saveToDatabase(result);
    }
    catch (...) {
        // keeps result from being lost
        resultsQueue_.push(result);
        WARN() << "[" << result.data().created()
               << "] a result object was pushed back to its queue";
        throw;
    }
    INFO() << "[" << result.data().created()
           << "] a result object was uploaded";
}
catch (const maps::Exception& e) {
    ERROR() << e;
}
catch (const std::exception& e) {
    ERROR() << e.what();
}

template void ResultHandler::process<AssignmentImage>(AssignmentImage);
template void ResultHandler::process<RideImage>(RideImage);

} // namespace maps::mrc::uploader
