#include <library/cpp/testing/gtest/gtest.h>

#include <maps/wikimap/mapspro/services/mrc/agent-proxy/signals_uploader/result_handler.h>

#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/queued_photo_id_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/types.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/exif.h>
#include <maps/wikimap/mapspro/services/mrc/libs/ugc_event_logger/include/logger.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>
#include <yandex/maps/mrc/unittest/local_server.h>

#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/concurrent/include/scoped_guard.h>
#include <maps/libs/geolib/include/point.h>
#include <yandex/maps/geolib3/sproto.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/process/include/process.h>
#include <yandex/maps/wiki/common/date_time.h>


#include <boost/filesystem.hpp>
#include <boost/optional/optional_io.hpp>

#include <opencv2/imgcodecs/imgcodecs_c.h>

#include <chrono>
#include <cmath>
#include <csignal>
#include <cstdint>
#include <thread>
#include <memory>

#include <sys/types.h>
#include <sys/stat.h>

using namespace std::literals::chrono_literals;

namespace maps::mrc::uploader::tests {

template <typename T>
boost::optional<T> pop(ResultHandler& handler)
{
    auto variants = handler.pop<T>();
    if (variants.empty()) {
        return boost::none;
    }
    return std::get<T>(variants.at(0));
}

using namespace maps::mrc::signal_queue;

namespace {

const geolib3::Point2 TEST_LOCATION(43.9996, 56.3218);
const geolib3::Heading TEST_HEADING(15.0);
const std::string TEST_DEVICE_ID = "12345";
const std::string TEST_DEVICE_ID_2 = "67890";
const std::string TEST_USER_ID = "DEADBEAF";
const chrono::TimePoint TEST_CREATION_TIMESTAMP = (
    std::chrono::time_point_cast<std::chrono::seconds>(
        chrono::TimePoint::clock::now()
    )
);

const std::string TEST_IMAGE_PATH = GetWorkPath() + "/opencv_1.jpg";
const std::string TEST_IMAGE_DATA = maps::common::readFileToString(TEST_IMAGE_PATH); //FIXME one should start disticting text data from binary one
const std::string SIGNALS_UPLOADER_PATH = BinaryPath("maps/wikimap/mapspro/services/mrc/agent-proxy/signals_uploader/mrc-signals-uploader");

const auto CLIENT_RIDE_IDS = std::vector<std::optional<std::string>>{
    std::nullopt,
    "b0c2d8c8-6fc6-45d0-9e8e-45e37bd29636",
};

db::TId makeTestAssignment(pqxx::transaction_base& txn)
{
    db::ugc::TaskGateway gtw{txn};
    db::ugc::Task task;
    task.setStatus(db::ugc::TaskStatus::New);
    gtw.insert(task);
    auto assignment = task.assignTo(TEST_USER_ID);
    gtw.update(task);
    db::ugc::AssignmentGateway{txn}.insert(assignment);
    return assignment.id();
}

AssignmentImage makeAssignmentImage(
    db::TId assignmentId,
    chrono::TimePoint date = TEST_CREATION_TIMESTAMP)
{
    yandex::maps::sproto::offline::mrc::results::Image image;
    image.created() = std::chrono::duration_cast<std::chrono::milliseconds>(
        date.time_since_epoch()).count();
    image.image() = TEST_IMAGE_DATA;
    AssignmentImage assignmentImage;
    assignmentImage.data() = std::move(image);
    assignmentImage.sourceId() = TEST_DEVICE_ID_2;
    assignmentImage.assignmentId() = assignmentId;

    return assignmentImage;
}

RideImage makeRideImageImpl(
    const std::string& imageData,
    chrono::TimePoint date,
    const std::optional<std::string>& clientRideId)
{
    yandex::maps::sproto::offline::mrc::results::Image image;
    image.created() = std::chrono::duration_cast<std::chrono::milliseconds>(
        date.time_since_epoch()).count();
    image.image() = imageData;
    RideImage rideImage;
    rideImage.data() = std::move(image);
    rideImage.sourceId() = TEST_DEVICE_ID;
    rideImage.userId() = TEST_USER_ID;
    if (clientRideId) {
        rideImage.clientRideId() = *clientRideId;
    }
    return rideImage;
}

RideImage makeRideImage(
    chrono::TimePoint date = TEST_CREATION_TIMESTAMP,
    const std::optional<std::string>& clientRideId = std::nullopt)
{
    return makeRideImageImpl(TEST_IMAGE_DATA, date, clientRideId);
}

RideImage makeEmptyRideImage(chrono::TimePoint date = TEST_CREATION_TIMESTAMP)
{
    return makeRideImageImpl(std::string{}, date, std::nullopt);
}

void setLocation(AssignmentImage& assignmentImage)
{
    auto& image = assignmentImage.data();
    image.estimatedPosition() =
        yandex::maps::sproto::offline::mrc::results::Location();
    image.estimatedPosition()->point().lon() = TEST_LOCATION.x();
    image.estimatedPosition()->point().lat() = TEST_LOCATION.y();
    image.estimatedPosition()->heading() = TEST_HEADING.value();
}


} // anonymous namespace

class Fixture: public testing::Test {

public:
    void setAssignmentId(db::TId assignmentId)
    {
        assignmentId_ = assignmentId;
    }

    using TestFixture
        = unittest::WithUnittestConfig<unittest::MdsStubFixture,
                                       unittest::DatabaseFixture,
                                       unittest::YavisionStubFixture>;

    Fixture() {
        testFixture_.reset(new TestFixture);
        ::mkdir(testFixture_->config().signalsUploader().queuePath().c_str(), 0755);
    }

    ~Fixture() {
        boost::filesystem::remove_all(testFixture_->config().signalsUploader().queuePath());
    }

    std::unique_ptr<TestFixture> testFixture_;
    db::TId assignmentId_ = 0;
};

TEST_F(Fixture, testProcessingImage)
{
    db::TId assignmentId = 0;
    {
        auto txn = testFixture_->pool().masterWriteableTransaction();
        assignmentId = makeTestAssignment(*txn);
        auto assignmentImage = makeAssignmentImage(assignmentId);
        ResultsQueue queue(testFixture_->config().signalsUploader().queuePath());
        queue.push(assignmentImage);
        setAssignmentId(assignmentId);
        txn->commit();
    }

    ugc_event_logger::Logger ugcEventLogger("file.log", std::chrono::seconds(600));
    ResultHandler handler{testFixture_->config(), ugcEventLogger};
    auto result = pop<AssignmentImage>(handler);
    EXPECT_TRUE(result);
    handler.process(*result);
    EXPECT_TRUE(!pop<AssignmentImage>(handler));

    auto txn = testFixture_->pool().masterReadOnlyTransaction();
    auto photos = db::FeatureGateway{*txn}.load(
        db::table::Feature::assignmentId.equals(assignmentId));
    EXPECT_EQ(photos.size(), 1u);
    const auto& photo = photos.front();
    EXPECT_EQ(photo.timestamp(), TEST_CREATION_TIMESTAMP);
    EXPECT_FALSE(photo.hasPos());
    EXPECT_FALSE(photo.hasHeading());
    EXPECT_TRUE(photo.hasUploadedAt());
}

TEST_F(Fixture, testProcessingImageWithLocation)
{
    db::TId assignmentId = 0;
    {
        auto txn = testFixture_->pool().masterWriteableTransaction();
        assignmentId = makeTestAssignment(*txn);
        auto assignmentImage = makeAssignmentImage(assignmentId);
        setLocation(assignmentImage);
        ResultsQueue queue(testFixture_->config().signalsUploader().queuePath());
        queue.push(assignmentImage);
        setAssignmentId(assignmentId);
        txn->commit();
    }

    ugc_event_logger::Logger ugcEventLogger("file.log", std::chrono::seconds(600));
    ResultHandler handler{testFixture_->config(), ugcEventLogger};
    auto result = pop<AssignmentImage>(handler);
    EXPECT_TRUE(result);
    handler.process(*result);
    EXPECT_TRUE(!pop<AssignmentImage>(handler));


    auto txn = testFixture_->pool().masterReadOnlyTransaction();
    auto photo = db::FeatureGateway{*txn}.loadOne(
        db::table::Feature::assignmentId.equals(assignmentId));
    EXPECT_EQ(photo.timestamp(), TEST_CREATION_TIMESTAMP);
    EXPECT_TRUE(photo.hasPos());
    EXPECT_NEAR(photo.geodeticPos().x(), TEST_LOCATION.x(), 0.0001);
    EXPECT_NEAR(photo.geodeticPos().y(), TEST_LOCATION.y(), 0.0001);
    EXPECT_TRUE(photo.hasHeading());
    EXPECT_EQ(photo.heading(), TEST_HEADING);
}

TEST_F(Fixture, testExecutingBinary)
{
    db::TId assignmentId = 0;
    ResultsQueue queue(testFixture_->config().signalsUploader().queuePath());
    constexpr std::size_t IMAGES_COUNT = 64 * 2;

    {
        auto txn = testFixture_->pool().masterWriteableTransaction();
        assignmentId = makeTestAssignment(*txn);

        for (std::size_t imagesCount = IMAGES_COUNT / 2; imagesCount > 0;
             --imagesCount) {
            queue.push(makeAssignmentImage(assignmentId, TEST_CREATION_TIMESTAMP + 10ms * imagesCount));
            queue.push(makeRideImage(TEST_CREATION_TIMESTAMP + 10ms * imagesCount));
        }
        setAssignmentId(assignmentId);
        txn->commit();
    }
    ASSERT_EQ(queue.count<AssignmentImage>() + queue.count<RideImage>(),
              IMAGES_COUNT);

    process::Process uploader(process::run(
        process::Command({
            SIGNALS_UPLOADER_PATH,
            "--config", testFixture_->configPath().string()
        })
    ));

    auto txn = testFixture_->pool().masterReadOnlyTransaction();
    {
        concurrent::ScopedGuard atScopeExit{
            [&uploader] { uploader.sendSignal(SIGTERM); }};

        // Wait either for all images to be uploaded or for 2 minutes timeout.
        auto uploadedImagesCount = db::FeatureGateway{*txn}.load().size();
        for (std::size_t seconds = 0;
                uploadedImagesCount < IMAGES_COUNT && seconds < 120;
                ++seconds) {
            uploadedImagesCount = db::FeatureGateway{*txn}.load().size();
            std::this_thread::sleep_for(std::chrono::seconds(1));
        }
    }
    // Check local queues are empty.
    ASSERT_EQ(queue.count<AssignmentImage>() + queue.count<RideImage>(), 0u);
    // Check all images are uploaded.
    ASSERT_EQ(db::FeatureGateway{*txn}.load().size(), IMAGES_COUNT);
}

TEST_F(Fixture, testProcessingRideImage)
{
    ResultsQueue queue(testFixture_->config().signalsUploader().queuePath());

    auto createdAt = TEST_CREATION_TIMESTAMP;
    for (const auto& clientRideId : CLIENT_RIDE_IDS) {
        {
            auto txn = testFixture_->pool().masterWriteableTransaction();
            db::QueuedPhotoIdGateway{*txn}.remove(sql_chemistry::AnyFilter{});
            txn->commit();
        }
        createdAt += std::chrono::seconds(1);
        queue.push(makeRideImage(createdAt, clientRideId));

        ugc_event_logger::Logger ugcEventLogger("file.log",
                                                std::chrono::seconds(600));
        ResultHandler handler{testFixture_->config(), ugcEventLogger};
        auto result = pop<RideImage>(handler);
        EXPECT_TRUE(result);
        handler.process(*result);
        EXPECT_TRUE(!pop<RideImage>(handler));

        auto txn = testFixture_->pool().masterReadOnlyTransaction();
        auto queuedIds = db::QueuedPhotoIdGateway{*txn}.load();
        EXPECT_EQ(queuedIds.size(), 1u);
        auto photos =
            db::FeatureGateway{*txn}.loadByIds({queuedIds.front().photoId()});
        EXPECT_EQ(photos.size(), 1u);
        const auto& photo = photos.front();
        EXPECT_TRUE(photo.timestamp() == createdAt);
        EXPECT_TRUE(!photo.hasPos());
        EXPECT_TRUE(photo.sourceId() == TEST_DEVICE_ID);
        EXPECT_TRUE(photo.userId() == TEST_USER_ID);
        EXPECT_TRUE(photo.hasSize());
        EXPECT_TRUE(photo.hasUploadedAt());

        auto mds = testFixture_->config().makeMdsClient();
        EXPECT_EQ(TEST_IMAGE_DATA, mds.get(photo.mdsKey()));

        EXPECT_EQ(photo.clientRideId(), clientRideId);
    }
}


TEST_F(Fixture, testProcessingEmptyRideImage)
{
    ResultsQueue queue(
        testFixture_->config().signalsUploader().queuePath());
    queue.push(makeEmptyRideImage());

    ugc_event_logger::Logger ugcEventLogger("file.log", std::chrono::seconds(600));
    ResultHandler handler{testFixture_->config(), ugcEventLogger};
    auto result = pop<RideImage>(handler);
    EXPECT_TRUE(result);
    handler.process(*result);
    EXPECT_TRUE(!pop<RideImage>(handler));

    auto txn = testFixture_->pool().masterReadOnlyTransaction();
    auto queuedIds = db::QueuedPhotoIdGateway{*txn}.load();
    EXPECT_TRUE(queuedIds.empty());
}

TEST_F(Fixture, testLogUgcEvents)
{
    db::TId assignmentId = 0;
    const std::string sourceIp = "192.168.1.1";
    const uint16_t sourcePort = 9999;
    {
        ResultsQueue queue(testFixture_->config().signalsUploader().queuePath());
        auto txn = testFixture_->pool().masterWriteableTransaction();

        assignmentId = makeTestAssignment(*txn);

        auto assignmentImage = makeAssignmentImage(assignmentId);
        assignmentImage.sourceIp() = sourceIp;
        assignmentImage.sourcePort() = sourcePort;
        assignmentImage.userId() = TEST_USER_ID;
        queue.push(assignmentImage);

        /// Also add result without ip
        queue.push(makeAssignmentImage(assignmentId, TEST_CREATION_TIMESTAMP + 10ms));

        auto rideImage = makeRideImage();
        rideImage.sourceIp() = sourceIp;
        rideImage.sourcePort() = sourcePort;
        queue.push(rideImage);

        setAssignmentId(assignmentId);
        txn->commit();
    }

    auto getCurrentTimestamp = [&]() {
        return chrono::TimePoint(chrono::TimePoint::duration(0));
    };

    const std::string logfile = "event.log";
    auto loggerHandler = std::make_unique<ugc_event_logger::Logger>(logfile, std::chrono::seconds(600), getCurrentTimestamp);

    auto handler = std::make_unique<ResultHandler>(testFixture_->config(), *loggerHandler);

    while(auto result = pop<AssignmentImage>(*handler)) {
        handler->process(*result);
    }

    while(auto result = pop<RideImage>(*handler)) {
        handler->process(*result);
    }

    handler.reset();
    loggerHandler.reset();
    const std::string EXPECTED_LOG =
R"({"timestamp":"0","user":{"ip":"192.168.1.1","port":9999,"uid":"DEADBEAF"},"object":{"type":"photo","id":"1"},"action":"create"}
{"timestamp":"0","user":{"ip":"192.168.1.1","port":9999,"uid":"DEADBEAF"},"object":{"type":"photo","id":"3"},"action":"create"}
)";

    std::string logContents = maps::common::readFileToString(logfile);
    EXPECT_EQ(logContents, EXPECTED_LOG);
}

TEST_F(Fixture, testDuplicateFeatures)
{
    db::TId assignmentId = 0;
    ResultsQueue queue(testFixture_->config().signalsUploader().queuePath());

    auto loadedFeatures = db::FeatureGateway{*testFixture_->pool().masterReadOnlyTransaction()}.load();
    EXPECT_EQ(loadedFeatures.size(), 0u);

    maps::chrono::TimePoint DATE = maps::chrono::sinceEpochToTimePoint<std::chrono::seconds>(123456789);

    auto txn = testFixture_->pool().masterWriteableTransaction();
    assignmentId = makeTestAssignment(*txn);
    setAssignmentId(assignmentId);
    txn->commit();

    constexpr std::size_t UNIQUE_IMAGES_COUNT = 2;

    auto runUploader = [this]{
        return process::Process(process::run(
            process::Command({
                SIGNALS_UPLOADER_PATH,
                "--config", testFixture_->configPath().string()
            })
        ));
    };

    // Add two unique features
    {
        queue.push(makeAssignmentImage(assignmentId, DATE));
        queue.push(makeRideImage(DATE + 10ms));

        auto uploader = runUploader();

        auto txn = testFixture_->pool().masterReadOnlyTransaction();
        concurrent::ScopedGuard atScopeExit{
            [&uploader] { uploader.sendSignal(SIGTERM); }};

        auto uploadedImagesCount = db::FeatureGateway{*txn}.load().size();
        // Wait either for all images to be uploaded or for 30 seconds timeout.
        for (std::size_t seconds = 0;
            uploadedImagesCount < UNIQUE_IMAGES_COUNT && seconds < 180;
            ++seconds) {
            uploadedImagesCount = db::FeatureGateway{*txn}.load().size();
            std::this_thread::sleep_for(std::chrono::seconds(1));
        }
    }

    // Add duplicate features
    {
        queue.push(makeAssignmentImage(assignmentId, DATE));
        queue.push(makeRideImage(DATE + 10ms));

        auto uploader = runUploader();

        concurrent::ScopedGuard atScopeExit{
            [&uploader] { uploader.sendSignal(SIGTERM); }};

        // Wait till the queue is processed
        std::size_t queueSize = queue.count<AssignmentImage>() + queue.count<RideImage>();

        for (std::size_t seconds = 0; queueSize > 0 && seconds < 180; ++seconds) {
            std::this_thread::sleep_for(std::chrono::seconds(1));
            queueSize = queue.count<AssignmentImage>() + queue.count<RideImage>();
        }
    }

    // Check local queues are empty.
    ASSERT_EQ(queue.count<AssignmentImage>() + queue.count<RideImage>(), 0u);

    // Check only unique images are uploaded.
    txn = testFixture_->pool().masterReadOnlyTransaction();
    auto gtw = db::FeatureGateway{*txn};
    EXPECT_EQ(gtw.count(db::table::Feature::dataset == db::Dataset::Agents), 1u);
    EXPECT_EQ(gtw.count(db::table::Feature::dataset == db::Dataset::Rides), 1u);
}

TEST_F(Fixture, testRidesQueuedPhotoIdIsNotForDriveImage)
{
    auto img = makeRideImage();
    img.userId() = db::feature::YANDEX_DRIVE_UID;

    auto queue =
        ResultsQueue{testFixture_->config().signalsUploader().queuePath()};
    queue.push(img);

    auto ugcEventLogger =
        ugc_event_logger::Logger{"file.log", std::chrono::seconds(600)};
    auto handler = ResultHandler{testFixture_->config(), ugcEventLogger};
    auto result = pop<RideImage>(handler);
    EXPECT_TRUE(result);
    handler.process(*result);
    EXPECT_TRUE(!pop<RideImage>(handler));

    auto txn = testFixture_->pool().masterReadOnlyTransaction();
    auto queuedIds = db::QueuedPhotoIdGateway{*txn}.load();
    EXPECT_TRUE(queuedIds.empty());
}

} // namespace maps::mrc::uploader::tests
