#include <maps/wikimap/mapspro/services/mrc/agent-proxy/yacare_servant/lib/globals.h>
#include <maps/wikimap/mapspro/services/mrc/agent-proxy/yacare_servant/lib/serialization.h>
#include <maps/wikimap/mapspro/services/mrc/agent-proxy/yacare_servant/lib/utils.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/disqualified_source_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ride_recording_report_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/track_point_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>

#include <maps/libs/auth/include/user_info.h>
#include <maps/libs/chrono/include/days.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/tile/include/geometry.h>

#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <yandex/maps/i18n.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/mds_path.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/for_each_batch.h>
#include <yandex/maps/mrc/signal_queue/result_queue.h>
#include <yandex/maps/mrc/signal_queue/results.h>
#include <yandex/maps/proto/mrc/common/error.sproto.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <maps/infra/yacare/include/yacare.h>
#include <maps/infra/yacare/include/helpers.h>
#include <maps/infra/yacare/include/params/tvm.h>
#include <maps/infra/yacare/include/params/tile.h>

#include <boost/range/algorithm_ext/erase.hpp>
#include <boost/lexical_cast.hpp>

#include <algorithm>
#include <iterator>
#include <map>
#include <string>

YCR_QUERY_PARAM(id, maps::mrc::db::TId);

namespace {

constexpr maps::tile::Zoom MIN_TILE_TARGETS_ZOOM = 12;

} //anonymous namespace

namespace maps::mrc::agent_proxy {

namespace {

template <class ObjectsVector, class Gateway>
void saveBatches(ObjectsVector&& objects, Gateway&& gtw)
{
    //FIXME: remove when sql_chemistry will be ready for streamable inserts
    using IteratorType = typename ObjectsVector::iterator;
    common::forEachBatch(objects, 1000,
                         [&](IteratorType from, IteratorType to) {
                             ObjectsVector batch;
                             batch.reserve(std::distance(from, to));
                             for (auto it = from; it != to; ++it) {
                                 batch.push_back(std::move(*it));
                             }
                             gtw.insert(batch);
                         });
}

void saveRecordingReportsFromAssignmentResults(
    pqxx::transaction_base& txn,
    db::TId assignmentId,
    const std::string& sourceId,
    const presults::Results& data
)
{
    auto& mds = Globals::mdsClient();

    for (const auto& report : data.reports()) {
        if (report.empty()) {
            continue;
        }
        db::ugc::AssignmentRecordingReport recordingReport{assignmentId, sourceId};
        db::ugc::AssignmentRecordingReportGateway{txn}.insert(recordingReport);
        std::string mdsPath = makeMdsPath(
            common::MdsObjectSource::Ugc,
            common::MdsObjectType::Report,
            assignmentId,
            recordingReport.id());
        auto resp = mds.post(mdsPath, report);
        recordingReport.setMdsKey(resp.key());
        db::ugc::AssignmentRecordingReportGateway{txn}.update(recordingReport);
    }
}

maps::http::Status saveRecordingReportsFromRideResults(
    pgpool3::TransactionHandle& txn,
    const std::string& sourceId,
    const presults::Results& data,
    common::MdsObjectSource mdsObjectSource)
{
    maps::http::Status status = maps::http::Status::OK;

    if (data.reports().empty()) {
        return status;
    }

    auto& mds = Globals::mdsClient();

    for (const auto& rawReport : data.reports()) {
        if (rawReport.empty()) {
            continue;
        }

        std::pair<chrono::TimePoint, chrono::TimePoint> timestamps;
        try {
            timestamps = readFirstAndLastRecordTimestamps(rawReport);
        } catch (maps::Exception& ex) {
            // MAPSMRC-2957: ignore corrupted recording protobufs but set 422
            //               error code
            status = maps::http::Status::UnprocessableEntity;

            WARN() << "Ignore corrupted ride recording received from "
                   << sourceId;
            ERROR() << ex;
            continue;
        }

        auto dbReport = db::rides::RideRecordingReport(sourceId,
                timestamps.first, timestamps.second);

        db::rides::RideRecordingReportGateway gtw{*txn};
        gtw.insert(dbReport);

        std::string mdsPath = common::makeMdsPath(
            mdsObjectSource,
            common::MdsObjectType::Report,
            sourceId,
            dbReport.id());
        auto mdsKey = mds.post(mdsPath, rawReport).key();

        dbReport.setMdsGroupId(mdsKey.groupId).setMdsPath(mdsKey.path);
        gtw.update(dbReport);
    }

    return status;
}

db::CameraDeviation parseCameraDirection(const yacare::Request& request)
{
    const auto& valueStr = request.input()["camera-direction"];
    if (valueStr.empty()) {
        return db::CameraDeviation::Front;
    }
    try {
        return FromString<db::CameraDeviation>(to_title(TString(valueStr)));
    }
    catch (const std::exception&) {
        throw yacare::errors::BadRequest() << "invalid camera direction: "
                << valueStr;
    }
}

maps::http::Status uploadRides(
    const yacare::Request& request,
    const std::string& userId,
    common::MdsObjectSource mdsObjectSource)
{
    maps::http::Status status = maps::http::Status::OK;

    auto txn = Globals::pool().masterWriteableTransaction();

    auto results = boost::lexical_cast<presults::Results>(request.body());
    auto deviceId = yacare::request().input()["deviceid"];
    auto clientRideId = std::optional<std::string>{};
    if (results.client_ride_id()) {
        clientRideId = results.client_ride_id().get();
    }

    saveBatches(
        extractTrackPoints(deviceId, results),
        db::TrackPointGateway(*txn)
    );

    status = saveRecordingReportsFromRideResults(
        txn, deviceId, results, mdsObjectSource);

    txn->commit();

    for (auto& image : results.images()) {
        signal_queue::RideImage rideImage;
        rideImage.data() = std::move(image);
        rideImage.sourceId() = deviceId;
        rideImage.userId() = userId;
        rideImage.sourceIp() = std::string(request.getClientIpAddress());
        auto maybePort = request.getClientPort();
        if (maybePort.has_value()) {
            rideImage.sourcePort() = maybePort.value();
        }
        if (clientRideId.has_value()) {
            rideImage.clientRideId() = clientRideId.value();
        }
        Globals::resultsQueue().push(rideImage);
    }

    return status;
}

auto makeEdgeFilter(db::CameraDeviation cameraDirection)
{
    static const float COVERAGE_FRACTION_THRESHOLD{0.95};
    static const chrono::Days COVERAGE_AGE_THRESHOLD{30};
    const auto now = chrono::TimePoint::clock::now();
    return [=](const fb::TEdge& edge) {
        for (auto& coverage : edge.coverages) {
            if (coverage.cameraDeviation == cameraDirection &&
                coverage.coverageFraction > COVERAGE_FRACTION_THRESHOLD &&
                now - coverage.actualizationDate < COVERAGE_AGE_THRESHOLD) {
                return true;
            }
        }
        return false;
    };
}

void throwTaskAlreadyAcquired(mrc::db::TId taskId)
{
    yandex::maps::sproto::mrc::common::Error error;
    error.code() =
        yandex::maps::sproto::mrc::common::Error::Code::TASK_ALREADY_ACQUIRED;
    error.description() =
        "The task " + std::to_string(taskId) + " is already acquired";
    throw YacareProtobufError(yacare::Status::Conflict, error);
}

void checkTaskIsAcquirable(const db::ugc::Task& task) {
    if (task.status() == db::ugc::TaskStatus::New) {
        return;
    }

    if (task.status() == db::ugc::TaskStatus::Draft) {
        throw yacare::errors::BadRequest()
            << "The task draft with " << task.id() << " cannot be acquired";
    } else {
        throwTaskAlreadyAcquired(task.id());
    }
}

std::unordered_map<db::TId, std::unordered_set<std::string>> getGroupIdToAllowedLogins(
    const std::vector<db::ugc::Task>& tasks,
    db::ugc::TasksGroupGateway& tasksGroupGtw)
{
    std::unordered_map<db::TId, std::unordered_set<std::string>> groupIdToAllowedLogins;

    std::unordered_set<db::TId> groupTasksIds;
    groupTasksIds.reserve(tasks.size());
    for (const auto& task : tasks) {
        if (task.tasksGroupId()) {
            groupTasksIds.insert(*task.tasksGroupId());
        }
    }

    auto currGroups = tasksGroupGtw.loadByIds(
        {std::make_move_iterator(groupTasksIds.begin()),
         std::make_move_iterator(groupTasksIds.end())}
    );
    for (auto& currGroup : currGroups) {
        const auto& allowedLogins = currGroup.allowedAssigneesLogins();
        if (allowedLogins) {
            groupIdToAllowedLogins[currGroup.id()].insert(allowedLogins->begin(), allowedLogins->end());
        }
    }
    return groupIdToAllowedLogins;
}

bool taskIsAvailableToUser(
    const db::ugc::Task& task,
    const std::optional<auth::UserInfo>& userInfo,
    const std::unordered_map<db::TId, std::unordered_set<std::string>>& groupIdToAllowedLogins)
{
    if (!task.tasksGroupId()) {
        return true;
    }

    auto it = groupIdToAllowedLogins.find(*task.tasksGroupId());
    // If it == groupIdToAllowedLogins.end(), then for this groupId allowedLogins is nullopt
    return it == groupIdToAllowedLogins.end() || (userInfo && it->second.count(userInfo->login()));
}

bool isCapturingDisabled(sql_chemistry::Transaction& txn,
                         const std::string& sourceId)
{
    auto disqs = db::DisqualifiedSourceGateway{txn}.load(
        db::table::DisqualifiedSource::sourceId == sourceId &&
            db::table::DisqualifiedSource::disqType ==
                db::DisqType::DisableCapturing,
        orderBy(db::table::DisqualifiedSource::endedAt).desc().limit(1));
    auto now = chrono::TimePoint::clock::now();
    return !disqs.empty() && now <= disqs.front().endedAt().value_or(now);
}

} //anonymous namespace


YCR_RESPOND_TO("GET /ugc/tasks/search", box, lang, maybeUserInfo, results = 0)
{
    Locale locale = boost::lexical_cast<Locale>(lang);
    std::locale stdLocale = i18n::bestLocale(locale);

    auto txn = Globals::pool().slaveTransaction();
    db::ugc::TaskGateway gtw{*txn};
    db::ugc::TasksGroupGateway tasksGroupGtw{*txn};

    auto tasks = gtw.load(
        db::ugc::table::Task::hull.intersects(box) &&
        db::ugc::table::Task::status == db::ugc::TaskStatus::New);
    gtw.loadNames(tasks);

    if (tasks.empty()) {
        response.setStatus(yacare::HTTPStatus::NoContent);
        return;
    }

    if (request.input().has("results") && (tasks.size() > results)) {
        //FIXME: this is stupid
        tasks.resize(results);
    }

    int tasksProcessed = 0;
    auto groupIdToAllowedLogins = getGroupIdToAllowedLogins(tasks, tasksGroupGtw);
    presponse::Response sprotoResponse;
    for (const auto& task: tasks) {
        if (taskIsAvailableToUser(task, maybeUserInfo, groupIdToAllowedLogins)) {
            ++tasksProcessed;
            sprotoResponse.reply()->geo_object().push_back(
                serializeTaskToGeoObject(task, locale, stdLocale)
            );
        }
    }

    if (!tasksProcessed) {
        response.setStatus(yacare::HTTPStatus::NoContent);
    } else {
        response[CONTENT_TYPE] = CONTENT_TYPE_PROTOBUF;
        response << sprotoResponse;
    }
}

YCR_RESPOND_TO("POST /ugc/tasks/acquire", id, lang, userInfo)
{
    Locale locale = boost::lexical_cast<Locale>(lang);
    std::locale stdLocale = i18n::bestLocale(locale);

    auto txn = Globals::pool().masterWriteableTransaction();
    db::ugc::TaskGateway taskGtw{*txn};
    db::ugc::TasksGroupGateway tasksGroupGtw{*txn};

    auto task = taskGtw.loadById(id);

    auto groupIdToAllowedLogins = getGroupIdToAllowedLogins({task}, tasksGroupGtw);
    if (!taskIsAvailableToUser(task, userInfo, groupIdToAllowedLogins)) {
        throw yacare::errors::Forbidden();
    }

    checkTaskIsAcquirable(task);

    taskGtw.loadNames(task);

    auto assignment = task.assignTo(userInfo.uid());
    try {
        taskGtw.update(task);
    } catch (const maps::sql_chemistry::EditConflict&) {
        throwTaskAlreadyAcquired(task.id());
    }

    db::ugc::AssignmentGateway{*txn}.insert(assignment);
    presponse::Response sprotoResponse;
    sprotoResponse.reply()->geo_object().push_back(
        serializeAssignmentToGeoObject(assignment, task, locale, stdLocale)
    );
    response[CONTENT_TYPE] = CONTENT_TYPE_PROTOBUF;
    response << sprotoResponse;
    response.setStatus(yacare::HTTPStatus::Created);

    txn->commit();
}

YCR_RESPOND_TO("GET /ugc/assignments/list", lang, userId)
{
    constexpr chrono::Days SELECTION_TIMESPAN{90};

    Locale locale = boost::lexical_cast<Locale>(lang);
    std::locale stdLocale = i18n::bestLocale(locale);

    auto txn = Globals::pool().slaveTransaction();

    auto now = chrono::TimePoint::clock::now();
    const chrono::TimePoint startTime = now - SELECTION_TIMESPAN;
    auto assignments = db::ugc::AssignmentGateway{*txn}.load(
        db::ugc::table::Assignment::assignedTo == std::to_string(userId) &&
        (db::ugc::table::Assignment::status == db::ugc::AssignmentStatus::Active
            || db::ugc::table::Assignment::acquiredAt >= startTime));

    http::ETag serverEtag = makeEtag(introspection::computeHash(assignments));
    if (!request.preconditionSatisfied(serverEtag)) {
        response.setStatus(yacare::HTTPStatus::NotModified);
        return;
    }

    db::TIds taskIds;
    taskIds.reserve(assignments.size());
    for (const auto& assignment: assignments) {
        taskIds.emplace_back(assignment.taskId());
    }
    db::ugc::TaskGateway gtw{*txn};
    auto tasks = gtw.loadByIds(std::move(taskIds));
    gtw.loadNames(tasks);
    std::map<db::TId, db::ugc::Task> tasksMap;
    for (auto& task: tasks) {
        tasksMap.emplace(task.id(), std::move(task));
    }

    presponse::Response sprotoResponse;
    for (const auto& assignment: assignments) {
        const auto& task = tasksMap.at(assignment.taskId());
        sprotoResponse.reply()->geo_object().push_back(
            serializeAssignmentToGeoObject(assignment, task, locale, stdLocale)
        );
    }
    response[CONTENT_TYPE] = CONTENT_TYPE_PROTOBUF;
    response[ETAG] = boost::lexical_cast<std::string>(serverEtag);
    response << sprotoResponse;
}

YCR_RESPOND_TO("GET /ugc/assignments/targets", id, userId)
{
    auto txn = Globals::pool().slaveTransaction();

    auto assignment = db::ugc::AssignmentGateway{*txn}.loadById(id);
    if (assignment.assignedTo() != std::to_string(userId)) {
        throw yacare::errors::Forbidden();
    }

    db::ugc::TaskGateway gtw{*txn};
    auto task = gtw.loadById(assignment.taskId());
    gtw.loadTargets(task);

    presponse::Response sprotoResponse;
    for (const auto& target: task.targets()) {
        sprotoResponse.reply()->geo_object().push_back(
            serializeTargetToGeoObject(target)
        );
    }
    response[CONTENT_TYPE] = CONTENT_TYPE_PROTOBUF;
    response << sprotoResponse;
}

YCR_RESPOND_TO("POST /ugc/assignments/upload", YCR_USING(yacare::NginxLimitBody<yacare::ConfigScope::Endpoint>{}.maxSize(50_MB)), id, userId)
{
    auto txn = Globals::pool().masterWriteableTransaction();

    auto assignment = db::ugc::AssignmentGateway{*txn}.loadById(id);

    if (assignment.assignedTo() != std::to_string(userId)) {
        throw yacare::errors::Forbidden();
    }

    auto results = boost::lexical_cast<presults::Results>(request.body());
    db::TId assignmentId = id;
    auto deviceId = yacare::request().input()["deviceid"];

    saveBatches(
        extractTrackPoints(assignmentId, deviceId, results),
        db::TrackPointGateway(*txn)
    );
    saveRecordingReportsFromAssignmentResults(*txn, assignmentId, deviceId, results);
    saveBatches(
        extractAssignmentObjects(assignmentId, results),
        db::ugc::AssignmentObjectGateway(*txn)
    );
    txn->commit();

    for (auto& image: results.images()) {
        signal_queue::AssignmentImage assignmentImage;
        assignmentImage.data() = std::move(image);
        assignmentImage.assignmentId() = assignmentId;
        assignmentImage.sourceId() = deviceId;
        assignmentImage.userId() = std::to_string(userId);
        assignmentImage.sourceIp() = std::string(request.getClientIpAddress());
        auto maybePort = request.getClientPort();
        if (maybePort.has_value()) {
            assignmentImage.sourcePort() = maybePort.value();
        }
        Globals::resultsQueue().push(assignmentImage);
    }
}

YCR_RESPOND_TO("PUT /ugc/assignments/complete", id, userId)
{
    auto txn = Globals::pool().masterWriteableTransaction();

    auto assignment = db::ugc::AssignmentGateway{*txn}.loadById(id);
    if (assignment.assignedTo() != std::to_string(userId)) {
        throw yacare::errors::Forbidden();
    }
    if (assignment.status() != db::ugc::AssignmentStatus::Active) {
        /*
         * WARN: as requested in
         * https://wiki.yandex-team.ru/maps/dev/core/mobile/services/mrc/
         */
        throw yacare::errors::Forbidden();
    }
    assignment.markAsCompleted();
    db::ugc::AssignmentGateway{*txn}.update(assignment);
    txn->commit();
}

YCR_RESPOND_TO("PUT /ugc/assignments/abandon", id, userId)
{
    auto txn = Globals::pool().masterWriteableTransaction();

    auto assignment = db::ugc::AssignmentGateway{*txn}.loadById(id);
    if (assignment.assignedTo() != std::to_string(userId)) {
        throw yacare::errors::Forbidden();
    }

    if (assignment.status() != db::ugc::AssignmentStatus::Active) {
        /*
         * WARN: as requested in
         * https://wiki.yandex-team.ru/maps/dev/core/mobile/services/mrc/
         */
        throw yacare::errors::Forbidden();
    }
    assignment.markAsAbandoned();
    db::ugc::AssignmentGateway{*txn}.update(assignment);

    auto task = db::ugc::TaskGateway{*txn}.loadById(assignment.taskId());
    task.setStatus(db::ugc::TaskStatus::New);
    db::ugc::TaskGateway{*txn}.update(task);
    txn->commit();
}

YCR_RESPOND_TO("POST /ugc/rides/upload",
               YCR_USING(yacare::NginxLimitBody<yacare::ConfigScope::Endpoint>{}.maxSize(50_MB)),
               userId)
{
    checkUserCanPublishUgcContent(request);

    const auto status = uploadRides(
        request,
        std::to_string(userId),
        common::MdsObjectSource::Ride);

    response.setStatus(status);
}

YCR_RESPOND_TO("POST /ugc/drive/upload",
               YCR_USING(yacare::NginxLimitBody<yacare::ConfigScope::Endpoint>{}.maxSize(50_MB)))
{
    const auto status = uploadRides(
        request,
        db::feature::YANDEX_DRIVE_UID,
        common::MdsObjectSource::Drive);

    response.setStatus(status);
}

/**
 *  Always return tiles with the latest version, as it is the only one we have
 */
YCR_RESPOND_TO("GET /ugc/targets/tiles", tile, v={})
{
    if (tile.z() <= MIN_TILE_TARGETS_ZOOM) {
        throw yacare::errors::BadRequest() << "Invalid tile zoom";
    }

    auto cameraDirection = parseCameraDirection(request);
    auto edgeFilter = makeEdgeFilter(cameraDirection);
    auto bbox = geolib3::convertMercatorToGeodetic(tile::mercatorBBox(tile));
    auto dayPart = Globals::suncalc().getDayPart(
        bbox.center(), chrono::TimePoint::clock::now());

    std::vector<road_graph::EdgeId> edgeIds;
    auto deviceId = request.input()["deviceid"];
    if (common::DayPart::Day == dayPart &&
        !isCapturingDisabled(*Globals::pool().slaveTransaction(), deviceId)) {
        for (auto edgeId : Globals::roadGraphRTree().baseEdgesInWindow(bbox)) {
            auto edge = Globals::graphCoverage().edgeById(edgeId.value());
            if (!edge || !edgeFilter(*edge)) {
                edgeIds.push_back(edgeId);
            }
        }
    }

    presponse::Response sprotoResponse;
    if (edgeIds.empty()) {
        sprotoResponse.reply()->geo_object() = {};
    }

    for (const auto& edgeId : edgeIds) {
        auto edge = geolib3::Polyline2(
            Globals::roadGraph().edgeData(edgeId).geometry());
        for (const auto& polyline : geolib3::intersection(edge, bbox)) {
            sprotoResponse.reply()->geo_object().push_back(
                serializePolylineToGeoObject(polyline));
        }
    }

    std::stringstream ss;
    ss << sprotoResponse;
    auto responseData = ss.str();
    http::ETag serverEtag = makeEtag(responseData);

    if (!request.preconditionSatisfied(serverEtag)) {
        response.setStatus(yacare::HTTPStatus::NotModified);
        return;
    }

    response[CONTENT_TYPE] = CONTENT_TYPE_PROTOBUF;
    response[ETAG] = boost::lexical_cast<std::string>(serverEtag);
    response << responseData;
}

YCR_RESPOND_TO("GET /ugc/targets/version")
{
    response << serializeCoverageVersion(
        static_cast<std::string>(Globals::graphCoverage().version()));
}

} // namespace maps::mrc::agent_proxy
