#include "serialization.h"
#include "utils.h"

#include <maps/libs/tile/include/tile.h>
#include <yandex/maps/geolib3/sproto.h>
#include <yandex/maps/i18n.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ride_recording_report_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/track_point_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>

namespace maps::mrc::agent_proxy {

//primitives serialization to sproto objects
pcommon::AssignmentStatus serializeAssignmentStatus(
    db::ugc::AssignmentStatus status)
{
    switch (status) {
        case db::ugc::AssignmentStatus::Active:
            return pcommon::AssignmentStatus::ACTIVE;
        case db::ugc::AssignmentStatus::Completed:
            return pcommon::AssignmentStatus::COMPLETED;
        case db::ugc::AssignmentStatus::Abandoned:
            return pcommon::AssignmentStatus::ABANDONED;
        case db::ugc::AssignmentStatus::Accepted:
            return pcommon::AssignmentStatus::COMPLETED;
        case db::ugc::AssignmentStatus::Revoked:
            return pcommon::AssignmentStatus::ABANDONED;
    }
    throw LogicError() << "Unexpected assignment status " << status;
}

pugc::Direction serializeDirection(db::ugc::Direction direction)
{
    switch (direction) {
        case db::ugc::Direction::Forward:
            return pugc::Direction::FORWARD;
        case db::ugc::Direction::Backward:
            return pugc::Direction::BACKWARD;
        case db::ugc::Direction::Bidirectional:
            return pugc::Direction::BIDIRECTIONAL;
    }
    throw LogicError() << "Unexpected direction " << direction;
}

pugc::Traffic serializeTraffic(db::ugc::Traffic traffic)
{
    switch (traffic) {
        case db::ugc::Traffic::RightHand:
            return pugc::Traffic::RIGHT_HAND;
        case db::ugc::Traffic::LeftHand:
            return pugc::Traffic::LEFT_HAND;
    }
    throw LogicError() << "Unexpected traffic " << traffic;
}

pi18n::Time serializeTime(chrono::TimePoint time, const std::locale& stdLocale)
{
    using std::chrono::seconds;
    using std::chrono::duration_cast;
    using std::chrono::time_point_cast;
    pi18n::Time result;
    result.value() = chrono::convertToUnixTime(time);
    result.tz_offset() = 0;
    result.text() = internationalize(
        stdLocale,
        i18n::units::Time(time_point_cast<seconds>(time))
    );
    return result;
}

pcommon::Task serializeTask(
    const db::ugc::Task& task,
    const std::locale& stdLocale
)
{
    pcommon::Task pbTask;
    pbTask.id() = std::to_string(task.id());
    pbTask.duration().value() = task.durationInSeconds().count();
    pbTask.duration().text() = internationalize(
        stdLocale,
        i18n::units::Duration(task.durationInSeconds().count())
    );
    pbTask.distance().value() = task.distanceInMeters();
    pbTask.distance().text() = internationalize(
        stdLocale,
        i18n::units::Distance(task.distanceInMeters())
    );
    return pbTask;
}

pcommon::Assignment serializeAssignment(
    const db::ugc::Assignment& assignment,
    const std::locale& stdLocale
)
{
    pcommon::Assignment pbAssignment;
    pbAssignment.id() = std::to_string(assignment.id());
    pbAssignment.status() = serializeAssignmentStatus(assignment.status());
    pbAssignment.acquired() = serializeTime(assignment.acquiredAt(), stdLocale);
    if (assignment.submittedAt()) {
        pbAssignment.completed() = serializeTime(
            *assignment.submittedAt(),
            stdLocale
        );
    }
    // Fill the deadline field (even thought it's optional) in order not
    // to break old versions of private mobile app that expect it to be defined
    pbAssignment.deadline() = serializeTime(assignment.acquiredAt(), stdLocale);
    return pbAssignment;
}

pgeo_object::GeoObject serializeToGeoObject(
    pmetadata::Metadata pbMetadata,
    const db::ugc::Task& task,
    const Locale& locale
)
{
    pgeo_object::GeoObject pbGeoObject;
    pbGeoObject.metadata().push_back(std::move(pbMetadata));
    pbGeoObject.name() = findBestTaskName(task, locale);
    pbGeoObject.bounded_by() = geolib3::sproto::encode(task.geodeticHull().boundingBox());
    for (size_t i = 0; i < task.geodeticHull().polygonsNumber(); ++i) {
        pbGeoObject.geometry().push_back(
            geolib3::sproto::encodeGeometry(task.geodeticHull().polygonAt(i))
        );
    }
    return pbGeoObject;
}

pgeo_object::GeoObject serializeToGeoObject(
    pmetadata::Metadata pbMetadata,
    const db::ugc::Target& target
)
{
    pgeo_object::GeoObject pbGeoObject;
    pbGeoObject.metadata().push_back(std::move(pbMetadata));
    //WARN: resulting GeoObject will not have any name
    pbGeoObject.bounded_by() = geolib3::sproto::encode(target.geodeticGeom().boundingBox());
    pbGeoObject.geometry().push_back(
        geolib3::sproto::encodeGeometry(target.geodeticGeom())
    );
    return pbGeoObject;
}

//complex object serialization to sproto GeoObject
pgeo_object::GeoObject serializeTaskToGeoObject(
    const db::ugc::Task& task,
    const Locale& locale,
    const std::locale& stdLocale
)
{
    pugc::TaskMetadata pbTaskMetadata;
    pbTaskMetadata.task() = serializeTask(task, stdLocale);
    pmetadata::Metadata pbMetadata;
    pbMetadata[pugc::TASK_METADATA] = std::move(pbTaskMetadata);
    return serializeToGeoObject(std::move(pbMetadata), task, locale);
}

pgeo_object::GeoObject serializeAssignmentToGeoObject(
    const db::ugc::Assignment& assignment,
    const db::ugc::Task& task,
    const Locale& locale,
    const std::locale& stdLocale
)
{
    pugc::AssignmentMetadata pbAssignmentMetadata;
    pbAssignmentMetadata.assignment() = serializeAssignment(assignment, stdLocale),
    pbAssignmentMetadata.task() = serializeTask(task, stdLocale);
    pmetadata::Metadata pbMetadata;
    pbMetadata[pugc::ASSIGNMENT_METADATA] = std::move(pbAssignmentMetadata);
    return serializeToGeoObject(std::move(pbMetadata), task, locale);
}

pgeo_object::GeoObject serializeTargetToGeoObject(const db::ugc::Target& target)
{
    pugc::TargetMetadata pbTargetMetadata;
    pbTargetMetadata.oneway() = target.isOneway();
    pbTargetMetadata.passing_direction() = serializeDirection(target.direction());
    pbTargetMetadata.traffic() = serializeTraffic(target.traffic());
    if (target.forwardPos()) {
        pbTargetMetadata.forward_pos() = *target.forwardPos();
    }

    if (target.backwardPos()) {
        pbTargetMetadata.backward_pos() = *target.backwardPos();
    }

    pmetadata::Metadata pbMetadata;
    pbMetadata[pugc::TARGET_METADATA] = std::move(pbTargetMetadata);
    return serializeToGeoObject(std::move(pbMetadata), target);
}

pgeo_object::GeoObject serializePolylineToGeoObject(
    const geolib3::Polyline2& polyline)
{
    pugc::TargetMetadata pbTargetMetadata;
    pbTargetMetadata.oneway() = true;
    pbTargetMetadata.passing_direction() = serializeDirection(db::ugc::Direction::Forward);
    pbTargetMetadata.traffic() = serializeTraffic(db::ugc::Traffic::RightHand);
    pmetadata::Metadata pbMetadata;
    pbMetadata[pugc::TARGET_METADATA] = std::move(pbTargetMetadata);

    pgeo_object::GeoObject pbGeoObject;
    pbGeoObject.metadata().push_back(std::move(pbMetadata));
    pbGeoObject.bounded_by() = geolib3::sproto::encode(polyline.boundingBox());
    pbGeoObject.geometry().push_back(geolib3::sproto::encodeGeometry(polyline));
    return pbGeoObject;
}

ptargets::Version serializeCoverageVersion(
    const std::string& value)
{
    ptargets::Version version;
    version.value() = value;
    return version;
}

db::TrackPoint convertToTrackPoint(const std::string& sourceId,
                                   const presults::TrackPoint& trackPoint)
{
    const auto& location = trackPoint.location();

    db::TrackPoint result{};
    result.setSourceId(sourceId)
        .setGeodeticPos(geolib3::sproto::decode(location.point()))
        .setTimestamp(
            chrono::TimePoint(std::chrono::milliseconds(trackPoint.time())));

    if (location.accuracy()) {
        result.setAccuracyMeters(location.accuracy().get());
    }
    if (location.heading()) {
        result.setHeading(geolib3::Heading(location.heading().get()));
    }
    if (location.speed()) {
        result.setSpeedMetersPerSec(location.speed().get());
    }
    return result;
}

db::TrackPoint
convertToTrackPoint(db::TId assignmentId,
                    const std::string& sourceId,
                    const presults::TrackPoint& trackPoint)
{
    return convertToTrackPoint(sourceId, trackPoint)
        .setAssignmentId(assignmentId);
}

db::TrackPoints
extractTrackPoints(db::TId assignmentId,
                   const std::string& sourceId,
                  const presults::Results& data)
{
    db::TrackPoints result;
    const auto& trackPoints = data.track();
    result.reserve(trackPoints.size());

    for (const auto& trackPoint : trackPoints) {
        result.push_back(
            convertToTrackPoint(assignmentId, sourceId, trackPoint));
    }

    return result;
}

db::TrackPoints
extractTrackPoints(const std::string& sourceId,
                   const presults::Results& data)
{
    db::TrackPoints result;
    const auto& trackPoints = data.track();
    result.reserve(trackPoints.size());

    for (const auto& trackPoint : trackPoints) {
        result.push_back(convertToTrackPoint(sourceId, trackPoint));
    }

    return result;
}

db::ugc::AssignmentObjectType
deserializeAssignmentObjectType(presults::ObjectType objectType)
{
    switch (objectType) {
    case presults::ObjectType::BARRIER:
        return db::ugc::AssignmentObjectType::Barrier;
    case presults::ObjectType::DEADEND:
        return db::ugc::AssignmentObjectType::Deadend;
    case presults::ObjectType::BAD_CONDITIONS:
        return db::ugc::AssignmentObjectType::BadConditions;
    case presults::ObjectType::NO_ENTRY:
        return db::ugc::AssignmentObjectType::NoEntry;
    default:
        break;
    }
    throw LogicError() << "Unexpected assignment object type " << objectType;
}

db::ugc::AssignmentObject convertToAssignmentObject(
    db::TId assignmentId,
    const presults::Object& object)
{
    REQUIRE(object.type() && object.point(),
            "Assignment object must have its type and position point defined");

    db::ugc::AssignmentObject result(
        assignmentId,
        chrono::TimePoint(std::chrono::milliseconds(object.created())),
        geolib3::sproto::decode(object.point().get()),
        deserializeAssignmentObjectType(object.type().get()));

    if (object.comment()) {
        result.setComment(object.comment().get());
    }
    return result;
}

db::ugc::AssignmentObjects extractAssignmentObjects(
    db::TId assignmentId,
    const presults::Results& data)
{
    db::ugc::AssignmentObjects result;
    const auto& objects = data.objects();
    result.reserve(objects.size());

    for (const auto& object : objects) {
        result.push_back(convertToAssignmentObject(assignmentId, object));
    }

    return result;
}

} // maps::mrc::agent_proxy
