#include "common.h"
#include "fixture.h"

#include <maps/infra/tvm2lua/ticket_parser2_lua/lib/lua_api.h>
#include <maps/infra/yacare/include/test_utils.h>
#include <maps/libs/auth/include/blackbox.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/http/include/http.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/walk_object_gateway.h>
#include <maps/libs/geolib/include/point.h>
#include <yandex/maps/geolib3/proto.h>
#include <yandex/maps/geolib3/sproto.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>
#include <yandex/maps/mrc/unittest/local_server.h>
#include <yandex/maps/mrc/unittest/unittest_config.h>
#include <yandex/maps/proto/common2/geometry.sproto.h>
#include <yandex/maps/proto/mrc/backoffice/backoffice.pb.h>
#include <yandex/maps/proto/offline-mrc/results.sproto.h>

#include <future>
#include <sstream>
#include <thread>

using namespace std::literals;

namespace presults = yandex::maps::sproto::offline::mrc::results;

namespace maps::mrc::agent_proxy::tests {
namespace {

const std::string HTTP_HOST = "localhost";
const std::string REMOTE_ADDR = "127.0.0.1";

const std::string OAUTH_HEADER = "OAuth b1070fac00ce46388a2f2645dfff09c6";
const std::string DEVICE_ID = "MyFluffyPhone";
const std::string TASK_ID = "task-id";

const std::string BACKOFFICE_OBJECT_URL = "http://localhost/v2/objects/create";
const std::string BACKOFFICE_PHOTO_URL = "http://localhost/v2/photos/create";

const std::string MDS_GROUP = "123";
const std::string MDS_PATH = "unittest/walk/images/1";
const std::string MDS_PATH_2 = "unittest/walk/images/2";

const auto AZIMUTH = 20;
const auto COMMENT = std::string{"comment"};
const auto IMAGE = std::string{"<jpg>"};
const auto POS = geolib3::Point2(10.0, 20.0);
const auto TIME = chrono::parseSqlDateTime("2021-05-09 11:38:00.000+03");


void initGeoPhoto(
    yandex::maps::proto::mrc::backoffice::GeoPhoto& geoPhoto)
{
    *geoPhoto.mutable_image() = IMAGE;
    auto shootingPoint = geoPhoto.mutable_shooting_point();
    *shootingPoint->mutable_point()->mutable_point() =
        geolib3::proto::encode(POS);
    shootingPoint->mutable_direction()->set_azimuth(AZIMUTH);
    shootingPoint->mutable_direction()->set_tilt(0);
    geoPhoto.set_taken_at(chrono::convertToUnixTime(TIME));
}

auto makeCreateObjectRequest(
    size_t photosNumber,
    std::optional<yandex::maps::proto::mrc::backoffice::Dataset> dataset = {})
{
    auto result = yandex::maps::proto::mrc::backoffice::CreateObjectRequest{};
    *result.mutable_point() = geolib3::proto::encode(POS);
    *result.mutable_comment() = COMMENT;
    if (dataset.has_value()) {
        result.set_dataset(dataset.value());
    }
    for (size_t i = 0; i < photosNumber; ++i) {
        initGeoPhoto(*result.add_photos());
    }
    EXPECT_TRUE(result.IsInitialized());
    return result;
}

auto makeCreatePhotoRequest(std::optional<yandex::maps::proto::mrc::backoffice::Dataset> dataset = {})
{
    auto result = yandex::maps::proto::mrc::backoffice::CreatePhotoRequest{};
    initGeoPhoto(*result.mutable_photo());
    if (dataset.has_value()) {
        result.set_dataset(dataset.value());
    }
    EXPECT_TRUE(result.IsInitialized());
    return result;
}

} // namespace

TEST_F(Fixture, test_backoffice_object_two_images)
{
    auto tvmId = std::string("100500");
    auto mockAuth = BlackboxFixture{};
    auto blackboxMock = addBlackboxMock(mockAuth.blackboxUrl());
    auto mdsMock = http::addMock(
        makeMdsUploadUrl(config(), MDS_PATH), [](const http::MockRequest&) {
            return http::MockResponse("<post key=\"" + MDS_GROUP + "/" +
                                      MDS_PATH + "\"></post>");
        });
    auto mdsMock2 = http::addMock(
        makeMdsUploadUrl(config(), MDS_PATH_2), [&](const http::MockRequest&) {
            return http::MockResponse("<post key=\"" + MDS_GROUP + "/" +
                                      MDS_PATH_2 + "\"></post>");
        });

    auto createObjectRequest = makeCreateObjectRequest(2 /*photosNumber*/);
    EXPECT_TRUE(createObjectRequest.IsInitialized());
    auto buf = TString{};
    EXPECT_TRUE(createObjectRequest.SerializeToString(&buf));

    http::MockRequest req(http::POST, BACKOFFICE_OBJECT_URL);
    req.headers.try_emplace(maps::auth::SERVICE_TICKET_HEADER, "fake");
    req.headers.try_emplace(maps::tvm2::SRC_TVM_ID_HEADER, tvmId);
    req.body.assign(buf.data(), buf.size());
    auto resp = yacare::performTestRequest(req);
    EXPECT_EQ(resp.status, HTTP_STATUS_OK);
    auto createObjectResponse =
        yandex::maps::proto::mrc::backoffice::CreateObjectResponse{};
    EXPECT_TRUE(createObjectResponse.ParseFromString(TString{resp.body}));
    EXPECT_EQ(createObjectResponse.photo_ids().size(), 2);

    auto txn = pgPool().masterReadOnlyTransaction();
    auto objects = db::WalkObjectGateway{*txn}.load();
    ASSERT_EQ(objects.size(), 1u);
    EXPECT_EQ(objects[0].dataset(), db::Dataset::BackofficeObject);
    EXPECT_EQ(objects[0].comment(), COMMENT);
    auto photos = db::FeatureGateway{*txn}.load();
    EXPECT_EQ(photos.size(), 2u);
    auto photoIds = std::vector<std::string>{};
    for (const auto& photo : photos) {
        EXPECT_EQ(photo.walkObjectId(), objects[0].id());
        EXPECT_EQ(photo.dataset(), db::Dataset::BackofficeObject);
        photoIds.push_back(std::to_string(photo.id()));
    }
    EXPECT_TRUE(std::is_permutation(createObjectResponse.photo_ids().begin(),
                                    createObjectResponse.photo_ids().end(),
                                    photoIds.begin(),
                                    photoIds.end()));
}

TEST_F(Fixture, test_backoffice_object_inherit_dataset)
{
    auto tvmId = std::string("100500");
    auto mockAuth = BlackboxFixture{};
    auto blackboxMock = addBlackboxMock(mockAuth.blackboxUrl());
    auto mdsMock = http::addMock(
        makeMdsUploadUrl(config(), MDS_PATH), [](const http::MockRequest&) {
            return http::MockResponse("<post key=\"" + MDS_GROUP + "/" +
                                      MDS_PATH + "\"></post>");
        });

    auto createObjectRequest =
        makeCreateObjectRequest(
            1 /*photosNumber*/,
            yandex::maps::proto::mrc::backoffice::YANG_PEDESTRIANS);

    EXPECT_TRUE(createObjectRequest.IsInitialized());
    auto buf = TString{};
    EXPECT_TRUE(createObjectRequest.SerializeToString(&buf));

    http::MockRequest req(http::POST, BACKOFFICE_OBJECT_URL);
    req.headers.try_emplace(maps::auth::SERVICE_TICKET_HEADER, "fake");
    req.headers.try_emplace(maps::tvm2::SRC_TVM_ID_HEADER, tvmId);
    req.body.assign(buf.data(), buf.size());
    auto resp = yacare::performTestRequest(req);
    EXPECT_EQ(resp.status, HTTP_STATUS_OK);
    auto createObjectResponse =
        yandex::maps::proto::mrc::backoffice::CreateObjectResponse{};
    EXPECT_TRUE(createObjectResponse.ParseFromString(TString{resp.body}));
    EXPECT_EQ(createObjectResponse.photo_ids().size(), 1);

    auto txn = pgPool().masterReadOnlyTransaction();
    auto objects = db::WalkObjectGateway{*txn}.load();
    ASSERT_EQ(objects.size(), 1u);
    EXPECT_EQ(objects[0].dataset(), db::Dataset::YangPedestrians);
    auto photos = db::FeatureGateway{*txn}.load();
    EXPECT_EQ(photos.size(), 1u);
    auto photoIds = std::vector<std::string>{};
    for (const auto& photo : photos) {
        EXPECT_EQ(photo.walkObjectId(), objects[0].id());
        EXPECT_EQ(photo.dataset(), db::Dataset::YangPedestrians);
        photoIds.push_back(std::to_string(photo.id()));
    }
    EXPECT_TRUE(std::is_permutation(createObjectResponse.photo_ids().begin(),
                                    createObjectResponse.photo_ids().end(),
                                    photoIds.begin(),
                                    photoIds.end()));
}


TEST_F(Fixture, test_backoffice_object_no_images)
{
    auto tvmId = std::string("100500");
    auto mockAuth = BlackboxFixture{};
    auto blackboxMock = addBlackboxMock(mockAuth.blackboxUrl());

    auto createObjectRequest = makeCreateObjectRequest(0 /*photosNumber*/);
    EXPECT_TRUE(createObjectRequest.IsInitialized());
    auto buf = TString{};
    EXPECT_TRUE(createObjectRequest.SerializeToString(&buf));

    http::MockRequest req(http::POST, BACKOFFICE_OBJECT_URL);
    req.headers.try_emplace(maps::auth::SERVICE_TICKET_HEADER, "fake");
    req.headers.try_emplace(maps::tvm2::SRC_TVM_ID_HEADER, tvmId);
    req.body.assign(buf.data(), buf.size());
    auto resp = yacare::performTestRequest(req);
    EXPECT_EQ(resp.status, HTTP_STATUS_BAD_REQUEST);
}

TEST_F(Fixture, test_backoffice_photo)
{
    auto tvmId = std::string("100500");
    auto mockAuth = BlackboxFixture{};
    auto blackboxMock = addBlackboxMock(mockAuth.blackboxUrl());
    auto mdsMock = http::addMock(
        makeMdsUploadUrl(config(), MDS_PATH), [](const http::MockRequest&) {
            return http::MockResponse("<post key=\"" + MDS_GROUP + "/" +
                                      MDS_PATH + "\"></post>");
        });

    auto createPhotoRequest = makeCreatePhotoRequest();
    EXPECT_TRUE(createPhotoRequest.IsInitialized());
    auto buf = TString{};
    EXPECT_TRUE(createPhotoRequest.SerializeToString(&buf));

    http::MockRequest req(http::POST, BACKOFFICE_PHOTO_URL);
    req.headers.try_emplace(maps::auth::SERVICE_TICKET_HEADER, "fake");
    req.headers.try_emplace(maps::tvm2::SRC_TVM_ID_HEADER, tvmId);
    req.body.assign(buf.data(), buf.size());
    auto resp = yacare::performTestRequest(req);
    EXPECT_EQ(resp.status, HTTP_STATUS_OK);
    auto createPhotoResponse =
        yandex::maps::proto::mrc::backoffice::CreatePhotoResponse{};
    EXPECT_TRUE(createPhotoResponse.ParseFromString(TString{resp.body}));

    auto txn = pgPool().masterReadOnlyTransaction();
    auto photos = db::FeatureGateway{*txn}.load();
    EXPECT_EQ(photos.size(), 1u);
    EXPECT_EQ(photos[0].dataset(), db::Dataset::BackofficePhoto);
    EXPECT_EQ(std::to_string(photos[0].id()), createPhotoResponse.photo_id());
}

TEST_F(Fixture, test_backoffice_photo_inherit_dataset)
{
    auto tvmId = std::string("100500");
    auto mockAuth = BlackboxFixture{};
    auto blackboxMock = addBlackboxMock(mockAuth.blackboxUrl());
    auto mdsMock = http::addMock(
        makeMdsUploadUrl(config(), MDS_PATH), [](const http::MockRequest&) {
            return http::MockResponse("<post key=\"" + MDS_GROUP + "/" +
                                      MDS_PATH + "\"></post>");
        });

    auto createPhotoRequest =
        makeCreatePhotoRequest(yandex::maps::proto::mrc::backoffice::ALTAY_PEDESTRIANS);
    EXPECT_TRUE(createPhotoRequest.IsInitialized());
    auto buf = TString{};
    EXPECT_TRUE(createPhotoRequest.SerializeToString(&buf));

    http::MockRequest req(http::POST, BACKOFFICE_PHOTO_URL);
    req.headers.try_emplace(maps::auth::SERVICE_TICKET_HEADER, "fake");
    req.headers.try_emplace(maps::tvm2::SRC_TVM_ID_HEADER, tvmId);
    req.body.assign(buf.data(), buf.size());
    auto resp = yacare::performTestRequest(req);
    EXPECT_EQ(resp.status, HTTP_STATUS_OK);
    auto createPhotoResponse =
        yandex::maps::proto::mrc::backoffice::CreatePhotoResponse{};
    EXPECT_TRUE(createPhotoResponse.ParseFromString(TString{resp.body}));

    auto txn = pgPool().masterReadOnlyTransaction();
    auto photos = db::FeatureGateway{*txn}.load();
    EXPECT_EQ(photos.size(), 1u);
    EXPECT_EQ(photos[0].dataset(), db::Dataset::AltayPedestrians);
    EXPECT_EQ(std::to_string(photos[0].id()), createPhotoResponse.photo_id());
}

} // maps::mrc::agent_proxy::tests
