#include "common.h"
#include "fixture.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/track_point_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>

#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/geolib/include/polygon.h>
#include <yandex/maps/mrc/signal_queue/result_queue.h>
#include <yandex/maps/mrc/signal_queue/results.h>
#include <yandex/maps/proto/common2/geometry.sproto.h>
#include <yandex/maps/proto/mrc/common/error.sproto.h>
#include <yandex/maps/proto/offline-mrc/results.sproto.h>
#include <maps/infra/yacare/include/yacare.h>
#include <maps/infra/yacare/include/test_utils.h>

using namespace std::chrono_literals;

namespace pmrc = yandex::maps::sproto::mrc;
namespace presults = yandex::maps::sproto::offline::mrc::results;
namespace pgeometry = yandex::maps::sproto::common2::geometry;

using namespace yacare::tests;

namespace maps::mrc::agent_proxy::tests {

namespace {

const std::string LOGIN_1 = "user1@yandex.ru";
const std::string LOGIN_2 = "user2@yandex.ru";

const geolib3::MultiPolygon2 TEST_HULL{{
    geolib3::Polygon2{geolib3::PointsVector{
        {0, 0},
        {0, 1},
        {1, 1},
        {0, 0}
    }},
    geolib3::Polygon2{geolib3::PointsVector{
        {0, 0},
        {-1, 0},
        {-1, -1},
        {0, 0}
    }}
}};

const geolib3::Polygon2 TARGET_REGION(
        geolib3::PointsVector{
            {-1, -1},
            {-1, 1},
            {1, 1},
            {1, -1}
        }
);

db::ugc::Task makeTestTask(
    db::ugc::TaskStatus status = db::ugc::TaskStatus::New)
{
    db::ugc::Task task{};
    task
        .setStatus(status)
        .setDuration(std::chrono::hours(24) * 80)
        .setDistanceInMeters(40000000)
        .setGeodeticHull(TEST_HULL)
    ;

    return task;
}

db::ugc::TaskName makeTestTaskName(
    const db::ugc::Task& task,
    const std::string& locale,
    const std::string& name)
{
  return {task.id(), boost::lexical_cast<Locale>(locale), name};
}

presults::Results makeAssignmentResults()
{
    presults::Results results;

    presults::Image image;
    image.created() = 1;
    image.image() = "raw data";

    results.images().push_back(image);

    presults::TrackPoint trackPoint;
    trackPoint.time() = 1;
    trackPoint.location().point().lon() = 46.;
    trackPoint.location().point().lat() = 54.;
    trackPoint.location().heading() = 15.;

    results.track().push_back(trackPoint);
    trackPoint.time() = 1;
    results.track().push_back(trackPoint);

    results.reports().push_back("REPORT DATA");

    pgeometry::Point point;
    point.lon() = 46.;
    point.lat() = 54.;

    presults::Object object;
    object.created() = 1;
    object.type() = presults::ObjectType::BARRIER;
    object.point() = point;
    object.comment() = "something brief";

    results.objects().push_back(object);
    object.type() = presults::ObjectType::DEADEND;
    results.objects().push_back(object);
    object.type() = presults::ObjectType::BAD_CONDITIONS;
    results.objects().push_back(object);
    object.type() = presults::ObjectType::NO_ENTRY;
    results.objects().push_back(object);

    return results;
}

http::MockRequest generateSearchRequest(maps::geolib3::Point2 point, maps::geolib3::Vector2 span) {
    auto ll = std::to_string(point.x()) + "," + std::to_string(point.y());
    auto spn = std::to_string(span.x()) + "," + std::to_string(span.y());
    return http::MockRequest(
        http::GET, http::URL("http://localhost/ugc/tasks/search")
                       .addParam("spn", spn)
                       .addParam("ll", ll)
                       .addParam("lang", "ru_RU"));
}

db::ugc::TasksGroup createExampleTasksGroup(
    db::ugc::UseRouting useRouting,
    const std::vector<geolib3::Polygon2>& polygons,
    std::optional<std::vector<std::string>> allowedAssigneesLogins)
{
    db::ugc::TasksGroup group{
        db::ugc::TasksGroupStatus::Generating,
        "Moscow 2021.03",
        geolib3::convertGeodeticToMercator(geolib3::MultiPolygon2(polygons)),
        useRouting,
        {1, 2, 3, 4, 5, 6, 7},
        db::ugc::UseToll::No,
        10000 // recommended length
    };

    group.setAllowedAssigneesLogins(std::move(allowedAssigneesLogins));
    return group;
}

void saveTasksGroup(
    maps::pgpool3::TransactionHandle&& txn, db::ugc::TasksGroup& tasksGroup)
{
    db::ugc::TasksGroupGateway gtw(*txn);
    gtw.insert(tasksGroup);
    txn->commit();
}

db::ugc::TasksGroup createAndSaveExampleTasksGroup(
    maps::pgpool3::TransactionHandle&& txn,
    db::ugc::UseRouting useRouting,
    const std::vector<geolib3::Polygon2>& polygons,
    std::optional<std::vector<std::string>> allowedAssigneesLogins)
{
    auto tasksGroup = createExampleTasksGroup(
        useRouting,
        polygons,
        std::move(allowedAssigneesLogins));
    saveTasksGroup(std::move(txn), tasksGroup);
    return tasksGroup;
}

} // namespace

TEST_F(Fixture, acquire_task) {
    auth::UserInfo userInfo;
    userInfo.setUid(USER_ID);
    userInfo.setLogin(LOGIN_1);
    UserInfoFixture userInfoFixture(std::move(userInfo));

    auto task = makeTestTask();

    {
        auto txn = pgPool().masterWriteableTransaction();
        db::ugc::TaskGateway{*txn}.insert(task);
        db::ugc::TaskNameGateway{*txn}.insert(
            makeTestTaskName(task, "ru_RU", "Такса"));
        txn->commit();
    }

    http::MockRequest request(
        http::POST,
        http::URL("http://localhost/ugc/tasks/acquire")
            .addParam("id", task.id())
            .addParam("lang", "ru_RU"));

    {
        auto response = yacare::performTestRequest(request);
        ASSERT_EQ(response.status, (int)yacare::Status::Created);

        auto txn = pgPool().masterReadOnlyTransaction();
        db::ugc::TaskGateway{*txn}.reload(task);
        EXPECT_EQ(task.status(), db::ugc::TaskStatus::Acquired);

        const auto assignment = db::ugc::AssignmentGateway{*txn}.tryLoadOne(
            db::ugc::table::Assignment::taskId == task.id());
        ASSERT_TRUE(assignment);
        EXPECT_EQ(assignment->status(), db::ugc::AssignmentStatus::Active);
    }

    {
        // Try to acquire the same task twice
        auto response = yacare::performTestRequest(request);
        ASSERT_EQ(response.status, (int)yacare::Status::Conflict);
        const auto error =
            boost::lexical_cast<pmrc::common::Error>(response.body);
        EXPECT_EQ(
            *error.code(), pmrc::common::Error::Code::TASK_ALREADY_ACQUIRED);

        auto txn = pgPool().masterReadOnlyTransaction();
        const auto assignments = db::ugc::AssignmentGateway{*txn}.load(
            db::ugc::table::Assignment::taskId == task.id());
        EXPECT_EQ(assignments.size(), 1u);
    }
}

TEST_F(Fixture, assignment_upload_upload_test)
{
    UserIdHeaderFixture userIdHeaderFixture;
    db::TId assignmentId = 0;
    {
        auto txn = pgPool().masterWriteableTransaction();

        auto task = makeTestTask();
        db::ugc::TaskGateway{*txn}.insert(task);

        auto assignment = task.assignTo(USER_ID);
        db::ugc::AssignmentGateway{*txn}.insert(assignment);
        txn->commit();

        assignmentId = assignment.id();
    }

    auto results = makeAssignmentResults();
    std::stringstream sstream;
    sstream << results;

    http::MockRequest request(
        http::POST,
        http::URL("http://localhost/ugc/assignments/upload")
            .addParam("id", assignmentId)
            .addParam("deviceid", SOURCE_ID)
            .addParam("uid", USER_ID));

    request.headers.emplace(USER_ID_HEADER, USER_ID);

    const std::string clientIp = "192.168.1.1";
    const uint16_t clientPort = 9999;
    setClientHostPort(request, clientIp, clientPort);

    request.body = sstream.str();

    auto response = yacare::performTestRequest(request);

    ASSERT_EQ(response.status, 200);

    auto txn = pgPool().masterWriteableTransaction();

    auto trackPoints = db::TrackPointGateway{*txn}.load(
        db::table::TrackPoint::assignmentId.equals(assignmentId)
    );
    EXPECT_EQ(trackPoints.size(), 2ul);

    auto reports = db::ugc::AssignmentRecordingReportGateway{*txn}
        .loadByAssignmentId(assignmentId);
    EXPECT_EQ(reports.size(), 1ul);
    EXPECT_EQ(reports[0].sourceId(), SOURCE_ID);

    auto objects = db::ugc::AssignmentObjectGateway{*txn}
        .loadByAssignmentId(assignmentId);
    EXPECT_EQ(objects.size(), 4ul);

    signal_queue::ResultsQueue queue(config().signalsUploader().queuePath());
    EXPECT_EQ(queue.count<signal_queue::AssignmentImage>(), 1ul);

    auto image = queue.pop<signal_queue::AssignmentImage>();
    ASSERT_TRUE(image);

    ASSERT_TRUE(image->userId());
    EXPECT_EQ(*image->userId(), USER_ID);

    ASSERT_TRUE(image->sourceIp());
    EXPECT_EQ(*image->sourceIp(), clientIp);

    ASSERT_TRUE(image->sourcePort());
    EXPECT_EQ(*image->sourcePort(), clientPort);
}

TEST_F(Fixture, tasks_search_big_bbox_test)
{
    std::vector<std::pair<maps::geolib3::Point2, maps::geolib3::Vector2>> requests = {
        {maps::geolib3::Point2{0.0, 0.0}, maps::geolib3::Vector2{120.0, 120.0}},
        {maps::geolib3::Point2{90.0, 0.0}, maps::geolib3::Vector2{130.0, 150.0}},
        {maps::geolib3::Point2{90.0, 90.0}, maps::geolib3::Vector2{160.0, 160.0}},
        {maps::geolib3::Point2{90.0, 90.0}, maps::geolib3::Vector2{180.0, 160.0}},
    };

    for (auto&& [ll, spn] : requests) {
        auto request = generateSearchRequest(ll, spn);
        auto response = yacare::performTestRequest(request);
        ASSERT_TRUE(response.status < 400);
    }
}

TEST_F(Fixture, tasks_with_allowed_assignees_search_test)
{
    std::vector<std::string> allowedLogins = {LOGIN_1};
    auto tasksGroup = createAndSaveExampleTasksGroup(
        pgPool().masterWriteableTransaction(),
        db::ugc::UseRouting::Yes,
        {TARGET_REGION},
        allowedLogins
    );

    {
        auto txn = pgPool().masterWriteableTransaction();

        auto task = makeTestTask();
        task.setTasksGroupId(tasksGroup.id());
        db::ugc::TaskGateway{*txn}.insert(task);
        db::ugc::TaskNameGateway{*txn}.insert(
            makeTestTaskName(task, "ru_RU", "Таск 1"));
        txn->commit();
    }

    {
        auth::UserInfo userInfo;
        userInfo.setUid(USER_ID);
        userInfo.setLogin(LOGIN_1);
        UserInfoFixture userInfoFixture(std::move(userInfo));

        auto request = generateSearchRequest(maps::geolib3::Point2{0.0, 0.0}, maps::geolib3::Vector2{1.0, 1.0});
        auto response = yacare::performTestRequest(request);
        ASSERT_EQ(response.status, 200);
    }

    {
        auth::UserInfo userInfo;
        userInfo.setUid(USER_ID);
        userInfo.setLogin(LOGIN_2);
        UserInfoFixture userInfoFixture(std::move(userInfo));

        auto request = generateSearchRequest(maps::geolib3::Point2{0.0, 0.0}, maps::geolib3::Vector2{1.0, 1.0});
        auto response = yacare::performTestRequest(request);
        ASSERT_EQ(response.status, 204);
    }

    {
        auto request = generateSearchRequest(maps::geolib3::Point2{0.0, 0.0}, maps::geolib3::Vector2{1.0, 1.0});
        auto response = yacare::performTestRequest(request);
        ASSERT_EQ(response.status, 204);
    }

    // Empty allowedLogins field
    auto noAllowedLoginsTasksGroup = createAndSaveExampleTasksGroup(
        pgPool().masterWriteableTransaction(),
        db::ugc::UseRouting::Yes,
        {TARGET_REGION},
        std::nullopt
    );

    {
        auto txn = pgPool().masterWriteableTransaction();

        auto task = makeTestTask();
        task.setTasksGroupId(noAllowedLoginsTasksGroup.id());
        db::ugc::TaskGateway{*txn}.insert(task);
        db::ugc::TaskNameGateway{*txn}.insert(
            makeTestTaskName(task, "ru_RU", "Таск 2"));
        txn->commit();
    }

    {
        auth::UserInfo userInfo;
        userInfo.setUid(USER_ID);
        userInfo.setLogin(LOGIN_2);
        UserInfoFixture userInfoFixture(std::move(userInfo));

        auto request = generateSearchRequest(maps::geolib3::Point2{0.0, 0.0}, maps::geolib3::Vector2{1.0, 1.0});
        auto response = yacare::performTestRequest(request);
        ASSERT_EQ(response.status, 200);
    }

    {
        auto request = generateSearchRequest(maps::geolib3::Point2{0.0, 0.0}, maps::geolib3::Vector2{1.0, 1.0});
        auto response = yacare::performTestRequest(request);
        ASSERT_EQ(response.status, 200);
    }
}

TEST_F(Fixture, tasks_with_allowed_assignees_acquire_test) {
    std::vector<std::string> allowedLogins = {LOGIN_1};
    UserIdHeaderFixture userIdHeaderFixture;
    auto tasksGroup = createAndSaveExampleTasksGroup(
        pgPool().masterWriteableTransaction(),
        db::ugc::UseRouting::Yes,
        {TARGET_REGION},
        allowedLogins
    );

    auto task = makeTestTask();
    task.setTasksGroupId(tasksGroup.id());

    {
        auto txn = pgPool().masterWriteableTransaction();
        db::ugc::TaskGateway{*txn}.insert(task);
        db::ugc::TaskNameGateway{*txn}.insert(
            makeTestTaskName(task, "ru_RU", "Таск"));
        txn->commit();
    }

    {
        auth::UserInfo userInfo;
        userInfo.setUid(USER_ID_RESTRICTED);
        userInfo.setLogin(LOGIN_2);
        UserInfoFixture userInfoFixture(std::move(userInfo));

        http::MockRequest request(
            http::POST,
            http::URL("http://localhost/ugc/tasks/acquire")
                .addParam("id", task.id())
                .addParam("lang", "ru_RU"));

        auto response = yacare::performTestRequest(request);
        ASSERT_EQ(response.status, (int)yacare::Status::Forbidden);
    }

    {
        auth::UserInfo userInfo;
        userInfo.setUid(USER_ID);
        userInfo.setLogin(LOGIN_1);
        UserInfoFixture userInfoFixture(std::move(userInfo));

        http::MockRequest request(
            http::POST,
            http::URL("http://localhost/ugc/tasks/acquire")
                .addParam("id", task.id())
                .addParam("lang", "ru_RU"));

        auto response = yacare::performTestRequest(request);
        ASSERT_EQ(response.status, (int)yacare::Status::Created);

        auto txn = pgPool().masterReadOnlyTransaction();
        db::ugc::TaskGateway{*txn}.reload(task);
        EXPECT_EQ(task.status(), db::ugc::TaskStatus::Acquired);

        const auto assignment = db::ugc::AssignmentGateway{*txn}.tryLoadOne(
            db::ugc::table::Assignment::taskId == task.id());
        ASSERT_TRUE(assignment);
        EXPECT_EQ(assignment->status(), db::ugc::AssignmentStatus::Active);
    }
}

} // namespace maps::mrc::agent_proxy::tests
