#include <maps/infra/yacare/include/tvm.h>
#include <maps/infra/yacare/include/yacare.h>
#include <maps/libs/auth/include/blackbox.h>
#include <maps/libs/auth/include/tvm.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/vault_boy/include/secrets.h>
#include <maps/wikimap/mapspro/services/mrc/browser/lib/configuration.h>
#include <maps/wikimap/mapspro/services/mrc/browser/lib/error_reporter.h>
#include <maps/wikimap/mapspro/services/mrc/browser/lib/tools.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/wiki_config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>

#include <boost/algorithm/string.hpp>

#include <filesystem>
#include <optional>

namespace browser = maps::mrc::browser;
namespace fs = std::filesystem;

namespace {

const std::string APP_NAME = "mrc-browser-pro";

} // anonymous namespace

yacare::VirtualHost vhost {
    yacare::VirtualHost::SLB { "core-nmaps-mrc-browser-pro" },
    yacare::VirtualHost::FQDN { "core-nmaps-mrc-browser-pro.maps.yandex.ru" },
    yacare::VirtualHost::FQDN { "core-nmaps-mrc-browser-pro.maps.yandex.net" },
    yacare::VirtualHost::FQDN { "core-nmaps-mrc-browser-pro.common.testing.maps.yandex.ru" },
    yacare::VirtualHost::FQDN { "core-nmaps-mrc-browser-pro.common.datatesting.maps.yandex.ru" },
};

// Service accesses to external database to process requests
static yacare::ThreadPool heavyPool("heavy", yacare::ThreadPool::CPUCOUNT(10), 4096);

YCR_SET_DEFAULT(vhost, heavyPool);

YCR_MAIN(argc, argv) try {
    maps::cmdline::Parser parser;
    auto configPath = parser.string("config")
            .help("path to MRC configuration");
    auto graphFolder = parser.string("graph_dir")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-graph-pro")
            .help("directory graph data is loaded from");
    auto pedestrianGraphFolder = parser.string("pedestrian_graph_dir")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-pedestrian-graph-pro")
            .help("directory pedestrian graph data is loaded from");
    auto photoToEdgeProFolder = parser.dir("photo_to_edge_pro_dir")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-photo-to-edge-pro")
            .help("yandex-maps-mrc-photo-to-edge-pro dataset directory");
    auto photoToPedestrianEdgeProFolder = parser.dir("photo_to_pedestrian_edge_pro_dir")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-photo-to-pedestrian-edge-pro")
            .help("yandex-maps-mrc-photo-to-pedestrian-edge-pro dataset directory");
    auto designsFolder = parser.string("design_dir")
            .defaultValue(maps::mrc::browser::DEFAULT_DESIGN_PATH)
            .help("directory with designs");
    auto iconsFolder = parser.string("icons_dir")
            .defaultValue(maps::mrc::browser::DEFAULT_DESIGN_ICONS_PATH)
            .help("directory with design icons");
    auto geoIdCoveragePath = parser.string("geoid_coverage_path")
            .defaultValue(maps::mrc::browser::DEFAULT_GEOID_COVERAGE_PATH)
            .help("path to coverage for geoid");
    auto wikiConfigPath = parser.string("wiki-config")
            .defaultValue(maps::mrc::common::WIKI_TEMPLATE_CONFIG_DIR)
            .help("Path to services config template for wikimap");

    parser.parse(argc, argv);

    INFO() << "Starting " << APP_NAME;
    const auto mrcCfg = maps::mrc::common::templateConfigFromCmdPath(
        configPath, maps::mrc::common::ServiceRole::DataConsumer);
    auto wikiCfg = maps::mrc::common::makeWikiConfig(wikiConfigPath);

    auto tvmtoolSettings = maps::auth::TvmtoolSettings();

    yacare::tvm::configureUserAuth(maps::auth::BlackboxApi(tvmtoolSettings));
    auto tvmClient = tvmtoolSettings.makeTvmClient();

    browser::Configuration::swap(
        std::make_shared<browser::Configuration>(
            mrcCfg,
            wikiCfg,
            fs::canonical(fs::path(std::string(graphFolder))).string(),
            fs::canonical(fs::path(std::string(pedestrianGraphFolder))).string(),
            fs::canonical(fs::path(std::string(photoToEdgeProFolder))).string(),
            fs::canonical(fs::path(std::string(photoToPedestrianEdgeProFolder))).string(),
            fs::canonical(fs::path(std::string(geoIdCoveragePath))).string(),
            std::nullopt /* mrcDataFolder */,
            iconsFolder,
            designsFolder,
            std::make_unique<maps::mrc::blackbox_client::BlackboxClient>(
                tvmtoolSettings),
            maps::mrc::browser::makePermissionChecker(mrcCfg, tvmClient)
        )
    );

    yacare::setErrorReporter(browser::errorReporter);
    yacare::run();
    browser::Configuration::swap({});
    INFO() << "Shutting down " << APP_NAME;
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
