#include <maps/infra/yacare/include/limit_rate.h>
#include <maps/infra/yacare/include/tvm.h>
#include <maps/infra/yacare/include/yacare.h>
#include <maps/libs/auth/include/blackbox.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/browser/lib/configuration.h>
#include <maps/wikimap/mapspro/services/mrc/browser/lib/permission_checker.h>
#include <maps/wikimap/mapspro/services/mrc/browser/lib/error_reporter.h>
#include <maps/wikimap/mapspro/services/mrc/browser/lib/tools.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/wiki_config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/ugc_event_logger/include/logger.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>

#include <filesystem>
#include <memory>
#include <string>

namespace browser = maps::mrc::browser;
namespace fs = std::filesystem;

namespace {

const std::string DEFAULT_MRC_DATA_DIR
    = "/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc";
const std::string SCHEMA_DIR
    = "/usr/share/yandex/maps/mrc/browser/data_set_schemas/";
const std::string SCHEMA_FILE_NAME = "source.json";

} // anonymous namespace

yacare::VirtualHost vhost{
    yacare::VirtualHost::SLB { "mrc-browser" },
    yacare::VirtualHost::SLB { "core-nmaps-mrc-browser" },
    yacare::VirtualHost::FQDN { "core-nmaps-mrc-browser.maps.yandex.ru" },
    yacare::VirtualHost::FQDN { "core-nmaps-mrc-browser.maps.yandex.net" },
    yacare::VirtualHost::FQDN { "core-nmaps-mrc-browser-internal.maps.yandex.net" },
    yacare::VirtualHost::FQDN { "core-nmaps-mrc-browser.testing.maps.yandex.ru" },
    yacare::VirtualHost::FQDN { "core-nmaps-mrc-browser.crowdtest.maps.yandex.ru" },
    yacare::VirtualHost::FQDN { "core-nmaps-mrc-browser-datatesting.crowdtest.maps.yandex.ru" },
    yacare::VirtualHost::FQDN { "core-nmaps-mrc-browser.common.datatesting.maps.yandex.ru" }
};

YCR_SET_DEFAULT(vhost);

YCR_RESPOND_TO("GET /cartograph/data_set_schema", l,
    YCR_USING(yacare::Tvm2ServiceRequire(maps::mrc::browser::SELF_TVM_SERVICE_ALIAS)),
    YCR_LIMIT_RATE(resource("maps_core_nmaps_mrc_browser_/cartograph/data_set_schema")))
{
    if (l.size() != 1)
        throw yacare::errors::BadRequest("wrong l param");
    const fs::path schemaPath = fs::path(SCHEMA_DIR) / l.front() / SCHEMA_FILE_NAME;
    if (!fs::exists(schemaPath))
        throw yacare::errors::NotFound();
    response[maps::mrc::browser::CONTENT_TYPE_HEADER]
        = maps::mrc::browser::CONTENT_TYPE_JSON;
    response << maps::common::readFileToString(schemaPath);
}

YCR_MAIN(argc, argv) try {
    maps::cmdline::Parser parser;
    auto configPath = parser.string("config")
            .help("path to MRC configuration");
    auto graphFolder = parser.string("graph_dir")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-graph")
            .help("directory graph data is loaded from");
    auto pedestrianGraphFolder = parser.string("pedestrian_graph_dir")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-pedestrian-graph")
            .help("directory pedestrian graph data is loaded from");
    auto photoToEdgeFolder = parser.dir("photo_to_edge_dir")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-photo-to-edge")
            .help("yandex-maps-mrc-photo-to-edge dataset directory");
    auto photoToPedestrianEdgeFolder = parser.dir("photo_to_pedestrian_edge_dir")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-photo-to-pedestrian-edge")
            .help("yandex-maps-mrc-photo-to-pedestrian-edge dataset directory");
    auto designsFolder = parser.string("design_dir")
            .defaultValue(maps::mrc::browser::DEFAULT_DESIGN_PATH)
            .help("directory with designs");
    auto iconsFolder = parser.string("icons_dir")
            .defaultValue(maps::mrc::browser::DEFAULT_DESIGN_ICONS_PATH)
            .help("directory with design icons");
    auto geoIdCoveragePath = parser.string("geoid_coverage_path")
            .defaultValue(maps::mrc::browser::DEFAULT_GEOID_COVERAGE_PATH)
            .help("path to coverage for geoid");
    auto wikiConfigDirPath = parser.string("wiki-config-dir")
            .defaultValue(maps::mrc::common::WIKI_TEMPLATE_CONFIG_DIR)
            .help("Path to services config template for wikimap");
    auto wikiConfigPath = parser.string("wiki-config")
            .help("path to services config for wikimap");
    auto mrcDataDir = parser.string("mrc-data-dir")
            .defaultValue(DEFAULT_MRC_DATA_DIR)
            .help("directory with export fb files");
    auto ugcEventLogPath = parser.string("ugc-event-log")
        .help("path to ugc events log")
        .defaultValue("/var/log/yandex/maps/mrc/ugc_events/ugc_events.log");
    auto secretVersion = parser.string("secret-version")
            .help("version for secrets from yav.yandex-team.ru");
    auto disableTvmClient = parser.flag("disable-tvm")
            .defaultValue(false)
            .help("disable tvm client");

    parser.parse(argc, argv);

    INFO() << "Starting";
    const auto mrcCfg = maps::mrc::common::templateConfigFromCmdPath(
        secretVersion,
        configPath,
        maps::mrc::common::ServiceRole::DataConsumer);

    auto wikiCfg = wikiConfigPath.defined()
        ? maps::wiki::common::ExtendedXmlDoc(wikiConfigPath)
        : maps::mrc::common::makeWikiConfig(wikiConfigDirPath);

    auto blackboxClient =
        std::unique_ptr<maps::mrc::blackbox_client::IBlackboxClient>{};

    maps::mrc::browser::IPermissionCheckerHolder permissionChecker;

    std::optional<NTvmAuth::TTvmClient> tvmClient;
    if (disableTvmClient) {
        blackboxClient = std::make_unique<maps::mrc::blackbox_client::Stub>();
        permissionChecker = std::make_unique<maps::mrc::browser::PermissionCheckerStub>();
    }
    else {
        auto tvmtoolSettings = maps::auth::TvmtoolSettings();
        blackboxClient =
            std::make_unique<maps::mrc::blackbox_client::BlackboxClient>(
                tvmtoolSettings);
        yacare::tvm::configureUserAuth(
            maps::auth::BlackboxApi(tvmtoolSettings),
            {yacare::tvm::AuthMethod::UserTicket, yacare::tvm::AuthMethod::SessionId}
        );

        tvmClient = tvmtoolSettings.makeTvmClient();
        permissionChecker = maps::mrc::browser::makePermissionChecker(mrcCfg, tvmClient.value());
    }

    browser::Configuration::swap(
        std::make_shared<browser::Configuration>(
            mrcCfg,
            wikiCfg,
            fs::canonical(fs::path(std::string(graphFolder))).string(),
            fs::canonical(fs::path(std::string(pedestrianGraphFolder))).string(),
            fs::canonical(fs::path(std::string(photoToEdgeFolder))).string(),
            fs::canonical(fs::path(std::string(photoToPedestrianEdgeFolder))).string(),
            fs::canonical(fs::path(std::string(geoIdCoveragePath))).string(),
            fs::canonical(fs::path(std::string(mrcDataDir))).string(),
            iconsFolder,
            designsFolder,
            std::move(blackboxClient),
            std::move(permissionChecker)
        )
    );

    const std::chrono::seconds LOG_ROTATION_PERIOD(3600);
    auto ugcEventLogger = std::make_shared<maps::mrc::ugc_event_logger::Logger>(ugcEventLogPath, LOG_ROTATION_PERIOD);
    auto configuration = browser::Configuration::instance();
    configuration->setUgcEventLogger(ugcEventLogger);

    yacare::setErrorReporter(browser::errorReporter);
    yacare::run();
    browser::Configuration::swap({});
    INFO() << "Shutting down";
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
